"""StructureComponentDisplay class."""
# 1. Standard python modules
import copy
import os
import shutil
import uuid

# 2. Third party modules
import numpy as np
import orjson
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.gmi.gui.group_set_dialog import GroupSetDialog
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.data import structure_data as scd
from xms.tuflowfv.data import structure_parameters as sp
from xms.tuflowfv.data.structure_gmi_mapper import StructureGmiMapper
from xms.tuflowfv.gui import gui_util


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

STRUCT_ARC_JSON = 'struct_arc_disp.json'
STRUCT_POLYGON_JSON = 'struct_poly_disp.json'

REG_KEY_STRUCT_ARC = 'struct_coverage_arc_default_options'
REG_KEY_STRUCT_POLY = 'struct_coverage_polygon_default_options'


class StructureComponentDisplay:
    """Helper class for structure component display options."""

    def __init__(self, struct_comp, query=None):
        """Create a helper class.

        Args:
            struct_comp (StructureComponent): The component this helper should help
            query (Query): The XMS inter-process communicator
        """
        self.comp = struct_comp
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''
        self._query = query

    def _unpack_xms_data(self, param_map, target_type):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (dict): The ActionRequest parameter map
            target_type (TargetType): The feature type

        Returns:
            tuple:
                int: component id of atts to display
                int: component id of the second feature of a set, if it exists
        """
        # Get the XMS attribute ids of the selected features (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID, UNINITIALIZED_COMP_ID

        # Get the component id map of the selected features (if any).
        comp1_id = UNINITIALIZED_COMP_ID
        comp2_id = UNINITIALIZED_COMP_ID
        id_files = param_map.get('id_files', [])
        if id_files and id_files[0]:
            if target_type == TargetType.arc:
                files_dict = {'ARC': (id_files[0], id_files[1])}
            else:  # target_type == TargetType.polygon
                files_dict = {'POLYGON': (id_files[0], id_files[1])}
            self.comp.load_coverage_component_id_map(files_dict)
            comp1_id, comp2_id = self._check_selected_types(target_type=target_type)
        # Clean up temp files dumped by SMS.
        shutil.rmtree(os.path.join(os.path.dirname(self.comp.main_file), 'temp'), ignore_errors=True)
        return comp1_id, comp2_id

    def _check_selected_types(self, target_type):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Args:
            target_type (TargetType): The feature type

        Returns:
            tuple:
                int: component id of atts to display
                int: component id of the second feature of a set, if it exists
        """
        num_features = len(self.selected_att_ids)
        if num_features == 1:  # One feature selected, use those atts
            comp_id = self.comp.get_comp_id(target_type, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id, UNINITIALIZED_COMP_ID
        else:  # More than one feature selected, check types
            try:  # Get the component ids of all the selected features
                self.selected_comp_ids = list(self.comp.comp_to_xms[self.comp.cov_uuid][target_type].keys())
            except KeyError:  # No component ids assigned for any of the selected features
                return UNINITIALIZED_COMP_ID, UNINITIALIZED_COMP_ID

            # If there are two features that are in the same set, display those attributes.
            if num_features == 2:
                comp1_id = self.comp.get_comp_id(target_type, self.selected_att_ids[0])
                comp2_id = self.comp.get_comp_id(target_type, self.selected_att_ids[1])
                dset = self.comp.data.sets
                mask = dset['comp1_id'] == comp1_id
                comp1_dset = dset.where(mask, drop=True)
                mask = dset['comp2_id'] == comp2_id
                comp2_dset = dset.where(mask, drop=True)
                if comp1_dset.sizes['set_id'] == 1 and comp2_dset.sizes['set_id'] == 1:
                    if comp1_dset.set_id.values.item() == comp2_dset.set_id.values.item():
                        return comp1_id, comp2_id

            # Add a warning message to the dialog if multiple features are selected
            self._set_multi_select_warning(target_type)

            # If there are multiple features selected that all have the same attributes, display those. Otherwise, show
            # a dialog with default values.
            if len(self.selected_comp_ids) != 1:
                return UNINITIALIZED_COMP_ID, UNINITIALIZED_COMP_ID
            return self.selected_comp_ids[0], UNINITIALIZED_COMP_ID

    def _set_multi_select_warning(self, target_type):
        """Set the warning message in the dialog if multiple features are selected.

        Args:
            target_type (TargetType): Coverage entity type to update
        """
        if len(self.selected_att_ids) > 1:
            feature = 'arcs' if target_type == TargetType.arc else 'polygons'
            self.dlg_message = f'Multiple {feature} selected. Changes will apply to all selected {feature}.'

    def _update_component_ids_from_files(self, target_type):
        """Read att and comp ids from files and update in XMS.

        Called in get_initial_display_options() after a model native import.

        Args:
            target_type (TargetType): Coverage entity type to update
        """
        comp_folder = os.path.dirname(self.comp.main_file)
        if target_type == TargetType.arc:
            initial_att_file = os.path.join(comp_folder, sp.STRUCT_INITIAL_ARC_ATT_ID_FILE)
            initial_comp_file = os.path.join(comp_folder, sp.STRUCT_INITIAL_ARC_COMP_ID_FILE)
        else:  # target_type == TargetType.polygon
            initial_att_file = os.path.join(comp_folder, sp.STRUCT_INITIAL_POLY_ATT_ID_FILE)
            initial_comp_file = os.path.join(comp_folder, sp.STRUCT_INITIAL_POLY_COMP_ID_FILE)

        att_ids = read_display_option_ids(initial_att_file)
        comp_ids = read_display_option_ids(initial_comp_file)
        io_util.removefile(initial_att_file)
        io_util.removefile(initial_comp_file)
        for att_id, comp_id in zip(att_ids, comp_ids):
            self.comp.update_component_id(target_type, att_id, comp_id)

    @staticmethod
    def _update_coverage_registry_defaults(reg_key):
        """Save the default display options to the registry.

        Args:
            reg_key (str): The name of the key in the registry
        """
        settings = SettingsManager()
        # Read the default display options for mapped BC components.
        options = settings.get_setting('xmstuflowfv', reg_key)
        if not options:
            return  # No defaults set in registry
        json_dict = orjson.loads(options)
        json_dict['is_ids'] = 1  # Change from draw at locations to draw at ids.
        # Copy from mapped component registry key to BC coverage registry key.
        json_text = orjson.dumps(json_dict)
        settings.save_setting('xmstuflowfv', reg_key, json_text)

    def _update_id_files(self, flux_function, target_type, new_comp_id):
        """Writes the display id files. This is for a single comp_id (i.e.coming out of the assign structure dialog).

        Args:
            flux_function (str): The hydraulic structure type. See structure_data.FLUX_FUNCTION_TYPES
            target_type (TargetType): The feature location
            new_comp_id (int): The new comp_id
        """
        comp_path = os.path.dirname(self.comp.main_file)
        # Read the set component id files.
        default_file = ''
        default_comp_ids = []
        valid_sets = None
        set_file = self.get_display_id_file(flux_function, target_type, comp_path)
        set_comps = read_display_option_ids(set_file)
        # if flux_function in scd.FLUX_FUNCTION_SET_TYPES:
        np_set_comps = np.array(set_comps)
        # Find any selected arcs/polygons whose old component id belonged to a set pair
        np_selected_comps = np.array(self.selected_comp_ids)
        mask = np.isin(np_set_comps, np_selected_comps)
        selected_sets = np_set_comps[mask]

        default_flux_function = scd.DEFAULT_FLUX_FUNCTION_ARC if target_type == TargetType.arc else \
            scd.DEFAULT_FLUX_FUNCTION_POLY
        if selected_sets.size > 0:
            # Append the old structure set comp id to the unassigned list as the relationship is now broken. May
            # be other unselected arcs with this same component id, and they need to be invalidated.
            default_file = self.get_display_id_file(default_flux_function, target_type, comp_path)
            default_comp_ids = read_display_option_ids(default_file)
            selected_sets = selected_sets.tolist()
            default_comp_ids.extend(selected_sets)
            write_display_option_ids(default_file, default_comp_ids)
            # Inverse the mask to get the new list of structure set component ids that are still valid.
            valid_sets = np_set_comps[np.logical_not(mask)].tolist()

        # Add the new component id to the target category.
        updated_comp_ids = set_comps if valid_sets is None else valid_sets
        if flux_function == default_flux_function and default_file:
            # We may have already read the target category id file if it is unassigned.
            updated_comp_ids = default_comp_ids
        updated_comp_ids.append(new_comp_id)
        self.update_all_id_files(comp_path)

        # Send back updated display lists to XMS after ActionRequest
        display_file = self.comp.disp_opts_files[0] if target_type == TargetType.arc else self.comp.disp_opts_files[1]
        self.comp.display_option_list.append(XmsDisplayMessage(file=display_file, edit_uuid=self.comp.cov_uuid))

    def _cleanup_broken_sets(self, target_type):
        """Cleanup unused sets from the dataset table.

        Args:
            target_type (TargetType): Feature type
        """
        # Find all the sets with either the first or second comp_id
        is_in = self.comp.data.sets.comp1_id.isin(self.selected_comp_ids)
        # If we can't find it in the comp1 list, check comp2.
        if not bool(is_in.all()):
            is_in = self.comp.data.sets.comp2_id.isin(self.selected_comp_ids)
        set_ds = self.comp.data.sets.where(is_in, drop=True)
        if set_ds.sizes['set_id'] > 0:  # We have at least one broken set.
            # Make sure we have all the set partner features' comp_id.
            broken_comp_ids = copy.deepcopy(self.selected_comp_ids)
            broken_comp_ids.extend(set_ds.comp1_id.values.tolist())
            broken_comp_ids.extend(set_ds.comp2_id.values.tolist())
            broken_comp_ids = set(broken_comp_ids)

            # Delete the old set_id for this set relationship.
            mask = self.comp.data.sets['set_id'] != set_ds.set_id.data[0]
            self.comp.data.sets = self.comp.data.sets.where(mask, drop=True)

            # Reset the flux function to default value.
            default_flux_function = scd.DEFAULT_FLUX_FUNCTION_ARC if target_type == TargetType.arc else \
                scd.DEFAULT_FLUX_FUNCTION_POLY
            dset = self.comp.data.arcs if target_type == TargetType.arc else self.comp.data.polygons
            if dset.sizes['comp_id'] > 0:
                for broken_comp_id in broken_comp_ids:
                    try:  # Set the first feature's comp_id back to default
                        dset.flux_function.loc[dict(comp_id=[broken_comp_id])] = default_flux_function
                    except KeyError:
                        pass

    @staticmethod
    def _write_display_option_id_file(vals, id_file):
        """Write an ID file for a category.

        Args:
            vals (xarray.Dataset): The IDs to write
            id_file (str): The ID file to write
        """
        if len(vals.comp_id) > 0:
            write_display_option_ids(id_file, vals.comp_id.data.astype(int))
        else:  # If we don't have any of this type, just delete the id file.
            io_util.removefile(id_file)

    def _find_feature_ids(self, num_selected, comp1_att_ids, comp2_att_ids):
        """Find the feature ids of the two features in a set.

        Args:
            num_selected (int): Number of features selected
            comp1_att_ids (list[int]): Feature id of the first partner feature
            comp2_att_ids (list[int]): Feature id of the seconf partner feature

        Returns:
            tuple:
                int: Feature 1 ID
                int: Feature 2 ID
        """
        feature1_id = -1
        feature2_id = -1
        if len(comp1_att_ids) > 0:
            feature1_id = comp1_att_ids[0]
        elif num_selected == 2:
            feature1_id = self.selected_att_ids[0]
        if len(comp2_att_ids) > 0:
            feature2_id = comp2_att_ids[0]
        elif num_selected == 2:
            feature2_id = self.selected_att_ids[1]
        return feature1_id, feature2_id

    def _get_feature_ids_from_comp_ids(self, target_type, comp1_id, comp2_id):
        """Get the XMS feature ids for a structure.

        Args:
            target_type (TargetType): The feature type
            comp1_id (int): comp_id of the first feature
            comp2_id (int):  comp_id of the second feature of a set, if it exists

        Returns:
            tuple:
                list: Feature ids for the first comp_id
                list: Feature ids for the second comp_id
        """
        comp1_att_ids = self.comp.get_xms_ids(entity_type=target_type, comp_id=comp1_id)
        comp1_att_ids = comp1_att_ids if comp1_att_ids != -1 else []  # Dumb method returns -1 if no features
        comp2_att_ids = self.comp.get_xms_ids(entity_type=target_type, comp_id=comp2_id)
        comp2_att_ids = comp2_att_ids if comp2_att_ids != -1 else []  # Dumb method returns -1 if no features
        return comp1_att_ids, comp2_att_ids

    def _get_and_write_categories(self, default_filename, registry_key, attr_name, cat_idx):
        """Helper method to get categories, write to JSON, and set comp UUID attr.

        Args:
            default_filename (str): Default JSON file basename.
            registry_key (str): Registry key to read default display options from.
            attr_name (str): Attribute name to set in comp data info attrs.
            cat_idx (str): Display options files index. 0 == Arcs, 1 == Polygons.
        """
        categories = gui_util.read_default_display_options_from_registry(registry_key)
        if not categories:
            # Read the default arc display options, and save ourselves a copy with randomized UUID.
            categories = CategoryDisplayOptionList()
            default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                        'default_data', default_filename)
            json_dict = read_display_options_from_json(default_file)
            categories.from_dict(json_dict)
        categories.comp_uuid = self.comp.uuid
        write_display_options_to_json(self.comp.disp_opts_files[cat_idx], categories)
        self.comp.data.info.attrs[attr_name] = categories.uuid

    @staticmethod
    def _update_display_options(new_path, file, new_comp_uuid, attribute_key, new_data):
        """Updates the display options for a structure component.

        Args:
            new_path: A string representing the new path where the file will be saved.
            file: A string representing the path to the file.
            new_comp_uuid: A string representing the new component UUID.
            attribute_key: A string representing the attribute key.
            new_data: The new data object.
        """
        basename = os.path.basename(file)
        fname = os.path.join(new_path, basename)
        json_dict = read_display_options_from_json(fname)
        json_dict['uuid'] = str(uuid.uuid4())
        json_dict['comp_uuid'] = new_comp_uuid

        categories = CategoryDisplayOptionList()
        categories.from_dict(json_dict)

        write_display_options_to_json(fname, categories)
        new_data.info.attrs[attribute_key] = json_dict['uuid']  # Store display list UUID in component data.

    def update_all_id_files(self, comp_path):
        """Update all the id files.

        Args:
            comp_path (str): Path to the component's folder
        """
        for flux_function in scd.FLUX_FUNCTION_TYPES:
            # Arcs
            id_file = self.get_display_id_file(flux_function, TargetType.arc, comp_path)
            vals = self.comp.data.arcs.where(self.comp.data.arcs.flux_function == flux_function, drop=True)
            self._write_display_option_id_file(vals, id_file)

            # Polygons
            id_file = self.get_display_id_file(flux_function, TargetType.polygon, comp_path)
            vals = self.comp.data.polygons.where(self.comp.data.polygons.flux_function == flux_function, drop=True)
            self._write_display_option_id_file(vals, id_file)

    def update_display_options_file(self, new_main_file, new_path):
        """Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file (str): Path to the new component's main file
            new_path (str): The new component's directory.
        """
        new_data = scd.StructureData(new_main_file)
        # If duplicating, clear the coverage UUID. Will query XMS for the new coverage's UUID on the create event.
        new_data.info.attrs['cov_uuid'] = ''

        # Set component UUID in display options file.
        new_comp_uuid = os.path.basename(new_path)

        self._update_display_options(new_path, self.comp.disp_opts_files[0], new_comp_uuid, 'arc_display_uuid',
                                     new_data)
        self._update_display_options(new_path, self.comp.disp_opts_files[1], new_comp_uuid, 'poly_display_uuid',
                                     new_data)

        new_data.commit()

    def assign_structure(self, params, parent, target_type):
        """Display the Assign BC dialog and persist data if accepted.

        Args:
            params (dict): The ActionRequest parameter map
            parent (QWidget): The parent window
            target_type (TargeType): The feature type

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # The old comp_ids are the comp_ids of the selected features after checking selected types
        old_comp1_id, old_comp2_id = self._unpack_xms_data(params[0], target_type=target_type)
        # Now get the comp_ids of the selected features regardless of the set rules
        load_arcs = True if target_type == TargetType.arc else False
        load_polys = True if target_type == TargetType.polygon else False
        self._query.load_component_ids(component=self.comp, arcs=load_arcs, polygons=load_polys)
        comp1_id = self.comp.get_comp_id(entity_type=target_type, xms_id=self.selected_att_ids[0])
        comp2_id = UNINITIALIZED_COMP_ID
        if len(self.selected_att_ids) > 1:
            comp2_id = self.comp.get_comp_id(entity_type=target_type, xms_id=self.selected_att_ids[1])
        # Get the selected feature
        comp1_att_ids, comp2_att_ids = self._get_feature_ids_from_comp_ids(target_type, comp1_id, comp2_id)

        # If the first feature has a component ID but the second does not, we either have a single select or a
        # multi-select where all the features have the same attributes
        single_select = old_comp1_id != UNINITIALIZED_COMP_ID and old_comp2_id == UNINITIALIZED_COMP_ID
        increment = True
        num_selected = len(self.selected_att_ids)

        comp1_id = old_comp1_id
        comp2_id = old_comp2_id
        swap_comps = False  # Flag so the individual attributes of feature in a set stay consistent in the GUI
        if comp1_id == UNINITIALIZED_COMP_ID and comp2_id == UNINITIALIZED_COMP_ID:
            self._set_multi_select_warning(target_type)
            # Create some default atts if this is the initial assignment.
            comp1_id = self.comp.data.add_structure_atts(target_type=target_type, dset=None)
            comp1_att_ids = [self.selected_att_ids[0]]
            if num_selected == 2:
                comp2_att_ids = [self.selected_att_ids[1]]
                comp2_id = self.comp.data.add_structure_atts(target_type=target_type, dset=None)
            increment = False
        dset = self.comp.data.arcs if target_type == TargetType.arc else self.comp.data.polygons
        struct1_data = dset.where(dset.comp_id == comp1_id, drop=True)
        struct2_data = None
        # Snag the old flux function value. We will need it on accept().
        old_flux_function = struct1_data.flux_function.data.item()
        single_select_set = single_select and old_flux_function in scd.FLUX_FUNCTION_SET_TYPES
        # Get the data for the other feature in a linked structure
        if num_selected == 2:
            struct2_data = dset.where(dset.comp_id == comp2_id, drop=True)
            # Check if we are a double selection of a set.
            mask1 = self.comp.data.sets['comp1_id'] == comp1_id
            mask2 = self.comp.data.sets['comp2_id'] == comp2_id
            ds = self.comp.data.sets.where(mask1 & mask2, drop=True)
            if ds.sizes['set_id'] == 1:
                # Don't increment the component id if we have two features of the same set selected.
                increment = False
        elif single_select_set:  # Get the attributes of the other feature in a linked structure, if it exists.
            comp2_id, swap_comps = self.comp.data.find_partner(comp1_id)
            struct2_data = dset.where(dset.comp_id == comp2_id, drop=True)
            # Update the second feature's att id now that we know its comp id.
            _, comp2_att_ids = self._get_feature_ids_from_comp_ids(target_type, comp1_id, comp2_id)
            if not comp2_att_ids:  # Check if the user deleted one of the map features in a set.
                msg = (f'Unable to find all features that define the structure of type: {old_flux_function}.')
                message_with_ok(parent=parent, message=msg, app_name=XmEnv.xms_environ_app_name(), icon='Warning',
                                win_icon=QIcon(get_xms_icon()))
                comp1_id = self.comp.data.add_structure_atts(target_type=target_type, dset=None)
                struct2_data = None
                single_select_set = False

        # Can only assign two polygons for a linked zone structure. If one selected, display the attributes for the set.
        if target_type == TargetType.polygon and num_selected != 2 and not single_select_set:
            message_with_ok(
                parent=parent,
                message='Select two polygons to assign a linked zone structure.',
                app_name='SMS',
                icon='Critical',
                win_icon=QIcon(get_xms_icon())
            )
            return [], []

        # Increment the component id anytime dialog triggered except from single-select structure set pair.
        if increment and num_selected != 1:
            comp1_id = self.comp.data.add_structure_atts(target_type=target_type, dset=struct1_data)
            if struct2_data is not None:
                comp2_id = self.comp.data.add_structure_atts(target_type=target_type, dset=struct2_data)

        # Find the feature ids of the selected feature set (if applicable) and swap if necessary.
        if swap_comps and struct2_data is not None:
            # If this is a set, make sure the features consistently show up in the same order in the GUI.
            feature1_id = comp2_att_ids[0] if len(comp2_att_ids) > 0 else -1
            feature2_id = comp1_att_ids[0] if len(comp1_att_ids) > 0 else -1
            d = copy.deepcopy(struct2_data)
            struct2_data = struct1_data
            struct1_data = d
            c = comp2_id
            comp2_id = comp1_id
            comp1_id = c
        else:
            feature1_id, feature2_id = self._find_feature_ids(num_selected, comp1_att_ids, comp2_att_ids)

        # Convert our data to the gmi format required by the dialog.
        struct = sp.StructureDef(num_selected=num_selected, target_type=target_type,
                                 single_select_set=single_select_set, feature1_id=feature1_id, feature2_id=feature2_id)
        gm_def = struct.structure_def()
        mapper = StructureGmiMapper(gm=gm_def, data1=struct1_data, data2=struct2_data)
        gm = mapper.to_gmi()
        dlg = GroupSetDialog(
            parent=parent,
            section=gm.arc_parameters,
            get_curve=None,
            add_curve=None,
            is_interior=False,
            dlg_name='xms.tuflowfv.structure_component_display_assign_structure',
            window_title='Assign Structure',
            multi_select_message=self.dlg_message,
            show_groups=True
        )
        if dlg.exec():
            # Convert and concatenate the GMI parameters back to our xarray dataset.
            dlg_data1, dlg_data2 = mapper.to_xarray()
            flux_function = dlg_data1.flux_function.data.item()
            is_set = (flux_function in scd.FLUX_FUNCTION_SET_TYPES) and (single_select_set or num_selected == 2)

            comp1_id, comp2_id = self._check_sets(target_type=target_type, flux_function=flux_function,
                                                  old_flux_function=old_flux_function, dlg_data1=dlg_data1,
                                                  dlg_data2=dlg_data2, comp1_id=comp1_id, comp2_id=comp2_id,
                                                  comp1_att_ids=comp1_att_ids)

            # Update struct type needs to come after set checks to handle linked sets
            self.comp.data.update_struct_type(target_type=target_type, is_set=is_set, comp_id=comp1_id)
            if comp2_id != UNINITIALIZED_COMP_ID:
                self.comp.data.update_struct_type(target_type=target_type, is_set=is_set, comp_id=comp2_id)

            # Flush the attribute dataset to disk
            self.comp.data.commit()

            # Delete the id dumped by xms files.
            shutil.rmtree(os.path.join(os.path.dirname(self.comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def _check_sets(self, target_type, flux_function, old_flux_function, dlg_data1, dlg_data2, comp1_id, comp2_id,
                    comp1_att_ids):
        """Check sets and update attributes in the structure component.

        Args:
            target_type (TargetType): The target type of the structure component.
            flux_function (str): The new flux function of the structure component.
            old_flux_function (str): The previous flux function of the structure component.
            dlg_data1 (Dataset): The updated attribute data for the first feature.
            dlg_data2 (Dataset or None): The updated attribute data for the second feature, if applicable.
            comp1_id (int): The component ID of the first feature.
            comp2_id (int): The component ID of the second feature, if applicable.
            comp1_att_ids (list): The attribute IDs of the first feature.

        Returns:
            tuple: A tuple containing the comp_ids of the first and second features.
        """
        if flux_function == old_flux_function:
            # Feature still has the same flux function. Store the updated attributes without incrementing the component
            # id to preserve set relationship.
            self.comp.data.update_structure(target_type=target_type, comp_id=comp1_id, new_atts=dlg_data1)
            if dlg_data2 is not None:
                self.comp.data.update_structure(target_type=target_type, comp_id=comp2_id, new_atts=dlg_data2)
        else:
            # Check if we changed the type value of a double selected set. If not, only update atts to preserve set.
            if (dlg_data2 is not None
                    and flux_function in scd.FLUX_FUNCTION_SET_TYPES
                    and old_flux_function in scd.FLUX_FUNCTION_SET_TYPES
            ):
                already_linked = True
                self.comp.data.update_structure(target_type=target_type, comp_id=comp1_id, new_atts=dlg_data1)
                self.comp.data.update_structure(target_type=target_type, comp_id=comp2_id, new_atts=dlg_data2)
            else:  # The user may have broken a set. Give the selected features new component ids.
                already_linked = False
                comp1_id = self.comp.data.add_structure_atts(target_type=target_type, dset=dlg_data1)
                if dlg_data2 is not None:
                    comp2_id = self.comp.data.add_structure_atts(target_type=target_type, dset=dlg_data2)
                self._cleanup_broken_sets(target_type=target_type)

            # Add a set relationship if one did not already exist and the new flux function is a set type.
            if not already_linked and flux_function in scd.FLUX_FUNCTION_SET_TYPES:
                # Create a new entry in the sets table for this linked structure
                self.comp.data.add_linked_structure(comp1_id=comp1_id, comp2_id=comp2_id)

        # Associate all selected features with a new component id.
        for att_id in self.selected_att_ids:
            if att_id in comp1_att_ids:  # Not a set or first feature in a set
                comp_id = int(comp1_id)  # Use int cast because sometimes it is a numpy data type
            else:  # Second feature in the set
                comp_id = int(comp2_id)  # Use int cast because sometimes it is a numpy data type
            self.comp.update_component_id(target_type, att_id, comp_id)

        # Write the updated comp id files for the necessary structure types.
        self._update_id_files(flux_function=flux_function, target_type=target_type, new_comp_id=comp1_id)
        if dlg_data2 is not None:
            self._update_id_files(flux_function=flux_function, target_type=target_type, new_comp_id=comp2_id)
        return comp1_id, comp2_id

    def display_options(self, parent, cov_uuid):
        """Shows the display options dialog for a display list.

        Args:
            parent (QWidget): The parent window
            cov_uuid (str): UUID of the coverage geometry associated with the display list, if any.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Get the arc display options.
        arc_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[0])
        arc_categories.from_dict(json_dict)
        categories_list = [arc_categories]

        # Get the polygon display options.
        polygon_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[1])
        polygon_categories.from_dict(json_dict)
        categories_list.append(polygon_categories)

        wkt = ''
        if 'wkt' in arc_categories.projection:
            wkt = arc_categories.projection['wkt']

        reg_keys = [REG_KEY_STRUCT_ARC, REG_KEY_STRUCT_POLY]
        dlg = CategoryDisplayOptionsDialog(category_lists=categories_list, parent=parent, package_name='xmstuflowfv',
                                           registry_keys=reg_keys)

        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec() and cov_uuid:
            # write display options JSON files
            category_lists = dlg.get_category_lists()
            for idx, category_list in enumerate(category_lists):
                category_list.projection['wkt'] = wkt
                write_display_options_to_json(self.comp.disp_opts_files[idx], category_list)
                self.comp.display_option_list.append(
                    XmsDisplayMessage(file=self.comp.disp_opts_files[idx], edit_uuid=cov_uuid,
                                      draw_type=DrawType.draw_at_ids)
                )
            # Save the default display options to the registry.
            self._update_coverage_registry_defaults(REG_KEY_STRUCT_ARC)
            self._update_coverage_registry_defaults(REG_KEY_STRUCT_POLY)
        return [], []

    def ensure_display_option_files_exist(self):
        """Copies default BC arc and point display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists.
        """
        if not os.path.isfile(self.comp.disp_opts_files[0]):  # First is arcs, second is polygons
            self._get_and_write_categories(STRUCT_ARC_JSON, REG_KEY_STRUCT_ARC, 'arc_display_uuid', 0)
            self._get_and_write_categories(STRUCT_POLYGON_JSON, REG_KEY_STRUCT_POLY, 'poly_display_uuid', 1)
            self.comp.data.commit()

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (Optional[dict]): Generic map of parameters, unused

        Returns:
            Empty message and ActionRequest lists
        """
        # Query XMS for parent coverage's UUID if we don't know it yet.
        if not self.comp.cov_uuid:
            self.comp.cov_uuid = query.parent_item_uuid()
            self.comp.data.info.attrs['cov_uuid'] = self.comp.cov_uuid
            self.comp.data.commit()
        if not self.comp.cov_uuid:
            return [('ERROR', 'Could not get the TUFLOWFV Structure coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.comp.main_file), sp.STRUCT_INITIAL_ARC_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            self._update_component_ids_from_files(TargetType.arc)
            self._update_component_ids_from_files(TargetType.polygon)
        # Send the display options json files to XMS.
        self.comp.display_option_list = [
            XmsDisplayMessage(file=self.comp.disp_opts_files[0], edit_uuid=self.comp.cov_uuid),  # arcs
            XmsDisplayMessage(file=self.comp.disp_opts_files[1], edit_uuid=self.comp.cov_uuid),  # polygons
        ]

    @staticmethod
    def get_display_id_file(flux_function, target_type, path):
        """Get the display option id file for a structure arc/polygon type.

        Args:
            flux_function (str): Structure type string. See structure_data.FLUX_FUNCTION_TYPES
            target_type (TargetType): The feature type
            path (str): Path to the folder containing the id files

        Returns:
            (str): See description
        """
        target_type_str = 'arc' if target_type == TargetType.arc else 'poly'
        filename = f'{flux_function}_{target_type_str}.display_ids'
        return os.path.join(path, filename)
