"""OutputPointsData class."""
# 1. Standard python modules
import os

# 2. Third party modules
import numpy as np
import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.data.tuflowfv_data import check_for_object_strings_dumb


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


OUTPUT_POINTS_MAIN_FILE = 'output_comp.nc'


def get_default_output_data_dict():
    """Returns an default data dict and the dtypes for an output point.

    Returns:
        tuple(dict, set): The default data and their dtypes
    """
    defaults = {
        'label': '',
        'comment': '',
        'vert_min': 0.0,
        'vert_max': 0.0,
    }
    dtypes = [
        object,
        object,
        object,
        np.float64,
        np.float64,
    ]
    return defaults, dtypes


def get_default_output_data(fill):
    """Returns an default data dict for an output point.

    Args:
        fill (bool): True if Dataset should be initialized with an default row, False if it should be empty

    Returns:
        dict: The default data
    """
    defaults, dtypes = get_default_output_data_dict()
    return {
        variable: ('comp_id', np.array([value] if fill else [], dtype=dtype))
        for dtype, (variable, value) in zip(dtypes, defaults.items())
    }


class OutputPointsData(XarrayBase):
    """Class for storing the TUFLOWFV Output Points properties."""

    def __init__(self, filename):
        """Constructor.

        Args:
            filename (str): file name
        """
        # Initialize member variables before calling super() so they are available for commit() call
        self._filename = filename
        self._info = None
        self._points = None
        # Create the default file before calling super because we have our own attributes to write.
        self._get_default_datasets(filename)
        super().__init__(filename)

    def _get_default_datasets(self, data_file):
        """Create default datasets if needed.

        Args:
            data_file (str): Name of the data file. If it doesn't exist, it will be created.
        """
        if not os.path.exists(data_file) or not os.path.isfile(data_file):
            info = {
                'FILE_TYPE': 'TUFLOWFV_OUTPUT_DATA',
                'VERSION': pkg_resources.get_distribution('xmstuflowfv').version,
                'cov_uuid': '',
                'next_comp_id': 0,
                'export_format': 'CSV',
                'display_uuid': '',
            }
            self._info = xr.Dataset(attrs=info)
            coords = {'comp_id': []}
            points_table = {
                'label': ('comp_id', np.array([], object)),
                'comment': ('comp_id', np.array([], object)),
                'vert_min': ('comp_id', np.array([], np.float64)),
                'vert_max': ('comp_id', np.array([], np.float64)),
            }
            self._points = xr.Dataset(data_vars=points_table, coords=coords)
            self.commit()

    @staticmethod
    def _get_new_atts(comp_id):
        """Get a new dataset with default attributes for an output point.

        Args:
            comp_id (int): The unique XMS component id of the output point. If UNINITIALIZED_COMP_ID, a new one is
                generated.

        Returns:
            (xr.Dataset): A new default dataset for an output point. Can later be concatenated to persistent dataset.
        """
        coords = {'comp_id': [comp_id]}
        points_table = {
            'label': ('comp_id', np.array([''], object)),
            'comment': ('comp_id', np.array([''], object)),
            'vert_min': ('comp_id', np.array([0.0], np.float64)),
            'vert_max': ('comp_id', np.array([0.0], np.float64)),
        }
        return xr.Dataset(data_vars=points_table, coords=coords)

    @property
    def points(self):
        """Get the output points component data parameters.

        Returns:
            xarray.Dataset: The material list dataset

        """
        if self._points is None:
            self._points = self.get_dataset('points', False)
        return self._points

    @points.setter
    def points(self, dset):
        """Setter for the output points dataset."""
        if dset:
            self._points = dset

    def update_point(self, comp_id, new_atts):
        """Update the output attributes of a point.

        Args:
            comp_id (int): Component id of the output point to update
            new_atts (xr.Dataset): The new attributes for the output point
        """
        self.points['label'].loc[dict(comp_id=[comp_id])] = new_atts['label']
        self.points['comment'].loc[dict(comp_id=[comp_id])] = new_atts['comment']
        self.points['vert_min'].loc[dict(comp_id=[comp_id])] = new_atts['vert_min']
        self.points['vert_max'].loc[dict(comp_id=[comp_id])] = new_atts['vert_max']

    def add_output_points(self, dset=None):
        """Adds the output point attributes from output_data to this instance of OutputPointsData.

        Args:
            dset (Optional[OutputPointsData]): another OutputPointsData instance, default if not provided

        Returns:
            dict: The old ids of the output_data as key and the new ids as the data
        """
        try:
            new_comp_id = int(self.info.attrs['next_comp_id'])
            self.info.attrs['next_comp_id'] += 1  # Increment the unique XMS component id.
            if dset is None:  # Generate a new default Dataset
                dset = self._get_new_atts(comp_id=new_comp_id)
            else:  # Update the component id of an existing Dataset
                dset.coords['comp_id'] = [new_comp_id for _ in dset.coords['comp_id']]
            self._points = xr.concat([self.points, dset], 'comp_id')
            return new_comp_id
        except Exception:
            return UNINITIALIZED_COMP_ID

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._points is not None:
            self._points.close()
            self._drop_h5_groups(['points'])
            check_for_object_strings_dumb(self._points, ['label', 'comment'])
            self._points.to_netcdf(self._filename, group='points', mode='a')

    def vacuum(self):
        """Rewrite all data to a new file to reclaim disk space."""
        _ = self.info
        _ = self.points
        filesystem.removefile(self._filename)
        self.commit()
