"""StructureGmiMapper class."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Type

# 4. Local modules


CULVERT_TYPE_LOOKUP = {
    'circular': 1,
    'rectangular': 2,
    'gated circular (unidirectional)': 4,
    'gated rectangular (unidirectional)': 5,
}
CULVERT_TYPE_INVERT_LOOKUP = {  # Get the user-friendly combobox text option
    1: 'Circular',
    2: 'Rectangular',
    4: 'Gated circular (unidirectional)',
    5: 'Gated rectangular (unidirectional)',
}


class StructureGmiMapper:
    """Class for mapping xarray data to and from GMI formatted data dict."""
    def __init__(self, gm, data1, data2):
        """Constructor.

        Args:
            gm (GenericModel): The generic model
            data1 (xarray.Dataset): The xarray data. Should be sliced to a single `comp_id` row.
            data2 (xarray.Dataset): The xarray data for a set's other feature, if it exists. Should be sliced to a
                                    single `comp_id` row.
        """
        self._gm = gm
        self._data = data1
        self._other_data = data2
        self._have_other = self._other_data is not None and self._other_data.sizes['comp_id'] > 0

    @staticmethod
    def _map_culvert_types(val, to_int):
        """Convert between integer TUFLOWFV culvert type values and user-friendly GUI combobox option text.

        Args:
            val ([int, str]): The value to convert. Should be an int if to_int is True
            to_int (bool): True if we are converting the combobox option text to its TUFLOWFV integer value

        Returns:
            See description
        """
        if to_int:
            return CULVERT_TYPE_LOOKUP[val.lower()]
        return CULVERT_TYPE_INVERT_LOOKUP[int(float(val))]  # float is more permissive than int

    def to_xarray(self):
        """Convert the GMI data dict to an xarray dataset.

        Returns:
            tuple:
                xarray.Dataset: An xarray dataset containing the generic model's data dict.
                xarray.Dataset: An xarray dataset containing the generic model's second linked feature data dict, if it
                    exists.
        """
        gp = self._gm.arc_parameters
        for group_name in gp.group_names:
            grp = gp.group(group_name)
            if not grp.is_active:
                continue  # Only update attributes from the active group
            else:  # The groups are the flux functions
                self._data['flux_function'].values[0] = group_name
                if self._have_other:
                    self._other_data['flux_function'].values[0] = group_name
            for param_name in grp.parameter_names:
                param = grp.parameter(param_name)
                val = param.value

                # Convert the "upstream" value from the option-box-compatible strings to an int
                if param_name == 'upstream':
                    val = 1 if val == 'Name 1' else 0

                # Check for sets so we can keep the name and flux function of the other feature in sync
                if param_name == 'name2':  # Special case of a set so we keep the names of the features in sync
                    if self._have_other:
                        self._other_data['name'].values[0] = param.value
                    continue
                elif param_name == 'type':  # Map from combobox option text to TUFLOWFV integer value.
                    culvert_type = self._map_culvert_types(val=param.value, to_int=True)
                    self._data['type'].values[0] = culvert_type
                    if self._have_other:
                        self._other_data['type'].values[0] = culvert_type
                    continue

                if param.parameter_type == Type.BOOLEAN:
                    val = 1 if val else 0
                self._data[param_name].values[0] = val
                if self._have_other:
                    self._other_data[param_name].values[0] = val
                # Make sure upstream/downstream OR is in sync with the pair
                if param_name == 'upstream' and self._have_other:
                    self._other_data['upstream'].values[0] = not val
        # Return None if the flux function is not a set. Easier to check than an empty Dataset.
        other_data = self._other_data if self._have_other else None
        return self._data, other_data

    def to_gmi(self):
        """Convert the xarray dataset to a GMI data dict.

        Returns:
            GenericModel: The generic model containing the xarray data.
        """
        active_type = self._data.flux_function.data[0]
        gp = self._gm.arc_parameters
        for group_name in gp.group_names:
            grp = gp.group(group_name)
            if group_name == active_type:
                grp.is_active = True
            for param_name in grp.parameter_names:
                param = grp.parameter(param_name)

                # Convert the "upstream" value from the int to the option-box-compatible strings
                if param_name == 'upstream':
                    param.value = 'Name 1' if self._data['upstream'].data[0] else 'Name 2'
                    continue

                # Special case for sets. Need to make sure the name of both features stay in sync. 'name2' is not really
                # a dataset variable. It is the 'name' variable in the set's other feature.
                if param_name == 'name2':
                    if self._have_other:
                        param.value = self._other_data['name'].data[0]
                    continue
                elif param_name == 'type':  # Map the integer TUFLOWFV value to combobox option text
                    param.value = self._map_culvert_types(val=self._data['type'].values[0], to_int=False)
                    continue

                if param.parameter_type == Type.BOOLEAN:
                    param.value = True if self._data[param_name].data[0] else False
                elif param.parameter_type == Type.INTEGER:
                    param.value = int(self._data[param_name].data[0])
                else:
                    param.value = self._data[param_name].data[0]
        return self._gm
