"""Provide progress feedback when running an TUFLOWFV model from SMS."""
# 1. Standard python modules
import datetime
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.core.filesystem import filesystem as xfs
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.tuflowfv.dmi.xms_data import XmsData
from xms.tuflowfv.file_io import io_util
from xms.tuflowfv.file_io.bc_csv_reader import parse_tuflowfv_time


SEC_PER_HOUR = 3600.0


class TuflowFvTracker:
    """Class to track the progress of a running TUFLOWFV model."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    run_duration = 0.0  # In hours
    isodate_start = None  # Simulation/hotstart start time if time format is TUFLOWFV ISODATE
    hours_start = 0.0  # Simulation/hotstart start time if time format is hours

    @staticmethod
    def progress_function():
        """Method called inside the progress loop for computing percent done."""
        need_progress_update = False
        current_progress = 0.0
        try:  # Parse the command line output to determine current timestep
            with open(TuflowFvTracker.echo_file, 'r') as f:
                f.seek(TuflowFvTracker.echo_pos)  # Skip past lines we already read
                echo_line = f.readline()
                while echo_line:  # Have a complete line
                    # Look for a time duration line
                    if (echo_line.endswith('\n') or echo_line.endswith('\r')) and echo_line.strip().startswith('t = '):
                        echo_vals = echo_line.split()
                        try:
                            if TuflowFvTracker.isodate_start:
                                isodate = echo_vals[2] + ' ' + echo_vals[3].strip('.')  # Remove trailing period
                                current_time = parse_tuflowfv_time(isodate).to_pydatetime()
                                time_delta = current_time - TuflowFvTracker.isodate_start
                                current_time = time_delta / datetime.timedelta(hours=1)
                            else:
                                current_time = float(echo_vals[2]) - TuflowFvTracker.hours_start
                            current_progress = (current_time / TuflowFvTracker.run_duration) * 100.0
                        except ValueError:
                            pass
                        need_progress_update = True
                        TuflowFvTracker.echo_pos = f.tell()  # Store position in file of lines already read
                    echo_line = f.readline()  # Keep reading until EOF
        except Exception:
            pass  # File might not exist yet

        if need_progress_update:  # Update model process progress in SMS Simulation Run Queue
            TuflowFvTracker.query.update_progress_percent(current_progress)

    @staticmethod
    def compute_run_duration():
        """Get the model run duration.

        Returns:
            float: The run time in hours
        """
        xms_data = XmsData(TuflowFvTracker.query)
        data = xms_data.sim_data
        # If we are using a hot start, check if we are using the hot start file time
        restart_time = None
        if data.initial_conditions.attrs['use_restart_file'] and data.initial_conditions.attrs['use_restart_file_time']:
            restart = xfs.resolve_relative_path(data.info.attrs['proj_dir'],
                                                data.initial_conditions.attrs['restart_file'])
            restart_time = io_util.read_restart_file_time(restart)

        if data.time.attrs['use_isodate']:  # Time format is TUFLOWFV ISODATE
            if restart_time is not None:  # Restart file time is seconds from the reference datetime
                TuflowFvTracker.isodate_start = string_to_datetime(data.time.attrs['ref_date'])
                TuflowFvTracker.isodate_start += datetime.timedelta(seconds=restart_time)
            else:  # Either not using a restart or using a restart with the simulation start datetime
                TuflowFvTracker.isodate_start = string_to_datetime(data.time.attrs['start_date'])
            end_time = string_to_datetime(data.time.attrs['end_date'])
            duration = (end_time - TuflowFvTracker.isodate_start).total_seconds()
            return duration / SEC_PER_HOUR
        else:  # Time format is hours
            if restart_time is not None:  # Restart file time is seconds from the reference time
                TuflowFvTracker.hours_start = data.time.attrs['ref_date_hours'] + (restart_time / SEC_PER_HOUR)
            else:  # Either not using a restart or using a restart with the simulation start time
                TuflowFvTracker.hours_start = data.time.attrs['start_hours']
            return data.time.attrs['end_hours'] - TuflowFvTracker.hours_start

    @staticmethod
    def start_tracking():
        """Entry point for the TUFLOWFV  progress script."""
        # Export all TUFLOWFV executables to the same folder.
        io_util.move_to_shared_folder(os.getcwd())
        TuflowFvTracker.query = Query(progress_script=True)
        TuflowFvTracker.run_duration = TuflowFvTracker.compute_run_duration()
        TuflowFvTracker.prog = TuflowFvTracker.query.xms_agent.session.progress_loop
        TuflowFvTracker.prog.set_progress_function(TuflowFvTracker.progress_function)
        TuflowFvTracker.echo_file = TuflowFvTracker.prog.command_line_output_file
        TuflowFvTracker.prog.start_loop()
