"""Creates a Boundary Conditions coverage hidden component and Datasets."""
# 1. Standard python modules
import logging
import os
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids

# 4. Local modules
from xms.tuflowfv.components import bc_component as bcc
from xms.tuflowfv.components import bc_component_display as bc_disp
from xms.tuflowfv.components.tuflowfv_component import get_component_data_object
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.file_io.io_util import create_component_folder


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def build_global_bc_dataset(sim_data, global_bcs, gridded):
    """Build an xarray Dataset for multiple global/gridded BC types (stored in the simulation data).

    Args:
        sim_data (SimData): The simulation to add the global BCs to. Will overwrite any previously defined global BCs.
        global_bcs (list[dict]): List of attributes for each global/gridded BC. Keys should match data variables in the
            BcData 'bcs' Dataset. All parameters need to be present for each global BC (i.e. this method does not fill
            in default data values if missing).
        gridded (bool): True if the is a spatially varying global BC, False if a global time series
    """
    data_vars_dict = {variable: [] for variable in global_bcs[0] if variable != 'curve'}
    if gridded:  # Spatially varying gridded BCs
        sim_data.gridded_bcs = bcd.BcData(sim_data._filename, group_path=bcd.GRIDDED_BC_GROUP)
        dset = sim_data.gridded_bcs
    else:  # Global BCs (non-spatially varying)
        sim_data.global_bcs = bcd.BcData(sim_data._filename, group_path=bcd.GLOBAL_BC_GROUP)
        dset = sim_data.global_bcs

    for comp_id, global_bc in enumerate(global_bcs):  # Build a data cube
        bc_type = global_bc['type']
        for variable, value in global_bc.items():
            if variable == 'curve':
                dset.set_bc_curve(comp_id=comp_id + 1, bc_type=bc_type, dataset=value)
            else:
                data_vars_dict[variable].append(value)
    data_vars = {variable: ('comp_id', np.array(values)) for variable, values in data_vars_dict.items()}
    coords = {'comp_id': np.arange(1, len(global_bcs) + 1)}
    dset.bcs = xr.Dataset(data_vars=data_vars, coords=coords)
    dset.info.attrs['next_comp_id'] = len(global_bcs) + 1


def build_bc_dataset(bc_dict, comp_id=bcd.UNINITIALIZED_COMP_ID):
    """Build an xarray Dataset for a single BC arc/point and fill with values read from the .fvc file.

    Args:
        bc_dict (dict): The variables for the BC arc/point read from the .fvc file. Keys should match data variables
            in the BcData 'bcs' and 'points' Datasets.
        comp_id (int): Comp id to assign to the new dataset. Leave default if appending the BC to the BcData as that
            will increment the component id to the next available one.

    Returns:
        xr.Dataset: See description
    """
    default_data = bcd.get_default_bc_data(fill=True, gridded=False)
    coords = {'comp_id': [comp_id]}  # This will get reset when we append to the entire Dataset
    dset = xr.Dataset(data_vars=default_data, coords=coords)
    for variable, value in bc_dict.items():
        # TODO: The second part of this if is a temporary kludge until we support all the modules that can add
        #       variable values to the BC (e.g. sediment, particle tracing)
        if variable == 'curve' or variable not in dset:
            continue  # Don't add the BC curve here, we will add it after adding it to the entire Dataset
        dset[variable].loc[dict(comp_id=[comp_id])] = value
    return dset


class BcComponentBuilder:
    """Class for building BC components."""

    def __init__(self, cov_uuid, from_2dm, bc_atts, nodestring_id_to_feature_id, nodestring_id_to_feature_name):
        """Constructor.

        Args:
            cov_uuid (str): UUID of the component's coverage geometry
            from_2dm (bool): True if coverage is read from the .2dm file or the points-only coverage in the .fvc file.
                If False (read from a GIS shapefile), we will export as a shapefile when exporting the simulation.
            bc_atts (dict): {bc_id: {variable: value}} The BC attributes read from the CSV
            nodestring_id_to_feature_id (dict): Mapping of nodestring id in the file to feature arc id we created
            nodestring_id_to_feature_name (dict): Mapping of nodestring id in the file to feature arc name from the
                file. Since the feature names are optional, there may not be a value for a given key
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._cov_uuid = cov_uuid
        self._from_2dm = from_2dm
        self._bc_atts = bc_atts
        self._nodestring_id_to_feature_id = nodestring_id_to_feature_id
        self._nodestring_id_to_feature_name = nodestring_id_to_feature_name
        self._main_file = ''

    def _initialize_component(self):
        """Create the component's folder and data_object.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new BC component
        """
        # Create a new UUID and folder for the component data
        comp_uuid = str(uuid.uuid4())
        bc_comp_dir = create_component_folder(comp_uuid)
        self._main_file = os.path.join(bc_comp_dir, bcd.BC_MAIN_FILE)
        # Create the data_object Component to send back to SMS
        return get_component_data_object(main_file=self._main_file, comp_uuid=comp_uuid, unique_name='BcComponent')

    def _initialize_display(self, bc_comp, arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids, poly_att_ids,
                            poly_comp_ids):
        """Initialize the component display.

        Args:
            bc_comp (BcComponent): The Python BC component object
            arc_att_ids (list[int]): The arc feature ids
            arc_comp_ids (list[int]): The arc component ids
            point_att_ids (list[int]): The point feature ids
            point_comp_ids (list[int]): The point component ids
            poly_att_ids (list[int]): The polygon feature ids
            poly_comp_ids (list[int]): The polygon component ids
        """
        display_helper = bc_disp.BcComponentDisplay(bc_comp)
        display_helper.update_id_files()
        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        bc_comp_dir = os.path.dirname(self._main_file)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_POINT_ATT_ID_FILE)
        write_display_option_ids(id_file, point_att_ids)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_POINT_COMP_ID_FILE)
        write_display_option_ids(id_file, point_comp_ids)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_ARC_ATT_ID_FILE)
        write_display_option_ids(id_file, arc_att_ids)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_ARC_COMP_ID_FILE)
        write_display_option_ids(id_file, arc_comp_ids)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_POLY_ATT_ID_FILE)
        write_display_option_ids(id_file, poly_att_ids)
        id_file = os.path.join(bc_comp_dir, bc_disp.BC_INITIAL_POLY_COMP_ID_FILE)
        write_display_option_ids(id_file, poly_comp_ids)

    def _add_atts(self, bc_data):
        """Add attribute data for the features.

        Args:
            bc_data (BcData): The BcComponent data

        Returns:
            tuple: arc att ids, arc comp ids, point att ids, point comp ids
        """
        arc_att_ids = []
        arc_comp_ids = []
        point_att_ids = []
        point_comp_ids = []
        poly_att_ids = []
        poly_comp_ids = []
        for nodestring_id, feature_id in self._nodestring_id_to_feature_id.items():
            saved_nodestring_id = nodestring_id
            # Append a row to the Dataset for this BC
            bc_dict = self._bc_atts.get(nodestring_id)
            feature_name = nodestring_id = self._nodestring_id_to_feature_name.get(nodestring_id)
            if not bc_dict and self._nodestring_id_to_feature_name:  # Check if it had a string name in the file
                nodestring_id = feature_name
                bc_dict = self._bc_atts.get(nodestring_id)
            elif bc_dict and feature_name is None:
                feature_name = self._nodestring_id_to_feature_name.get(feature_id)
            if not bc_dict:
                continue  # Will become monitor

            if feature_name:
                bc_dict['name'] = str(feature_name)
            else:
                bc_dict['name'] = str(saved_nodestring_id)
            if bc_dict['type'] in bcd.ARC_BC_TYPES:
                bc_location = bcd.BC_LOCATION_ARC
            elif bc_dict['type'] in bcd.POINT_BC_TYPES:
                bc_location = bcd.BC_LOCATION_POINT
            else:  # QC_POLY
                bc_location = bcd.BC_LOCATION_POLY
            comp_id = bc_data.add_bc_atts(bc_location=bc_location, dset=build_bc_dataset(bc_dict))

            # Add the BC curve
            bc_curve = bc_dict.get('curve')
            if bc_curve is not None:
                bc_data.set_bc_curve(comp_id=comp_id, bc_type=bc_dict['type'], dataset=bc_curve)

            # Keep track of the component and feature ids for initializing display later.
            if bc_location == bcd.BC_LOCATION_ARC:
                arc_comp_ids.append(comp_id)
                arc_att_ids.append(feature_id)
            elif bc_location == bcd.BC_LOCATION_POINT:
                point_comp_ids.append(comp_id)
                point_att_ids.append(feature_id)
            else:  # bcd.BC_LOCATION_POLY
                poly_comp_ids.append(comp_id)
                poly_att_ids.append(feature_id)
        return arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids, poly_att_ids, poly_comp_ids

    def build_empty_bc_component(self):
        """Creates the BC coverage's hidden component but without any attributes.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new BC component
        """
        do_comp = self._initialize_component()
        # Load component data with stuff we read in
        bc_comp = bcc.BcComponent(self._main_file)
        comp_data = bc_comp.data
        comp_data.info.attrs['cov_uuid'] = self._cov_uuid
        comp_data.commit()
        # Set coverage UUID member on component so we can load component ids without committing data.
        bc_comp.cov_uuid = comp_data.info.attrs['cov_uuid']
        return do_comp

    def build_bc_component(self):
        """Creates the BC coverage's hidden component.

        This method is for building a BC coverage from scratch.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new BC component
        """
        # Set things up
        do_comp = self._initialize_component()
        bc_comp = bcc.BcComponent(self._main_file)
        comp_data = bc_comp.data
        comp_data.info.attrs['cov_uuid'] = self._cov_uuid
        comp_data.globals.attrs['export_format'] = '2dm' if self._from_2dm else 'Shapefile'
        bc_comp.cov_uuid = comp_data.info.attrs['cov_uuid']

        # Add the BC points/arcs/polygons
        arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids, poly_att_ids, poly_comp_ids = self._add_atts(
            comp_data
        )
        # Write data to disk
        comp_data.commit()
        self._initialize_display(bc_comp, arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids, poly_att_ids,
                                 poly_comp_ids
                                 )
        return do_comp
