"""Class to read a TUFLOWFV 2dm file."""
# 1. Standard python modules
import logging
import os
import shlex
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint.ugrid_builder import UGridBuilder
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import Arc, Coverage, Point, Projection, UGrid as DoUGrid
from xms.grid.ugrid import UGrid as XmUGrid

# 4. Local modules
from xms.tuflowfv.file_io.bc_component_builder import BcComponentBuilder
from xms.tuflowfv.file_io.material_component_builder import MaterialComponentBuilder


READ_BUFFER_SIZE = 100 * 1024


class TwoDMReader:
    """Class to read a 2dm file."""
    def __init__(self, filename, manager, wkt='', do_projection=None, create_bc_component=True,
                 create_mat_component=True):
        """Constructor.

        Args:
            filename (str): Path to the control file. If not provided (not testing or control file read), will retrieve
                from Query.
            wkt (str): Projection of the data in the file
            manager (StructureManager): Used to filter out structure arcs
            do_projection (Projection): The data_objects Projection created after reading the control file. If reading
                independently, we just use the current display projection in SMS.
            create_bc_component (bool): If True will create the BC component (empty if independent read). May be False
                when reading multiple simulations because we only need a portion of the data in the 2dm.
            create_mat_component (bool): If True will create the Materials component (empty if independent read). May be
                 False when reading multiple simulations because we only need a portion of the data in the 2dm.
        """
        self._logger = logging.getLogger('xms.coverage')
        self._filename = filename
        self._wkt = wkt  # Projection of all the data
        self._manager = manager
        self._do_projection = do_projection
        self._create_bc_component = create_bc_component
        self._create_mat_component = create_mat_component
        # Ensure no spaces in the name
        self._grid_name = os.path.basename(os.path.splitext(filename)[0]).replace(' ', '_')
        self._lines = []  # Lines of the .2dm file, indexed by self._line_num
        self._line_num = 0  # After reading the geometry, this will be the line index of the 'BEGPARAMDEF' card line.
        self._cells = []
        self._points = []
        self._nodestrings = []  # [[ns1_nd1, ... ns1_ndN], ...[nsN_nd1, ...nsN_ndN]]
        self._materials_on_grid = []  # Material assignment from file in order elements appear in the file
        self._point_locations = []  # The x,y,z point location tuples used to create BC arcs from nodestrings
        self._bc_points = {}  # Key is grid node index (0-based)
        self._current_nodestring = []

        self.do_ugrid = None  # data_objects UGrid for sending back to XMS, Always built
        self.cogrid = None  # CoGrid that has same data as do_ugrid, but the data_objects UGrid is kind of useless.
        self.point_id_idx = {}  # Mapping of grid 1-base point ids in file to XmUGrid point 0-base indices
        self.cell_id_idx = {}  # Mapping of grid 1-base element ids in file to XmUGrid cell 0-base indices
        self.bc_coverage = None  # Only built if 2dm contains nodestrings
        self.bc_comp = None
        self.nodestring_id_to_feature_id = {}  # Mapping of 1-base nodestring id in file to 1-base BC feature arc id
        self.nodestring_id_to_nodestring_name = {}  # Mapping of 1-base nodestring id in file to optional string in file
        self.material_coverage = None  # Always built but material assignment may or may not be meaningful
        self.material_comp = None
        self.material_assignments = {}  # Material ID to list of polygon IDs with that material assignment

    def _parse_header(self):
        """Read the lines at the beginning of the file."""
        try:
            found_name = False
            while not found_name:
                lexed_line = shlex.split(self._lines[self._line_num])
                if lexed_line:
                    card = lexed_line[0].strip().upper()
                    if card == 'MESHNAME':
                        found_name = True
                        self._grid_name = lexed_line[1].strip().strip('"\'').replace(' ', '_')
                    elif card in ['E3T', 'E4Q', 'ND', 'BEGPARAMDEF']:
                        return  # Older 2dm files did not have the MESHNAME card, give up if we start reading geometry.
                self._line_num += 1
        except Exception as e:
            self._logger.error(f'Invalid 2dm file format detected: {str(e)}')

    def _process_next_card(self):
        """Read the line in the file.

        Returns:
            bool: True if reading should continue, False if we are done.
        """
        if len(self._lines) <= self._line_num:
            return False
        line = self._lines[self._line_num].strip().split()
        self._line_num += 1
        if not line:
            return True
        card = line[0].upper()
        if card == 'NUM_MATERIALS_PER_ELEM':
            if len(line) < 2 or line[1] != '1':
                self._logger.warning('Invalid value found for NUM_MATERIALS_PER_ELEM')
            return True
        elif card == 'BEGPARAMDEF':  # Beginning of GMI model attribute section at bottom
            # Back up the current line number in case derived implementations want to continue reading model parameters
            # at the bottom of the 2dm file.
            self._line_num -= 1
            return False
        elif card in ['E3T', 'E4Q']:
            self._cells.append(line)
        elif card == 'ND':
            self._points.append(line)
        elif card == 'NS':
            self._current_nodestring.extend(line[1:])
            # With the final value being an optional string name, we need to check both [-2] and [-3]
            if int(line[-2]) < 0 or int(line[-3]) < 0:
                self._nodestrings.append(self._current_nodestring)
                self._current_nodestring = []
        else:
            self._logger.warning(f'Unrecognized data found on line {self._line_num}')
        return True

    def _build_mesh_points(self):
        """Build the grid locations list and create mapping between id in file and the point indices."""
        # Need the locations later for converting nodestrings to feature arcs, so store them off before constructing
        # the grid.
        self._point_locations = np.full((len(self._points), 3), 0.0)
        for point_idx, point_line in enumerate(self._points):
            self.point_id_idx[int(point_line[1])] = point_idx
            self._point_locations[point_idx][0] = float(point_line[2])
            self._point_locations[point_idx][1] = float(point_line[3])
            self._point_locations[point_idx][2] = float(point_line[4])

    def _build_cellstream_vals(self, idx):
        """Get the XmGrid cellstream definition for the current cell line.

        Returns:
            list: [XmUGrid.cell_type_enum, node1_id, ..., noden_id]
        """
        cell_line = self._cells[idx]
        self.cell_id_idx[int(cell_line[1])] = idx
        self._materials_on_grid.append(int(cell_line[-1]))
        if cell_line[0].upper() == 'E4Q':  # Quad
            return [
                XmUGrid.cell_type_enum.QUAD,
                4,
                self.point_id_idx[int(cell_line[2])],
                self.point_id_idx[int(cell_line[3])],
                self.point_id_idx[int(cell_line[4])],
                self.point_id_idx[int(cell_line[5])],
            ]
        else:  # Tri
            return [
                XmUGrid.cell_type_enum.TRIANGLE,
                3,
                self.point_id_idx[int(cell_line[2])],
                self.point_id_idx[int(cell_line[3])],
                self.point_id_idx[int(cell_line[4])],
            ]

    def _build_mesh(self, proj):
        """Build the data_objects UGrid.

        Args:
            proj (Projection): The data_objects projection
        """
        ugrid_uuid = str(uuid.uuid4())
        xmc_file = os.path.join(XmEnv.xms_environ_process_temp_directory(), f'{ugrid_uuid}.xmc')
        # Parse grid geometry from previously read lines
        self._build_mesh_points()
        cellstream = [stream_val for i in range(len(self._cells)) for stream_val in self._build_cellstream_vals(i)]
        # Write the CoGrid file
        xmugrid = XmUGrid(self._point_locations, cellstream)
        co_builder = UGridBuilder()
        co_builder.set_is_2d()
        co_builder.set_ugrid(xmugrid)
        self.cogrid = co_builder.build_grid()
        self.cogrid.write_to_file(xmc_file, True)
        # Create the data_objects UGrid to send back to XMS
        self.do_ugrid = DoUGrid(xmc_file, name=self._grid_name, uuid=ugrid_uuid, projection=proj)

    def _build_material_coverage(self, proj):
        """Build the data_objects Materials coverage.

        Args:
            proj (Projection): The data_objects projection
        """
        cov_name = f'{self._grid_name}_-_Materials'
        polygon_merger = GridCellToPolygonCoverageBuilder(
            co_grid=self.cogrid, dataset_values=self._materials_on_grid, projection=proj, coverage_name=cov_name
        )
        self.material_coverage = polygon_merger.create_polygons_and_build_coverage()
        self.material_assignments = polygon_merger.dataset_polygon_ids

    def _build_bc_coverage(self, proj):
        """Build the data_objects BC coverage.

        Args:
            proj (Projection): The data_objects projection
        """
        if not self._nodestrings:
            return
        cov_name = f'{self._grid_name}_-_Nodestrings'
        self.bc_coverage = Coverage(name=cov_name, uuid=str(uuid.uuid4()), projection=proj)
        self.bc_coverage.arcs = self._build_bc_arcs()
        self.bc_coverage.complete()

    def _build_bc_arcs(self):
        """Convert the nodestrings to feature BC arcs (geometry only).

        Returns:
            list[Arc]: See description
        """
        arcs = []
        for idx, nodestring_line in enumerate(self._nodestrings):
            feature_id = idx + 1  # Renumber nodestrings 1-N but store mapping to id in file for assigning atts later
            # nodestring_line = NS node1_id node2_id ... -(nodeN_id) nodestring_id

            # Check for optional string name at end of nodestring
            string_id = False
            if len(nodestring_line) > 2:
                if int(nodestring_line[-3]) < 0:
                    # If the optional string label is present, store it, then pop it off the list so that all the
                    # other processing doesn't have to deal with it.
                    # Remember, .pop() happens first, so nodestring_line[-1] is our actual end marker, even though it
                    # seems like it ought to be nodestring_line[-2]
                    self.nodestring_id_to_nodestring_name[int(nodestring_line[-1])] = nodestring_line.pop()
                    string_id = True

            self.nodestring_id_to_feature_id[int(nodestring_line[-1])] = feature_id
            if not string_id:
                self.nodestring_id_to_nodestring_name[int(nodestring_line[-1])] = str(feature_id)
            start_node = self._build_bc_point(self.point_id_idx[int(nodestring_line[0])])
            end_point_id = int(nodestring_line[-2])
            # Since the last node in a nodestring is made negative to mark the end of the string, we need to invert it.
            end_point_id *= -1
            end_node = self._build_bc_point(self.point_id_idx[end_point_id])
            vertices = [
                self._build_bc_point(self.point_id_idx[int(vertex_point_id)])
                for vertex_point_id in nodestring_line[1:-2]
            ]
            arcs.append(Arc(feature_id=feature_id, start_node=start_node, vertices=vertices, end_node=end_node))
        return arcs

    def _build_bc_point(self, point_index):
        """Get a point for the BC coverage - ensures we do not create duplicate points.

        Args:
            point_index (int):

        Returns:
            Point: The data_objects coverage point at the grid location.
        """
        do_point = self._bc_points.get(point_index)
        if not do_point:
            location = self._point_locations[point_index]
            do_point = Point(feature_id=len(self._bc_points) + 1, x=location[0], y=location[1], z=location[2])
            self._bc_points[point_index] = do_point
        return do_point

    def _build_xms_data(self):
        """Add all the imported data to the xmsapi Query to send back to SMS."""
        # Create the projection for geometric data
        proj = Projection(wkt=self._wkt)
        self._build_mesh(proj)
        self._build_material_coverage(proj)
        self._build_bc_coverage(proj)

    def _parse_2dm(self):
        """Top-level entry point for the 2dm reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()
            self._parse_header()
            while self._process_next_card():
                continue
            self._logger.info('Finished reading 2dm file. Building XMS data...')
            self._build_xms_data()
        except Exception as e:
            self._logger.error(f'Unexpected error reading 2dm file on line {self._line_num}: {str(e)}')

    def read(self, materials=None, bcs=None):
        """Top-level entry point for the 2dm reader.

        Args:
            materials (dict): Dict of the material data arrays. If not provided default values will be assigned to the
                material attributes. {mat_id: {'variable': value}}
            bcs (dict): Dict of the bc data arrays. If not provided default values will be assigned to the
                material attributes.
        """
        self._parse_2dm()
        if self.do_ugrid and self._do_projection:  # Set projection if a control file read
            self.do_ugrid.projection = self._do_projection
        if self.material_coverage and self._do_projection:  # Set projection if a control file read
            self.material_coverage.projection = self._do_projection
        if self._create_mat_component:
            # Build the Material component. Material assignments always present but may not be useful.
            builder = MaterialComponentBuilder(cov_uuid=self.material_coverage.uuid, from_2dm=True,
                                               poly_assignments=self.material_assignments,
                                               existing_data=materials if materials else {})
            self.material_comp = builder.build_material_component()
        if self.bc_coverage:
            if self._do_projection:  # Set projection if a control file read
                self.bc_coverage.projection = self._do_projection
            if self._create_bc_component:  # File contained bcs, build a BC component
                builder = BcComponentBuilder(cov_uuid=self.bc_coverage.uuid, from_2dm=True, bc_atts=bcs,
                                             nodestring_id_to_feature_id=self.nodestring_id_to_feature_id,
                                             nodestring_id_to_feature_name=self.nodestring_id_to_nodestring_name)
                if bcs is None:  # Reading the .2dm independently
                    self.bc_comp = builder.build_empty_bc_component()
                else:  # Map attributes read from the .fvc
                    self.bc_comp = builder.build_bc_component()
