"""The TUFLOWFV simulation model control dialog."""
# 1. Standard python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QDoubleValidator, QIntValidator
from PySide2.QtWidgets import QCheckBox, QHeaderView, QLabel, QListWidgetItem, QSplitter, QVBoxLayout

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.time_format import ISO_DATETIME_FORMAT
from xms.guipy.validators.number_corrector import NumberCorrector
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.guipy.widgets.widget_builder import restore_splitter_geometry, save_splitter_geometry, style_splitter

# 4. Local modules
from xms.tuflowfv.data.material_data import MaterialData
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui import model_control_consts as const
from xms.tuflowfv.gui.advanced_material_button_delegate import AdvancedMaterialButtonDelegate
from xms.tuflowfv.gui.code_editor_widget import CodeEditorWidget, TuflowFVHighlighter
from xms.tuflowfv.gui.global_bcs_table import GlobalBcsTableWidget
from xms.tuflowfv.gui.gridded_bcs_table import GriddedBcsTableWidget
from xms.tuflowfv.gui.layer_faces_table import LayerFacesTableWidget
from xms.tuflowfv.gui.material_filter_model import MaterialFilterModel
from xms.tuflowfv.gui.model_control_dialog_ui import Ui_ModelControlDialog
from xms.tuflowfv.gui.output_blocks_table import OutputBlocksTableWidget
from xms.tuflowfv.gui.output_coverages_table import OutputCoveragesTable
from xms.tuflowfv.gui.tree_item_selector_table import TreeItemSelectorTableWidget
from xms.tuflowfv.gui.unit_label_switcher import UnitsLabelSwitcher
from xms.tuflowfv.gui.z_modifications_table import ZModificationsTableWidget


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class ModelControlDialog(XmsDlg):
    """The TUFLOWFV simulation model control dialog."""

    def __init__(self, parent, data, projection, tree_node, sim_uuid, time_formats):
        """Initializes the model control dialog.

        Args:
            parent (QWidget): Parent dialog
            data (SimData): Data class containing the Model Control parameters
            projection (Projection): The current display projection. Used for switching unit labels between SI and
                Imperial.
            tree_node (TreeNode): The root project explorer tree
            sim_uuid (str): UUID of the Model Control's simulation, not its component
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        super().__init__(parent, 'xms.tuflowfv.gui.model_control_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:TUFLOW_FV'
        self.data = data
        self.projection = projection
        self.tree_node = tree_node
        self.sim_uuid = sim_uuid
        self.time_format_std = time_formats[0]
        self.time_format_qt = time_formats[1]
        self.ui = Ui_ModelControlDialog()
        self.ui.setupUi(self)

        # Table widgets
        self.layer_faces_table = None
        self.z_modifications_table = None
        self.output_blocks_table = None
        self.output_points_coverages_table = None
        self.set_mat_table = None
        self.material_model = None  # Needs to be named 'material_model' for shared code
        self.set_mat_filter_model = None
        self.global_bcs_table = None
        self.gridded_bcs_table = None
        self.linked_simulations_table = None
        self.code_editor = None

        # Splitter not added in the UI file
        self.output_splitter = None

        # Commonly used edit field validators
        self.number_corrector = NumberCorrector(self)
        self.dbl_validator = QDoubleValidator(self)
        self.dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.dbl_noneg_validator = QDoubleValidator(self)
        self.dbl_noneg_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.dbl_noneg_validator.setBottom(0.0000000001)

        self._imperial = False
        if 'METER' not in projection.vertical_units.upper():
            self._imperial = True

        # Setup the dialog
        self._setup_ui()
        if self._imperial:
            label_switcher = UnitsLabelSwitcher(self)
            label_switcher.switch_to_imperial([QLabel, QCheckBox])

    def _setup_ui(self):
        """Setup dialog widgets."""
        self._add_tables()
        self._add_validators()
        self._add_cbx_options()
        self._hide_future_widgets()
        self._connect_slots()
        self._load_data()
        self._add_splitters()
        self._initialize_state()
        self.ui.mc_tabs.setCurrentIndex(0)

    def _add_splitters(self):
        """Adds a QSplitter between the output list and options pane so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        self.output_splitter = QSplitter(self)
        self.output_splitter.setOrientation(Qt.Horizontal)
        self.output_splitter.addWidget(self.ui.grp_output_blocks)
        self.output_splitter.addWidget(self.ui.grp_output_options)
        style_splitter(self.output_splitter)
        pos = self.ui.horiz_layout_output.indexOf(self.ui.grp_output_options)
        self.ui.horiz_layout_output.insertWidget(pos, self.output_splitter)

        # Style the splitters added in the ui file
        style_splitter(self.ui.splitter_bc)
        style_splitter(self.ui.splitter_bc_vert)

    def _add_tables(self):
        """Add pandas DataFrame table widgets for all tabs."""
        self._add_layer_faces_table()
        self._add_z_modifications_table()
        self._add_output_blocks_table()
        self._add_output_points_coverages_table()
        self._add_set_mat_table()
        self._add_global_bcs_table()
        self._add_gridded_bcs_table()
        self._add_linked_simulations_table()
        self._add_advanced_cards_editor()
        self._populate_linked_coverages()

    def _add_validators(self):
        """Add validators on all edit fields."""
        self._add_general_validators()
        self._add_time_validators()
        self._add_global_parameters_validators()
        self._add_wind_stress_validators()
        self._add_geometry_validators()
        self._add_initial_conditions_validators()
        self._add_output_validators()

    def _add_cbx_options(self):
        """Setup combobox options on all comboboxes."""
        self._add_general_cbx_options()
        self._add_global_parameters_cbx_options()
        self._add_wind_stress_cbx_options()
        self._add_output_cbx_options()
        self._add_geometry_cbx_options()

    def _hide_future_widgets(self):
        """Hide widgets in the GUI that we are not ready to support.

        Notes:
            Not great to have code that we aren't testing. These are features that were originally planned to be
            supported but are not ready for release with early iterations. When adding support for these features, do
            not expect everything to work properly.
        """
        # For now disable the 3D interface. This disables a whole bunch of widgets. Some may not be correctly tied to
        # this dependency.
        self.ui.grp_is_3d.setVisible(False)
        # We don't export anything from these widgets, but they will help us know how many variables we need to export
        # when we do support these modules.
        self.ui.grp_module_selection.setVisible(False)
        # Horizontal scalar diffusivity used for AD module
        self.ui.grp_global_horiz_diffusivity.setVisible(False)
        # The following outputs are only applicable to unsupported modules/3D. Enable as support is added.
        self.ui.tog_output_turb_diff.setVisible(False)  # AD output
        self.ui.tog_output_air_temp.setVisible(False)   # AD output
        self.ui.tog_output_dzb.setVisible(False)        # ST/morphological outpu
        self.ui.tog_output_lw_rad.setVisible(False)
        self.ui.tog_output_precip.setVisible(False)
        self.ui.tog_output_rel_hum.setVisible(False)
        self.ui.tog_output_rhow.setVisible(False)       # 3D/AD output
        self.ui.tog_output_sal.setVisible(False)
        self.ui.tog_output_turbz.setVisible(False)
        self.ui.tog_output_wq_all.setVisible(False)
        self.ui.tog_output_wq_diag_all.setVisible(False)

    """
    Tables
    """
    def _add_layer_faces_table(self):
        """Add the layer faces table to the 'Geometry' tab."""
        df = self.data.geometry.to_dataframe()
        self.layer_faces_table = LayerFacesTableWidget(df, self)
        self.ui.grp_layer_faces.layout().addWidget(self.layer_faces_table)

    def _add_z_modifications_table(self):
        """Add the z modifications table to the 'Geometry' tab."""
        df = self.data.z_modifications.to_dataframe()
        self.z_modifications_table = ZModificationsTableWidget(tree_node=self.tree_node, data_frame=df,
                                                               sim_data=self.data, parent=self)
        layout = QVBoxLayout(self)
        layout.addWidget(self.z_modifications_table)
        self.ui.grp_z_modifications.setLayout(layout)

    def _add_output_blocks_table(self):
        """Add the output blocks table to the 'Output' tab."""
        # Add the table view
        df = self.data.output.to_dataframe()
        self.output_blocks_table = OutputBlocksTableWidget(df=df, parent=self, data=self.data)
        self.ui.grp_output_blocks.layout().addWidget(self.output_blocks_table)
        self.output_blocks_table.table_view.selectionModel().selectionChanged.connect(self.on_output_block_changed)
        self.output_blocks_table.model.dataChanged.connect(self.on_output_data_changed)
        self.ui.scroll_area_output_blocks_options_contents.hide()

    def _add_output_points_coverages_table(self):
        """Add the output points coverage table to the point rows in the 'Output' tab."""
        # TODO: Make this table less squishy. Shouldn't ever get too big, but it is annoying.
        # Add the table view
        df = self.data.output_points_coverages.to_dataframe()
        self.output_points_coverages_table = OutputCoveragesTable(parent=self, data_frame=df, data=self.data,
                                                                  tree_node=self.tree_node,
                                                                  blocks_table=self.output_blocks_table)
        layout = QVBoxLayout(self)
        layout.addWidget(self.output_points_coverages_table)
        self.ui.grp_output_points_cov.setLayout(layout)

    def _add_set_mat_table(self):
        """Add the set mat table to the 'Materials' tab."""
        df = self.data.global_set_mat.to_dataframe()
        self.set_mat_table = QxTableView()
        self.material_model = QxPandasTableModel(df)
        self.set_mat_table.filter_model = MaterialFilterModel(self)
        self.set_mat_table.filter_model.setSourceModel(self.material_model)
        self.set_mat_table.setModel(self.set_mat_table.filter_model)
        self.ui.grp_set_mat.layout().addWidget(self.set_mat_table)

        # Set delegate for columns that are toggles.
        toggle_columns = [
            MaterialData.COL_ACTIVE,
            MaterialData.COL_OVERRIDE_BOTTOM_ROUGHNESS,
        ]
        check_delegate = CheckBoxNoTextDelegate(self)
        check_delegate.state_changed.connect(self.on_toggle_changed)
        self.material_model.set_checkbox_columns(toggle_columns)
        for toggle_column in toggle_columns:
            self.set_mat_table.setItemDelegateForColumn(toggle_column, check_delegate)

        # Set up delegate for the advanced button column.
        advanced_delegate = AdvancedMaterialButtonDelegate(parent=self, display_projection=self.projection,
                                                           set_mat=True)
        self.set_mat_table.setItemDelegateForColumn(MaterialData.COL_ADVANCED, advanced_delegate)

        # Hide the columns unused for set mat.
        self.set_mat_table.setColumnHidden(MaterialData.COL_ID, True)
        self.set_mat_table.setColumnHidden(MaterialData.COL_COLOR, True)
        self.set_mat_table.setColumnHidden(MaterialData.COL_NAME, True)
        self.set_mat_table.setColumnHidden(MaterialData.COL_ACTIVE, True)
        # Hide the columns we want to handle in the advanced dialog.
        for col in range(MaterialData.COL_ADVANCED + 1, self.material_model.columnCount()):
            self.set_mat_table.setColumnHidden(col, True)

        # Setup the headers
        self.set_mat_table.verticalHeader().hide()
        self.set_mat_table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.set_mat_table.horizontalHeader().setStretchLastSection(True)
        self.set_mat_table.resizeColumnsToContents()

        # Hide the unassigned row
        self.set_mat_table.hideRow(0)

    def _add_global_bcs_table(self):
        """Add the global BCs table to the 'Boundary conditions' tab."""
        self.global_bcs_table = GlobalBcsTableWidget(bc_data=self.data.global_bcs, parent=self, grid_id=None,
                                                     time_formats=(self.time_format_std, self.time_format_qt))
        layout = QVBoxLayout(self)
        layout.addWidget(self.global_bcs_table)
        self.ui.grp_global_bcs.setLayout(layout)

    def _add_gridded_bcs_table(self):
        """Add the global BCs table to the 'Boundary conditions' tab."""
        self.gridded_bcs_table = GriddedBcsTableWidget(sim_data=self.data, parent=self,
                                                       time_formats=(self.time_format_std, self.time_format_qt))
        layout = QVBoxLayout(self)
        layout.addWidget(self.gridded_bcs_table)
        self.ui.grp_gridded_bcs.setLayout(layout)

    def _add_linked_simulations_table(self):
        """Add the linked simulations table to the 'Simulation links' tab."""
        df = self.data.linked_simulations.to_dataframe()
        self.linked_simulations_table = TreeItemSelectorTableWidget(tree_node=self.tree_node, tree_type='TI_DYN_SIM',
                                                                    data_frame=df, max_rows=None, parent=self,
                                                                    sim_uuid=self.sim_uuid)
        layout = QVBoxLayout(self)
        layout.addWidget(self.linked_simulations_table)
        self.ui.grp_simulation_links.setLayout(layout)

    def _add_advanced_cards_editor(self):
        """Add the advanced cards editor."""
        self.code_editor = CodeEditorWidget(TuflowFVHighlighter)
        self.ui.scollable_area_advanced_contents.layout().addWidget(self.code_editor)
        self.code_editor.setPlainText(self.data.general.attrs['advanced_cards'])

    def _populate_linked_coverages(self):
        """Fill the list widgets for linked BC and materials coverages."""
        sim_item = tree_util.find_tree_node_by_uuid(self.tree_node, self.sim_uuid)
        if not sim_item:
            return
        # Find paths to all the linked BC coverages
        bc_items = tree_util.descendants_of_type(tree_root=sim_item, xms_types=['TI_COVER_PTR'], allow_pointers=True,
                                                 only_first=False, coverage_type='Boundary Conditions',
                                                 model_name='TUFLOWFV')
        for bc_item in bc_items:
            list_item = QListWidgetItem()
            gui_util.set_widget_text_to_tree_path(self.tree_node, bc_item.uuid, list_item)
            self.ui.lst_linked_bcs.addItem(list_item)
        # Find paths to all the materials coverages
        mat_items = tree_util.descendants_of_type(tree_root=sim_item, xms_types=['TI_COVER_PTR'], allow_pointers=True,
                                                  only_first=False, coverage_type='Materials', model_name='TUFLOWFV')
        for mat_item in mat_items:
            list_item = QListWidgetItem()
            gui_util.set_widget_text_to_tree_path(self.tree_node, mat_item.uuid, list_item)
            self.ui.lst_linked_materials.addItem(list_item)

    """
    Edit field validators
    """
    def _add_general_validators(self):
        """Add validators to edit fields in the 'General' tab."""
        nonneg_int = QIntValidator(self)
        nonneg_int.setBottom(0)
        self.ui.edt_output_display_interval.setValidator(self.dbl_noneg_validator)
        self.ui.edt_output_display_interval.installEventFilter(self.number_corrector)
        self.ui.edt_device_id.setValidator(nonneg_int)
        self.ui.edt_device_id.installEventFilter(self.number_corrector)

    def _add_time_validators(self):
        """Add double validators to edit field widgets for the 'Time' tab."""
        self.ui.edt_ref_date.setValidator(self.dbl_validator)
        self.ui.edt_ref_date.installEventFilter(self.number_corrector)
        self.ui.edt_start_time.setValidator(self.dbl_validator)
        self.ui.edt_start_time.installEventFilter(self.number_corrector)
        self.ui.edt_end_time.setValidator(self.dbl_validator)
        self.ui.edt_end_time.installEventFilter(self.number_corrector)
        self.ui.edt_cfl.setValidator(self.dbl_noneg_validator)
        self.ui.edt_cfl.installEventFilter(self.number_corrector)
        self.ui.edt_timestep_min.setValidator(self.dbl_noneg_validator)
        self.ui.edt_timestep_min.installEventFilter(self.number_corrector)
        self.ui.edt_timestep_max.setValidator(self.dbl_noneg_validator)
        self.ui.edt_timestep_max.installEventFilter(self.number_corrector)

    def _add_global_parameters_validators(self):
        """Add double validators to edit field widgets for the 'Global parameters' tab."""
        self.ui.edt_global_roughness.setValidator(self.dbl_noneg_validator)
        self.ui.edt_global_roughness.installEventFilter(self.number_corrector)
        self.ui.edt_global_viscosity_horiz.setValidator(self.dbl_validator)
        self.ui.edt_global_viscosity_horiz.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_viscosity_min.setValidator(self.dbl_validator)
        self.ui.edt_horiz_viscosity_min.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_viscosity_max.setValidator(self.dbl_validator)
        self.ui.edt_horiz_viscosity_max.installEventFilter(self.number_corrector)
        self.ui.edt_global_viscosity_vert.setValidator(self.dbl_validator)
        self.ui.edt_global_viscosity_vert.installEventFilter(self.number_corrector)
        self.ui.edt_parametric_coefficients1.setValidator(self.dbl_validator)
        self.ui.edt_parametric_coefficients1.installEventFilter(self.number_corrector)
        self.ui.edt_parametric_coefficients2.setValidator(self.dbl_validator)
        self.ui.edt_parametric_coefficients2.installEventFilter(self.number_corrector)
        self.ui.edt_vertical_viscosity_limits_min.setValidator(self.dbl_validator)
        self.ui.edt_vertical_viscosity_limits_min.installEventFilter(self.number_corrector)
        self.ui.edt_vertical_viscosity_limits_max.setValidator(self.dbl_validator)
        self.ui.edt_vertical_viscosity_limits_max.installEventFilter(self.number_corrector)
        self.ui.edt_turbulence_update.setValidator(self.dbl_noneg_validator)
        self.ui.edt_turbulence_update.installEventFilter(self.number_corrector)
        self.ui.edt_global_horiz_diffusivity.setValidator(self.dbl_validator)
        self.ui.edt_global_horiz_diffusivity.installEventFilter(self.number_corrector)
        self.ui.edt_global_horiz_diffusivity_coef1.setValidator(self.dbl_validator)
        self.ui.edt_global_horiz_diffusivity_coef1.installEventFilter(self.number_corrector)
        self.ui.edt_global_horiz_diffusivity_coef2.setValidator(self.dbl_validator)
        self.ui.edt_global_horiz_diffusivity_coef2.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_diffusivity_min.setValidator(self.dbl_validator)
        self.ui.edt_horiz_diffusivity_min.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_diffusivity_max.setValidator(self.dbl_validator)
        self.ui.edt_horiz_diffusivity_max.installEventFilter(self.number_corrector)

        self.ui.edt_global_vert_diffusivity.setValidator(self.dbl_validator)
        self.ui.edt_global_vert_diffusivity.installEventFilter(self.number_corrector)
        self.ui.edt_global_vert_diffusivity_coef1.setValidator(self.dbl_validator)
        self.ui.edt_global_vert_diffusivity_coef1.installEventFilter(self.number_corrector)
        self.ui.edt_global_vert_diffusivity_coef2.setValidator(self.dbl_validator)
        self.ui.edt_global_vert_diffusivity_coef2.installEventFilter(self.number_corrector)
        self.ui.edt_vert_diffusivity_min.setValidator(self.dbl_validator)
        self.ui.edt_vert_diffusivity_min.installEventFilter(self.number_corrector)
        self.ui.edt_vert_diffusivity_max.setValidator(self.dbl_validator)
        self.ui.edt_vert_diffusivity_max.installEventFilter(self.number_corrector)

        self.ui.edt_stability_wse.setValidator(self.dbl_validator)
        self.ui.edt_stability_wse.installEventFilter(self.number_corrector)
        self.ui.edt_stability_velocity.setValidator(self.dbl_validator)
        self.ui.edt_stability_velocity.installEventFilter(self.number_corrector)

        self.ui.edt_bc_update_dt.setValidator(self.dbl_validator)
        self.ui.edt_bc_update_dt.installEventFilter(self.number_corrector)

    def _add_wind_stress_validators(self):
        """Add double validators to edit field widgets for the 'Wind stress' tab."""
        self.ui.edt_wind_stress_wa.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_wa.installEventFilter(self.number_corrector)
        self.ui.edt_wind_stress_ca.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_ca.installEventFilter(self.number_corrector)
        self.ui.edt_wind_stress_wb.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_wb.installEventFilter(self.number_corrector)
        self.ui.edt_wind_stress_cb.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_cb.installEventFilter(self.number_corrector)
        self.ui.edt_wind_stress_bulk.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_bulk.installEventFilter(self.number_corrector)
        self.ui.edt_wind_stress_scale.setValidator(self.dbl_validator)
        self.ui.edt_wind_stress_scale.installEventFilter(self.number_corrector)

    def _add_geometry_validators(self):
        """Add double validators to edit field widgets for the 'Geometry' tab."""
        zero_to_one = QDoubleValidator(self)
        zero_to_one.setRange(0.0, 1.0, NumberCorrector.DEFAULT_PRECISION)
        self.ui.edt_bed_elevation_min.setValidator(self.dbl_validator)
        self.ui.edt_bed_elevation_min.installEventFilter(self.number_corrector)
        self.ui.edt_bed_elevation_max.setValidator(self.dbl_validator)
        self.ui.edt_bed_elevation_max.installEventFilter(self.number_corrector)
        self.ui.edt_sigma_layers.setValidator(zero_to_one)
        self.ui.edt_sigma_layers.installEventFilter(self.number_corrector)
        self.ui.edt_cell_3d_depth.setValidator(self.dbl_validator)
        self.ui.edt_cell_3d_depth.installEventFilter(self.number_corrector)
        self.ui.edt_min_bottom_thickness.setValidator(self.dbl_validator)
        self.ui.edt_min_bottom_thickness.installEventFilter(self.number_corrector)
        self.ui.edt_cell_dry_depth.setValidator(self.dbl_validator)
        self.ui.edt_cell_dry_depth.installEventFilter(self.number_corrector)
        self.ui.edt_cell_wet_depth.setValidator(self.dbl_validator)
        self.ui.edt_cell_wet_depth.installEventFilter(self.number_corrector)

    def _add_initial_conditions_validators(self):
        """Add double validators to edit field widgets for the 'Initial conditions' tab."""
        self.ui.edt_initial_water_level.setValidator(self.dbl_validator)
        self.ui.edt_initial_water_level.installEventFilter(self.number_corrector)

    def _add_output_validators(self):
        """Add double validators to edit field widgets for the 'Output' tab."""
        self.ui.edt_output_start_time.setValidator(self.dbl_noneg_validator)
        self.ui.edt_output_start_time.installEventFilter(self.number_corrector)
        self.ui.edt_output_final_time.setValidator(self.dbl_noneg_validator)
        self.ui.edt_output_final_time.installEventFilter(self.number_corrector)
        self.ui.edt_output_interval.setValidator(self.dbl_noneg_validator)
        self.ui.edt_output_interval.installEventFilter(self.number_corrector)
        self.ui.edt_statistics_dt.setValidator(self.dbl_noneg_validator)
        self.ui.edt_statistics_dt.installEventFilter(self.number_corrector)
        self.ui.edt_ouput_restart_file_dt.setValidator(self.dbl_noneg_validator)
        self.ui.edt_ouput_restart_file_dt.installEventFilter(self.number_corrector)

    """
    Combobox options
    """
    def _add_general_cbx_options(self):
        """Set up the combobox options for the 'General' tab."""
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_spatial_order_horiz)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_spatial_order_vert)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_hardware_solver)

    def _add_global_parameters_cbx_options(self):
        """Set up the combobox options for the 'Global parameters' tab."""
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_global_roughness)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_mixing_model_horiz)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_mixing_model_vert)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_global_horiz_diffusivity)

    def _add_wind_stress_cbx_options(self):
        """Set up the combobox options for the 'Wind stress' tab."""
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_wind_stress)

    def _add_output_cbx_options(self):
        """Set up the combobox options for the 'Output' tab."""
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_statistics)

    def _add_geometry_cbx_options(self):
        """Set up the combobox options for the 'Geometry' tab."""
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_vertical_mesh_type)
        gui_util.add_combobox_options(const.MC_CBX_OPTS, self.ui.cbx_layer_faces)

    """
    Slot connections
    """
    def _connect_slots(self):
        """Connect signals and slots for all widgets."""
        self._connect_general_slots()
        self._connect_time_slots()
        self._connect_global_parameters_slots()
        self._connect_wind_stress_slots()
        self._connect_geometry_slots()
        self._connect_output_slots()
        self._connect_bc_slots()
        self._connect_initial_conditions_slots()

    def _connect_general_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'General parameters' tab."""
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.ui.tog_use_salinity.stateChanged.connect(self.on_tog_use_salinity)
        self.ui.tog_use_temp.stateChanged.connect(self.on_tog_use_temperature)
        self.ui.tog_use_sediment.stateChanged.connect(self.on_tog_use_sediment)
        self.ui.tog_is_3d.stateChanged.connect(self.on_tog_is_3d)
        self.ui.cbx_hardware_solver.currentIndexChanged.connect(self.on_hardware_solver_changed)
        self.ui.cbx_hardware_solver.currentIndexChanged.connect(self.ui.grp_hardware_solver.toggle_group)

    def _connect_time_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Time' tab."""
        self.ui.tog_use_isodate.stateChanged.connect(self.on_tog_use_isodate)

    def _connect_global_parameters_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Global parameters' tab."""
        self.ui.cbx_global_roughness.currentIndexChanged.connect(self.on_global_roughness_changed)
        self.ui.cbx_mixing_model_horiz.currentIndexChanged.connect(self.on_horiz_mixing_model_changed)
        self.ui.cbx_mixing_model_vert.currentIndexChanged.connect(self.on_vert_mixing_model_changed)
        self.ui.btn_external_turbulence.clicked.connect(self.on_btn_external_turbulence)
        self.ui.tog_turbulence_update.stateChanged.connect(self.on_tog_turbulence_update)
        self.ui.cbx_global_horiz_diffusivity.currentIndexChanged.connect(self.on_horiz_diffusivity_model_changed)

    def _connect_wind_stress_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Wind stress' tab."""
        self.ui.cbx_wind_stress.currentIndexChanged.connect(self.on_wind_stress_changed)
        self.ui.cbx_wind_stress.currentIndexChanged.connect(self.ui.grp_define_wind_stress_parameters.toggle_group)

    def _connect_geometry_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Geometry' tab."""
        self.ui.cbx_vertical_mesh_type.currentIndexChanged.connect(self.on_vertical_mesh_type_changed)
        self.ui.cbx_layer_faces.currentIndexChanged.connect(self.on_layer_faces_changed)
        self.ui.btn_layer_faces_csv.clicked.connect(self.on_btn_layer_faces_csv)
        self.ui.tog_initial_wse.stateChanged.connect(self.on_tog_initial_wse)

    def _connect_output_slots(self):
        """Connect slots in the output tab."""
        self.ui.tog_define_statistics_dt.stateChanged.connect(self.on_tog_statistics_dt)

    def _connect_bc_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Boundary conditions' tab."""
        self.ui.btn_transport_file.clicked.connect(self.on_btn_transport_file)

    def _connect_initial_conditions_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Initial conditions' tab."""
        self.ui.btn_restart_file.clicked.connect(self.on_btn_restart_file)
        self.ui.grp_restart.toggled.connect(self.on_tog_use_restart_time)
        self.ui.tog_use_restart_time.stateChanged.connect(self.on_tog_use_restart_time)

    def _initialize_state(self):
        """Initialize the state of all the widgets after loading data."""
        self.on_tog_use_salinity(self.ui.tog_use_salinity.checkState())
        self.on_tog_use_temperature(self.ui.tog_use_temp.checkState())
        self.on_tog_use_sediment(self.ui.tog_use_sediment.checkState())
        self.on_tog_use_isodate(self.ui.tog_use_isodate.checkState())
        self.on_hardware_solver_changed(self.ui.cbx_hardware_solver.currentIndex())
        self.on_global_roughness_changed(self.ui.cbx_global_roughness.currentIndex())
        self.on_horiz_mixing_model_changed(self.ui.cbx_mixing_model_horiz.currentIndex())
        self.on_vert_mixing_model_changed(self.ui.cbx_mixing_model_vert.currentIndex())
        self.on_horiz_diffusivity_model_changed(self.ui.cbx_global_horiz_diffusivity.currentIndex())
        self.on_tog_turbulence_update(self.ui.tog_turbulence_update.checkState())
        self.on_wind_stress_changed(self.ui.cbx_wind_stress.currentIndex())
        self.on_tog_statistics_dt(self.ui.tog_define_statistics_dt.checkState())
        self.on_vertical_mesh_type_changed(self.ui.cbx_vertical_mesh_type.currentIndex())
        self.on_layer_faces_changed(self.ui.cbx_layer_faces.currentIndex())
        self.on_tog_is_3d(self.ui.tog_is_3d.checkState())
        # Do this one last
        self.on_tog_use_restart_time(None)

    """
    File I/O
    """
    def _load_data(self):
        """Populate widget values from persistent data."""
        self._load_general()
        self._load_time()
        self._load_global_parameters()
        self._load_wind_stress()
        self._load_geometry()
        self._load_initial_conditions()
        self._load_output()
        self._load_materials()

    def _load_general(self):
        """Populate widgets in the 'General' tab from persistent data."""
        general_attrs = self.data.general.attrs
        self.ui.tog_use_salinity.setCheckState(Qt.Checked if int(general_attrs['use_salinity']) == 1 else Qt.Unchecked)
        self.ui.tog_couple_salinity.setCheckState(
            Qt.Checked if int(general_attrs['couple_salinity']) == 1 else Qt.Unchecked
        )
        self.ui.tog_use_temp.setCheckState(
            Qt.Checked if int(general_attrs['use_temperature']) == 1 else Qt.Unchecked
        )
        self.ui.tog_couple_temp.setCheckState(
            Qt.Checked if int(general_attrs['couple_temperature']) == 1 else Qt.Unchecked
        )
        self.ui.tog_use_sediment.setCheckState(Qt.Checked if int(general_attrs['use_sediment']) == 1 else Qt.Unchecked)
        self.ui.tog_couple_sediment.setCheckState(
            Qt.Checked if int(general_attrs['couple_sediment']) == 1 else Qt.Unchecked
        )
        self.ui.tog_use_heat.setCheckState(Qt.Checked if int(general_attrs['use_heat']) == 1 else Qt.Unchecked)

        self.ui.tog_is_3d.setCheckState(Qt.Checked if int(general_attrs['is_3d']) == 1 else Qt.Unchecked)

        gui_util.set_combobox_from_data(self.ui.cbx_spatial_order_horiz, int(general_attrs['horizontal_order']))
        gui_util.set_combobox_from_data(self.ui.cbx_spatial_order_vert, int(general_attrs['vertical_order']))
        self.ui.grp_spatial_order.setChecked(int(general_attrs['define_spatial_order']))

        self.ui.grp_hardware_solver.setChecked(int(general_attrs['define_hardware_solver']))
        gui_util.set_combobox_from_data(self.ui.cbx_hardware_solver, general_attrs['hardware_solver'])
        self.ui.edt_device_id.setText(str(int(general_attrs['device_id'])))

        self.ui.grp_display_interval.setChecked(int(general_attrs['define_display_interval']))
        self.ui.edt_output_display_interval.setText(str(general_attrs['display_interval']))

        self.ui.tog_projection_warning.setCheckState(
            Qt.Checked if int(general_attrs['projection_warning']) == 1 else Qt.Unchecked
        )
        self.ui.tog_tutorial.setCheckState(Qt.Checked if str(general_attrs['tutorial']) == 'ON' else Qt.Unchecked)

    def _load_time(self):
        """Populate widgets in the 'Time' tab."""
        time_attrs = self.data.time.attrs
        # Convert Qt datetime strings stored in data
        gui_util.intialize_datetime_widget(time_attrs['ref_date'], self.ui.date_reference, self.time_format_qt)
        gui_util.intialize_datetime_widget(time_attrs['start_date'], self.ui.date_start_time, self.time_format_qt)
        gui_util.intialize_datetime_widget(time_attrs['end_date'], self.ui.date_end_time, self.time_format_qt)
        self.ui.edt_ref_date.setText(str(time_attrs['ref_date_hours']))
        self.ui.edt_start_time.setText(str(time_attrs['start_hours']))
        self.ui.edt_end_time.setText(str(time_attrs['end_hours']))
        self.ui.edt_cfl.setText(str(time_attrs['cfl']))
        self.ui.edt_timestep_min.setText(str(time_attrs['min_increment']))
        self.ui.edt_timestep_max.setText(str(time_attrs['max_increment']))
        self.ui.tog_use_isodate.setCheckState(Qt.Checked if int(time_attrs['use_isodate']) == 1 else Qt.Unchecked)

    def _load_global_parameters(self):
        """Populate widgets in the 'Global parameters' tab from persistent data."""
        model_attrs = self.data.globals.attrs
        self._load_horizontal_mixing(model_attrs)
        self._load_vertical_mixing(model_attrs)
        self._load_horizontal_diffusivity(model_attrs)
        self._load_vertical_diffusivity(model_attrs)
        self._load_stability_limits(model_attrs)
        self._load_global_bc(model_attrs)

    def _load_horizontal_mixing(self, model_attrs):
        """Load horizontal mixing widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        gui_util.set_combobox_from_data(self.ui.cbx_mixing_model_horiz, model_attrs['horizontal_mixing'])
        self.ui.edt_global_viscosity_horiz.setText(str(model_attrs['global_horizontal_viscosity']))
        self.ui.grp_horiz_viscosity_limits.setChecked(bool(int(model_attrs['define_horizontal_viscosity_limits'])))
        self.ui.edt_horiz_viscosity_min.setText(str(model_attrs['horizontal_viscosity_min']))
        self.ui.edt_horiz_viscosity_max.setText(str(model_attrs['horizontal_viscosity_max']))

    def _load_vertical_mixing(self, model_attrs):
        """Load vertical mixing widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        gui_util.set_combobox_from_data(self.ui.cbx_mixing_model_vert, model_attrs['vertical_mixing'])
        self.ui.edt_global_viscosity_vert.setText(str(model_attrs['global_vertical_viscosity']))
        self.ui.edt_parametric_coefficients1.setText(str(model_attrs['global_vertical_parametric_coefficients1']))
        self.ui.edt_parametric_coefficients2.setText(str(model_attrs['global_vertical_parametric_coefficients2']))
        turbulence_folder = model_attrs['external_vertical_viscosity']
        if turbulence_folder and self.data.does_file_exist(turbulence_folder):  # This is actually a directory
            self.ui.lbl_external_turbulence_selection.setText(turbulence_folder)
        self.ui.edt_turbulence_update.setText(str(model_attrs['turbulence_update']))
        self.ui.tog_turbulence_update.setChecked(bool(int(model_attrs['define_turbulence_update'])))
        self.ui.grp_vertical_viscosity_limits.setChecked(bool(int(model_attrs['define_vertical_viscosity_limits'])))
        self.ui.edt_vertical_viscosity_limits_min.setText(str(model_attrs['vertical_viscosity_min']))
        self.ui.edt_vertical_viscosity_limits_max.setText(str(model_attrs['vertical_viscosity_max']))
        self.ui.grp_mixing_model_vert.setChecked(bool(int(model_attrs['define_vertical_mixing'])))

    def _load_horizontal_diffusivity(self, model_attrs):
        """Load horizontal diffusivity widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        gui_util. set_combobox_from_data(self.ui.cbx_global_horiz_diffusivity,
                                         model_attrs['horizontal_scalar_diffusivity_type'])
        self.ui.edt_global_horiz_diffusivity.setText(str(model_attrs['horizontal_scalar_diffusivity']))
        self.ui.edt_global_horiz_diffusivity_coef1.setText(str(model_attrs['horizontal_scalar_diffusivity_coef1']))
        self.ui.edt_global_horiz_diffusivity_coef2.setText(str(model_attrs['horizontal_scalar_diffusivity_coef2']))
        self.ui.grp_global_horiz_diffusivity_limits.setChecked(
            bool(int(model_attrs['define_horizontal_diffusivity_limits']))
        )
        self.ui.edt_horiz_diffusivity_min.setText(str(model_attrs['horizontal_scalar_diffusivity_min']))
        self.ui.edt_horiz_diffusivity_max.setText(str(model_attrs['horizontal_scalar_diffusivity_max']))

    def _load_vertical_diffusivity(self, model_attrs):
        """Load vertical diffusivity widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        self.ui.grp_global_vert_diffusivity.setChecked(bool(int(model_attrs['define_vertical_scalar_diffusivity'])))
        self.ui.edt_global_vert_diffusivity.setText(str(model_attrs['vertical_scalar_diffusivity']))
        self.ui.edt_global_vert_diffusivity_coef1.setText(str(model_attrs['vertical_scalar_diffusivity_coef1']))
        self.ui.edt_global_vert_diffusivity_coef2.setText(str(model_attrs['vertical_scalar_diffusivity_coef2']))
        self.ui.grp_global_vert_diffusivity_limits.setChecked(
            bool(int(model_attrs['define_vertical_diffusivity_limits']))
        )
        self.ui.edt_vert_diffusivity_min.setText(str(model_attrs['vertical_scalar_diffusivity_min']))
        self.ui.edt_vert_diffusivity_max.setText(str(model_attrs['vertical_scalar_diffusivity_max']))

    def _load_stability_limits(self, model_attrs):
        """Load stability limits widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        self.ui.grp_stability_limits.setChecked(bool(int(model_attrs['define_stability_limits'])))
        self.ui.edt_stability_wse.setText(str(model_attrs['stability_wse']))
        self.ui.edt_stability_velocity.setText(str(model_attrs['stability_velocity']))

    def _load_global_bc(self, model_attrs):
        """Load BC default update dt widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        self.ui.grp_bc_update_dt.setChecked(bool(int(model_attrs['define_bc_default_update_dt'])))
        self.ui.edt_bc_update_dt.setText(str(model_attrs['bc_default_update_dt']))
        self.ui.grp_transport_mode.setChecked(bool(int(model_attrs['transport_mode'])))
        self.ui.tog_include_wave_stress.setChecked(bool(int(model_attrs['include_wave_stress'])))
        self.ui.tog_include_stokes_drift.setChecked(bool(int(model_attrs['include_stokes_drift'])))
        transport_file = model_attrs['transport_file']
        if transport_file and self.data.does_file_exist(transport_file):
            self.ui.lbl_transport_file_selection.setText(transport_file)

    def _load_wind_stress(self):
        """Populate widgets in the 'Wind stress' tab from persistent data."""
        attrs = self.data.wind_stress.attrs
        gui_util.set_combobox_from_data(self.ui.cbx_wind_stress, int(attrs['method']))
        self.ui.grp_define_wind_stress.setChecked(bool(int(attrs['define_wind'])))
        self.ui.grp_define_wind_stress_parameters.setChecked(bool(int(attrs['define_parameters'])))
        self.ui.edt_wind_stress_wa.setText(str(attrs['wa']))
        self.ui.edt_wind_stress_ca.setText(str(attrs['ca']))
        self.ui.edt_wind_stress_wb.setText(str(attrs['wb']))
        self.ui.edt_wind_stress_cb.setText(str(attrs['cb']))
        self.ui.edt_wind_stress_bulk.setText(str(attrs['bulk_coefficient']))
        self.ui.edt_wind_stress_scale.setText(str(attrs['scale_factor']))

    def _load_geometry(self):
        """Populate widgets in the 'Geometry' tab from persistent data."""
        geom_attrs = self.data.geometry.attrs
        self.ui.grp_bed_elevation_limits.setChecked(bool(int(geom_attrs['define_bed_elevation_limits'])))
        self.ui.edt_bed_elevation_min.setText(str(geom_attrs['min_bed_elevation']))
        self.ui.edt_bed_elevation_max.setText(str(geom_attrs['max_bed_elevation']))
        gui_util.set_combobox_from_data(self.ui.cbx_vertical_mesh_type, geom_attrs['vertical_mesh_type'])
        gui_util.set_combobox_from_data(self.ui.cbx_layer_faces, geom_attrs['layer_faces'])
        csv_file = geom_attrs['layer_faces_file']
        if csv_file and self.data.does_file_exist(csv_file):
            self.ui.lbl_layer_faces_csv_selection.setText(csv_file)
        self.ui.edt_sigma_layers.setText(str(geom_attrs['sigma_layers']))
        self.ui.edt_cell_3d_depth.setText(str(geom_attrs['cell_3d_depth']))
        self.ui.edt_min_bottom_thickness.setText(str(geom_attrs['min_bed_layer_thickness']))
        self.ui.grp_cell_wet_dry_depths.setChecked(bool(int(geom_attrs['define_cell_depths'])))
        self.ui.edt_cell_dry_depth.setText(str(geom_attrs['dry_cell_depth']))
        self.ui.edt_cell_wet_depth.setText(str(geom_attrs['wet_cell_depth']))

    def _load_initial_conditions(self):
        """Populate widgets in the 'Initial Conditions' tab from persistent data."""
        initial_attrs = self.data.initial_conditions.attrs
        self.ui.tog_initial_wse.setChecked(bool(int(initial_attrs['define_initial_water_level'])))
        self.ui.edt_initial_water_level.setText(str(initial_attrs['initial_water_level']))
        self.ui.grp_restart.setChecked(bool(int(initial_attrs['use_restart_file'])))
        self.ui.tog_use_restart_time.setChecked(bool(int(initial_attrs['use_restart_file_time'])))
        restart_file = initial_attrs['restart_file']
        if restart_file and self.data.does_file_exist(restart_file):
            self.ui.lbl_restart_file_selection.setText(restart_file)

    def _load_output(self):
        """Populate widgets in the 'Output' tab."""
        output_attrs = self.data.output.attrs
        self.ui.edt_log_dir.setText(output_attrs['log_dir'])
        self.ui.edt_output_dir.setText(output_attrs['output_dir'])
        self.ui.grp_write_check_files.setChecked(bool(int(output_attrs['write_check_files'])))
        self.ui.edt_write_check_files.setText(output_attrs['check_files_dir'])
        self.ui.grp_write_empty_gis_files.setChecked(bool(int(output_attrs['write_empty_gis_files'])))
        self.ui.edt_write_empty_gis_files.setText(output_attrs['empty_gis_files_dir'])
        self.ui.grp_output_restart_file.setChecked(bool(int(output_attrs['write_restart_file'])))
        self.ui.edt_ouput_restart_file_dt.setText(str(output_attrs['restart_file_interval']))
        self.ui.tog_overwrite_output_restart_file.setChecked(bool(int(output_attrs['overwrite_restart_file'])))

    def _load_materials(self):
        """Populate widgets in the 'Materials' tab."""
        attrs = self.data.global_set_mat.attrs
        self.ui.grp_set_mat.setChecked(bool(int(attrs['define_set_mat'])))
        gui_util.set_combobox_from_data(self.ui.cbx_global_roughness, attrs['global_roughness'])
        self.ui.edt_global_roughness.setText(str(attrs['global_roughness_coefficient']))
        self.ui.grp_global_roughness.setChecked(bool(int(attrs['define_global_roughness'])))

    def _save_data(self):
        """Saves the dialog data to the SimData.

        Does not flush data to disk, just updates in memory xarray.Datasets. Call flush() on the
        SimData to write to disk.
        """
        self._save_general()
        self._save_time()
        self._save_global_parameters()
        self._save_wind_stress()
        self._save_geometry_data()
        self._save_initial_conditions()
        self._save_output()
        self._save_materials()
        self._save_simulation_links()
        self._save_advanced_cards()

    def _save_general(self):
        """Save data from the 'General' tab to in-memory Datasets."""
        general_attrs = self.data.general.attrs
        general_attrs['use_salinity'] = 1 if self.ui.tog_use_salinity.isChecked() else 0
        general_attrs['couple_salinity'] = 1 if self.ui.tog_couple_salinity.isChecked() else 0
        general_attrs['use_temperature'] = 1 if self.ui.tog_use_temp.isChecked() else 0
        general_attrs['couple_temperature'] = 1 if self.ui.tog_couple_temp.isChecked() else 0
        general_attrs['use_sediment'] = 1 if self.ui.tog_use_sediment.isChecked() else 0
        general_attrs['couple_sediment'] = 1 if self.ui.tog_couple_sediment.isChecked() else 0
        general_attrs['use_heat'] = 1 if self.ui.tog_use_heat.isChecked() else 0
        general_attrs['is_3d'] = 1 if self.ui.tog_is_3d.isChecked() else 0
        general_attrs['define_spatial_order'] = 1 if self.ui.grp_spatial_order.isChecked() else 0
        general_attrs['horizontal_order'] = self.ui.cbx_spatial_order_horiz.itemData(
            self.ui.cbx_spatial_order_horiz.currentIndex()
        )
        general_attrs['vertical_order'] = self.ui.cbx_spatial_order_vert.itemData(
            self.ui.cbx_spatial_order_vert.currentIndex()
        )
        general_attrs['define_hardware_solver'] = 1 if self.ui.grp_hardware_solver.isChecked() else 0
        general_attrs['hardware_solver'] = self.ui.cbx_hardware_solver.itemData(
            self.ui.cbx_hardware_solver.currentIndex()
        )
        general_attrs['device_id'] = int(self.ui.edt_device_id.text())
        general_attrs['define_display_interval'] = 1 if self.ui.grp_display_interval.isChecked() else 0
        general_attrs['display_interval'] = float(self.ui.edt_output_display_interval.text())
        general_attrs['projection_warning'] = 1 if self.ui.tog_projection_warning.checkState() == Qt.Checked else 0
        general_attrs['tutorial'] = 'ON' if self.ui.tog_tutorial.checkState() == Qt.Checked else 'OFF'

    def _save_time(self):
        """Save data from the 'Time' tab to in-memory Datasets."""
        time_attrs = self.data.time.attrs
        time_attrs['ref_date'] = self.ui.date_reference.dateTime().toPython().strftime(ISO_DATETIME_FORMAT)
        time_attrs['start_date'] = self.ui.date_start_time.dateTime().toPython().strftime(ISO_DATETIME_FORMAT)
        time_attrs['end_date'] = self.ui.date_end_time.dateTime().toPython().strftime(ISO_DATETIME_FORMAT)
        time_attrs['ref_date_hours'] = float(self.ui.edt_ref_date.text())
        time_attrs['start_hours'] = float(self.ui.edt_start_time.text())
        time_attrs['end_hours'] = float(self.ui.edt_end_time.text())
        time_attrs['cfl'] = float(self.ui.edt_cfl.text())
        time_attrs['min_increment'] = float(self.ui.edt_timestep_min.text())
        time_attrs['max_increment'] = float(self.ui.edt_timestep_max.text())
        time_attrs['use_isodate'] = 1 if self.ui.tog_use_isodate.checkState() == Qt.Checked else 0

    def _save_global_parameters(self):
        """Save data from the 'Global parameters' tab to in-memory Datasets."""
        model_attrs = self.data.globals.attrs
        self._save_horizontal_mixing(model_attrs)
        self._save_vertical_mixing(model_attrs)
        self._save_horizontal_diffusivity(model_attrs)
        self._save_vertical_diffusivity(model_attrs)
        self._save_stability_limits(model_attrs)
        self._save_global_bc(model_attrs)

    def _save_horizontal_mixing(self, model_attrs):
        """Save horizontal mixing widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['horizontal_mixing'] = self.ui.cbx_mixing_model_horiz.itemData(
            self.ui.cbx_mixing_model_horiz.currentIndex()
        )
        model_attrs['global_horizontal_viscosity'] = float(self.ui.edt_global_viscosity_horiz.text())
        model_attrs['define_horizontal_viscosity_limits'] = int(self.ui.grp_horiz_viscosity_limits.isChecked())
        model_attrs['horizontal_viscosity_min'] = float(self.ui.edt_horiz_viscosity_min.text())
        model_attrs['horizontal_viscosity_max'] = float(self.ui.edt_horiz_viscosity_max.text())

    def _save_vertical_mixing(self, model_attrs):
        """Save vertical mixing widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['vertical_mixing'] = self.ui.cbx_mixing_model_vert.itemData(
            self.ui.cbx_mixing_model_vert.currentIndex()
        )
        model_attrs['global_vertical_viscosity'] = float(self.ui.edt_global_viscosity_vert.text())
        model_attrs['global_vertical_parametric_coefficients1'] = float(self.ui.edt_parametric_coefficients1.text())
        model_attrs['global_vertical_parametric_coefficients2'] = float(self.ui.edt_parametric_coefficients2.text())
        turbulence_folder = self.ui.lbl_external_turbulence_selection.text()
        if turbulence_folder != gui_util.NULL_SELECTION:
            model_attrs['external_vertical_viscosity'] = turbulence_folder
        model_attrs['define_turbulence_update'] = int(self.ui.tog_turbulence_update.isChecked())
        model_attrs['turbulence_update'] = float(self.ui.edt_turbulence_update.text())
        model_attrs['define_vertical_viscosity_limits'] = int(self.ui.grp_vertical_viscosity_limits.isChecked())
        model_attrs['vertical_viscosity_min'] = float(self.ui.edt_vertical_viscosity_limits_min.text())
        model_attrs['vertical_viscosity_max'] = float(self.ui.edt_vertical_viscosity_limits_max.text())
        model_attrs['define_vertical_mixing'] = int(self.ui.grp_mixing_model_vert.isChecked())

    def _save_horizontal_diffusivity(self, model_attrs):
        """Save horizontal diffusivity widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['horizontal_scalar_diffusivity_type'] = self.ui.cbx_global_horiz_diffusivity.itemData(
            self.ui.cbx_global_horiz_diffusivity.currentIndex()
        )
        model_attrs['horizontal_scalar_diffusivity'] = float(self.ui.edt_global_horiz_diffusivity.text())
        model_attrs['horizontal_scalar_diffusivity_coef1'] = float(self.ui.edt_global_horiz_diffusivity_coef1.text())
        model_attrs['horizontal_scalar_diffusivity_coef2'] = float(self.ui.edt_global_horiz_diffusivity_coef2.text())
        model_attrs['define_horizontal_diffusivity_limits'] = int(
            self.ui.grp_global_horiz_diffusivity_limits.isChecked()
        )
        model_attrs['horizontal_scalar_diffusivity_min'] = float(self.ui.edt_horiz_diffusivity_min.text())
        model_attrs['horizontal_scalar_diffusivity_max'] = float(self.ui.edt_horiz_diffusivity_max.text())

    def _save_vertical_diffusivity(self, model_attrs):
        """Save vertical diffusivity widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['define_vertical_scalar_diffusivity'] = int(self.ui.grp_global_vert_diffusivity.isChecked())
        model_attrs['vertical_scalar_diffusivity'] = float(self.ui.edt_global_vert_diffusivity.text())
        model_attrs['vertical_scalar_diffusivity_coef1'] = float(self.ui.edt_global_vert_diffusivity_coef1.text())
        model_attrs['vertical_scalar_diffusivity_coef2'] = float(self.ui.edt_global_vert_diffusivity_coef2.text())
        model_attrs['define_vertical_diffusivity_limits'] = int(self.ui.grp_global_vert_diffusivity_limits.isChecked())
        model_attrs['vertical_scalar_diffusivity_min'] = float(self.ui.edt_vert_diffusivity_min.text())
        model_attrs['vertical_scalar_diffusivity_max'] = float(self.ui.edt_vert_diffusivity_max.text())

    def _save_stability_limits(self, model_attrs):
        """Save stability limits widgets in the 'Global parameters' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['define_stability_limits'] = int(self.ui.grp_stability_limits.isChecked())
        model_attrs['stability_wse'] = float(self.ui.edt_stability_wse.text())
        model_attrs['stability_velocity'] = float(self.ui.edt_stability_velocity.text())

    def _save_global_bc(self, model_attrs):
        """Save BC default update dt and transport widgets in the 'Boundary conditions' tab.

        Args:
            model_attrs (dict): The material Dataset attrs
        """
        model_attrs['define_bc_default_update_dt'] = int(self.ui.grp_bc_update_dt.isChecked())
        model_attrs['bc_default_update_dt'] = float(self.ui.edt_bc_update_dt.text())
        model_attrs['include_wave_stress'] = 1 if self.ui.tog_include_wave_stress.isChecked() else 0
        model_attrs['include_stokes_drift'] = 1 if self.ui.tog_include_stokes_drift.isChecked() else 0
        model_attrs['transport_mode'] = int(self.ui.grp_transport_mode.isChecked())
        transport_file = self.ui.lbl_transport_file_selection.text()
        model_attrs['transport_file'] = ''
        if transport_file != gui_util.NULL_SELECTION:
            model_attrs['transport_file'] = transport_file

    def _save_wind_stress(self):
        """Save data from the 'Wind stress' tab to in-memory Datasets."""
        wind_attrs = self.data.wind_stress.attrs
        wind_attrs['method'] = self.ui.cbx_wind_stress.itemData(self.ui.cbx_wind_stress.currentIndex())
        wind_attrs['define_wind'] = int(self.ui.grp_define_wind_stress.isChecked())
        wind_attrs['define_parameters'] = int(self.ui.grp_define_wind_stress_parameters.isChecked())
        wind_attrs['wa'] = float(self.ui.edt_wind_stress_wa.text())
        wind_attrs['ca'] = float(self.ui.edt_wind_stress_ca.text())
        wind_attrs['wb'] = float(self.ui.edt_wind_stress_wb.text())
        wind_attrs['cb'] = float(self.ui.edt_wind_stress_cb.text())
        wind_attrs['bulk_coefficient'] = float(self.ui.edt_wind_stress_bulk.text())
        wind_attrs['scale_factor'] = float(self.ui.edt_wind_stress_scale.text())

    def _save_geometry_data(self):
        """Save data from the 'Geometry' tab to in-memory Datasets."""
        dset = self.layer_faces_table.model.data_frame.to_xarray()
        dset.attrs['define_bed_elevation_limits'] = int(self.ui.grp_bed_elevation_limits.isChecked())
        dset.attrs['min_bed_elevation'] = float(self.ui.edt_bed_elevation_min.text())
        dset.attrs['max_bed_elevation'] = float(self.ui.edt_bed_elevation_max.text())
        dset.attrs['vertical_mesh_type'] = self.ui.cbx_vertical_mesh_type.itemData(
            self.ui.cbx_vertical_mesh_type.currentIndex()
        )
        dset.attrs['layer_faces'] = self.ui.cbx_layer_faces.itemData(self.ui.cbx_layer_faces.currentIndex())
        csv_file = self.ui.lbl_layer_faces_csv_selection.text()
        dset.attrs['layer_faces_file'] = ''
        if csv_file != gui_util.NULL_SELECTION:
            dset.attrs['layer_faces_file'] = csv_file
        dset.attrs['sigma_layers'] = int(self.ui.edt_sigma_layers.text())
        dset.attrs['cell_3d_depth'] = float(self.ui.edt_cell_3d_depth.text())
        dset.attrs['min_bed_layer_thickness'] = float(self.ui.edt_min_bottom_thickness.text())
        dset.attrs['define_cell_depths'] = int(self.ui.grp_cell_wet_dry_depths.isChecked())
        dset.attrs['dry_cell_depth'] = float(self.ui.edt_cell_dry_depth.text())
        dset.attrs['wet_cell_depth'] = float(self.ui.edt_cell_wet_depth.text())
        self.data.geometry = dset
        self.data.z_modifications = self.z_modifications_table.model.data_frame.to_xarray()

    def _save_initial_conditions(self):
        """Save data from the 'Initial conditions' tab to in-memory Datasets."""
        initial_attrs = self.data.initial_conditions.attrs
        initial_attrs['define_initial_water_level'] = 1 if self.ui.tog_initial_wse.isChecked() else 0
        initial_attrs['initial_water_level'] = float(self.ui.edt_initial_water_level.text())
        initial_attrs['use_restart_file'] = 1 if self.ui.grp_restart.isChecked() else 0
        initial_attrs['use_restart_file_time'] = 1 if self.ui.tog_use_restart_time.isChecked() else 0
        restart_file = self.ui.lbl_restart_file_selection.text()
        if restart_file != gui_util.NULL_SELECTION:
            initial_attrs['restart_file'] = restart_file

    def _save_output(self):
        """Save data from the 'Output' tab to in-memory Datasets."""
        df = self.output_blocks_table.model.data_frame
        selected_list = self.output_blocks_table.table_view.selectionModel().selectedIndexes()
        if selected_list:  # If there is a selected row extract its options for the options pane widgets
            row = selected_list[0].row()
            self._store_old_output_block_options(row, df)
        dset = df.to_xarray()
        dset.attrs['log_dir'] = self.ui.edt_log_dir.text()
        dset.attrs['output_dir'] = self.ui.edt_output_dir.text()
        dset.attrs['write_check_files'] = 1 if self.ui.grp_write_check_files.isChecked() else 0
        dset.attrs['check_files_dir'] = self.ui.edt_write_check_files.text()
        dset.attrs['write_empty_gis_files'] = 1 if self.ui.grp_write_empty_gis_files.isChecked() else 0
        dset.attrs['empty_gis_files_dir'] = self.ui.edt_write_empty_gis_files.text()
        dset.attrs['write_restart_file'] = 1 if self.ui.grp_output_restart_file.isChecked() else 0
        dset.attrs['restart_file_interval'] = float(self.ui.edt_ouput_restart_file_dt.text())
        dset.attrs['overwrite_restart_file'] = 1 if self.ui.tog_overwrite_output_restart_file.isChecked() else 0
        self.data.output = dset
        # Store the output coverages in case the user never changed output blocks.
        self._store_old_output_coverage_options()

    def _save_materials(self):
        """Save data from the 'Materials' tab to in-memory Datasets."""
        dset = self.material_model.data_frame.to_xarray()
        dset.attrs['define_set_mat'] = int(self.ui.grp_set_mat.isChecked())
        dset.attrs['global_roughness'] = self.ui.cbx_global_roughness.itemData(
            self.ui.cbx_global_roughness.currentIndex()
        )
        dset.attrs['global_roughness_coefficient'] = float(self.ui.edt_global_roughness.text())
        dset.attrs['define_global_roughness'] = int(self.ui.grp_global_roughness.isChecked())
        self.data.global_set_mat = dset

    def _save_simulation_links(self):
        """Save the simulation links table to the simulation data."""
        df = self.linked_simulations_table.model.data_frame
        # Drop invalid selections
        df = df[~df.uuid.isin(['', gui_util.NULL_SELECTION])]
        self.data.linked_simulations = df.to_xarray()

    def _save_advanced_cards(self):
        """Save the advanced cards text to the simulation data."""
        self.data.general.attrs['advanced_cards'] = self.code_editor.toPlainText()

    def _store_old_output_block_options(self, row, df):
        """Store output block options when the current row changes.

        Args:
            row (int): The deselected row (1 base)
            df (pandas.DataFrame): The output block model DataFrame
        """
        row = row + 1  # Offset to match non-zero indexed rows
        df['define_interval'][row] = int(self.ui.grp_define_interval.isChecked())
        df['output_interval'][row] = float(self.ui.edt_output_interval.text())
        df['define_start'][row] = int(self.ui.grp_define_start.isChecked())
        df['output_start'][row] = float(self.ui.edt_output_start_time.text())
        df['output_start_date'][row] = self.ui.date_output_start_time.dateTime().toPython().strftime(
            ISO_DATETIME_FORMAT
        )
        df['define_final'][row] = int(self.ui.grp_define_final.isChecked())
        df['output_final'][row] = float(self.ui.edt_output_final_time.text())
        df['output_final_date'][row] = self.ui.date_output_final_time.dateTime().toPython().strftime(
            ISO_DATETIME_FORMAT
        )
        df['define_compression'][row] = int(self.ui.grp_define_compression.isChecked())
        df['compression'][row] = int(self.ui.tog_compression.isChecked())
        df['define_statistics'][row] = int(self.ui.grp_define_statistics.isChecked())
        df['statistics_type'][row] = self.ui.cbx_statistics.itemData(self.ui.cbx_statistics.currentIndex())
        df['define_statistics_dt'][row] = int(self.ui.tog_define_statistics_dt.isChecked())
        df['statistics_dt'][row] = float(self.ui.edt_statistics_dt.text())
        if self.ui.grp_suffix.isChecked():
            df['suffix'][row] = self.ui.edt_suffix.text()
        else:
            df['suffix'][row] = ''

        df['depth_output'][row] = int(self.ui.tog_output_depth.isChecked())
        df['wse_output'][row] = int(self.ui.tog_output_wse.isChecked())
        df['bed_shear_stress_output'][row] = int(self.ui.tog_output_bed_shear_stress.isChecked())
        df['surface_shear_stress_output'][row] = int(self.ui.tog_output_surface_shear_stress.isChecked())
        df['velocity_output'][row] = int(self.ui.tog_output_velocity.isChecked())
        df['velocity_mag_output'][row] = int(self.ui.tog_output_velocity_mag.isChecked())
        df['vertical_velocity_output'][row] = int(self.ui.tog_output_vertical_velocity.isChecked())
        df['bed_elevation_output'][row] = int(self.ui.tog_output_bed_elevation.isChecked())
        df['turb_visc_output'][row] = int(self.ui.tog_output_turb_visc.isChecked())
        df['turb_diff_output'][row] = int(self.ui.tog_output_turb_diff.isChecked())
        df['air_temp_output'][row] = int(self.ui.tog_output_air_temp.isChecked())
        df['evap_output'][row] = int(self.ui.tog_output_evap.isChecked())
        df['dzb_output'][row] = int(self.ui.tog_output_dzb.isChecked())
        df['hazard_z1_output'][row] = int(self.ui.tog_output_hazard_z1.isChecked())
        df['hazard_zaem1_output'][row] = int(self.ui.tog_output_hazard_zaem1.isChecked())
        df['hazard_zqra_output'][row] = int(self.ui.tog_output_hazard_zqra.isChecked())
        df['lw_rad_output'][row] = int(self.ui.tog_output_lw_rad.isChecked())
        df['mslp_output'][row] = int(self.ui.tog_output_mslp.isChecked())
        df['precip_output'][row] = int(self.ui.tog_output_precip.isChecked())
        df['rel_hum_output'][row] = int(self.ui.tog_output_rel_hum.isChecked())
        df['rhow_output'][row] = int(self.ui.tog_output_rhow.isChecked())
        df['sal_output'][row] = int(self.ui.tog_output_sal.isChecked())
        df['sw_rad_output'][row] = int(self.ui.tog_output_sw_rad.isChecked())
        df['temp_output'][row] = int(self.ui.tog_output_temp.isChecked())
        df['turbz_output'][row] = int(self.ui.tog_output_turbz.isChecked())
        df['w10_output'][row] = int(self.ui.tog_output_w10.isChecked())
        df['wq_all_output'][row] = int(self.ui.tog_output_wq_all.isChecked())
        df['wq_diag_all_output'][row] = int(self.ui.tog_output_wq_diag_all.isChecked())
        df['wvht_output'][row] = int(self.ui.tog_output_wvht.isChecked())
        df['wvper_output'][row] = int(self.ui.tog_output_wvper.isChecked())
        df['wvdir_output'][row] = int(self.ui.tog_output_wvdir.isChecked())
        df['wvstr_output'][row] = int(self.ui.tog_output_wvstr.isChecked())

    def _load_new_output_block_options(self, row_data):
        """Load output block options when the current row changes.

        Args:
            row_data (pandas.Series): The output block model Series for the newly selected row
        """
        self.ui.grp_define_interval.setChecked(bool(int(row_data['define_interval'])))
        self.ui.edt_output_interval.setText(str(row_data['output_interval']))
        self.ui.grp_define_start.setChecked(bool(int(row_data['define_start'])))
        self.ui.edt_output_start_time.setText(str(row_data['output_start']))
        gui_util.intialize_datetime_widget(str(row_data['output_start_date']), self.ui.date_output_start_time,
                                           self.time_format_qt)
        self.ui.grp_define_final.setChecked(bool(int(row_data['define_final'])))
        self.ui.edt_output_final_time.setText(str(row_data['output_final']))
        gui_util.intialize_datetime_widget(str(row_data['output_final_date']), self.ui.date_output_final_time,
                                           self.time_format_qt)
        self.ui.grp_define_compression.setChecked(bool(int(row_data['define_compression'])))
        self.ui.tog_compression.setCheckState(
            Qt.Checked if int(row_data['compression']) == 1 else Qt.Unchecked
        )
        self.ui.grp_define_statistics.setChecked(bool(int(row_data['define_statistics'])))
        gui_util.set_combobox_from_data(self.ui.cbx_statistics, row_data['statistics_type'])
        self.ui.tog_define_statistics_dt.setCheckState(
            Qt.Checked if int(row_data['define_statistics_dt']) == 1 else Qt.Unchecked
        )
        self.ui.edt_statistics_dt.setText(str(row_data['statistics_dt']))
        # Set the check state of the filename suffix based on if one has been previously defined. This is a GUI only
        # widget and we do not store its check state. It is implied that the option is disabled if the value of the
        # suffix edit field is empty string.
        suffix = str(row_data['suffix'])
        self.ui.grp_suffix.setChecked(suffix != '')
        self.ui.edt_suffix.setText(suffix)

        self.ui.tog_output_wse.setCheckState(Qt.Checked if int(row_data['wse_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_depth.setCheckState(
            Qt.Checked if int(row_data['depth_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_wse.setCheckState(Qt.Checked if int(row_data['wse_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_bed_shear_stress.setCheckState(
            Qt.Checked if int(row_data['bed_shear_stress_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_surface_shear_stress.setCheckState(
            Qt.Checked if int(row_data['surface_shear_stress_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_velocity.setCheckState(Qt.Checked if int(row_data['velocity_output']) == 1 else Qt.Unchecked)

        self.ui.tog_output_velocity_mag.setCheckState(
            Qt.Checked if int(row_data['velocity_mag_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_vertical_velocity.setCheckState(
            Qt.Checked if int(row_data['vertical_velocity_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_bed_elevation.setCheckState(
            Qt.Checked if int(row_data['bed_elevation_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_turb_visc.setCheckState(
            Qt.Checked if int(row_data['turb_visc_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_turb_diff.setCheckState(
            Qt.Checked if int(row_data['turb_diff_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_air_temp.setCheckState(Qt.Checked if int(row_data['air_temp_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_evap.setCheckState(Qt.Checked if int(row_data['evap_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_dzb.setCheckState(Qt.Checked if int(row_data['dzb_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_hazard_z1.setCheckState(
            Qt.Checked if int(row_data['hazard_z1_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_hazard_zaem1.setCheckState(
            Qt.Checked if int(row_data['hazard_zaem1_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_hazard_zqra.setCheckState(
            Qt.Checked if int(row_data['hazard_zqra_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_lw_rad.setCheckState(Qt.Checked if int(row_data['lw_rad_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_mslp.setCheckState(Qt.Checked if int(row_data['mslp_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_precip.setCheckState(Qt.Checked if int(row_data['precip_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_rel_hum.setCheckState(Qt.Checked if int(row_data['rel_hum_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_rhow.setCheckState(Qt.Checked if int(row_data['rhow_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_sal.setCheckState(Qt.Checked if int(row_data['sal_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_sw_rad.setCheckState(Qt.Checked if int(row_data['sw_rad_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_temp.setCheckState(Qt.Checked if int(row_data['temp_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_turbz.setCheckState(Qt.Checked if int(row_data['turbz_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_w10.setCheckState(Qt.Checked if int(row_data['w10_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_wq_all.setCheckState(Qt.Checked if int(row_data['wq_all_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_wq_diag_all.setCheckState(
            Qt.Checked if int(row_data['wq_diag_all_output']) == 1 else Qt.Unchecked
        )
        self.ui.tog_output_wvht.setCheckState(Qt.Checked if int(row_data['wvht_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_wvper.setCheckState(Qt.Checked if int(row_data['wvper_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_wvdir.setCheckState(Qt.Checked if int(row_data['wvdir_output']) == 1 else Qt.Unchecked)
        self.ui.tog_output_wvstr.setCheckState(Qt.Checked if int(row_data['wvstr_output']) == 1 else Qt.Unchecked)

    def _enable_widgets_for_flux_mass_transport_output(self, is_flux_or_mass):
        """Hide widgets for the flux, mass , or tansport output block types.

        Args:
            is_flux_or_mass (bool): True if the current row's output block format is 'Flux' or 'Mass'
        """
        self.ui.grp_output_datasets.setVisible(not is_flux_or_mass)
        self.ui.grp_define_statistics.setVisible(not is_flux_or_mass)

    """
    Slots
    """
    def on_output_block_changed(self, selected, deselected):
        """Slot called when row in the output block table changes.

        Args:
            selected (list): The new selection
            deselected (list): The old selection
        """
        df = self.output_blocks_table.model.data_frame
        if deselected and deselected[0].isValid():
            old_row = deselected[0].indexes()[0].row()
            old_row_data = df.iloc[old_row]
            if old_row_data['format'] == 'Points':
                self._store_old_output_coverage_options()
            self._store_old_output_block_options(old_row, df)

        if not selected or not selected[0].isValid():
            self.ui.grp_output_options.setTitle('Output block options')
            self.ui.scroll_area_output_blocks_options_contents.hide()
            return  # If no rows selected, don't show the option widgets in the right pane
        self.ui.scroll_area_output_blocks_options_contents.show()

        # If a flux or mass block, only show the output interval widgets.
        new_row = selected[0].indexes()[0].row()  # Because we are using iloc, do not add 1 to row for 1 based array
        new_row_data = df.iloc[new_row]
        output_type = new_row_data['format']
        flux_or_mass = output_type in ['Flux', 'Mass', 'Transport']
        self._enable_widgets_for_flux_mass_transport_output(flux_or_mass)

        # Hide the compression group if not applicable
        show_compression = output_type in ['XMDF', 'NetCDF']
        self.ui.grp_define_compression.setVisible(show_compression)

        # Only show the output points feature coverage selector if points type
        self.ui.grp_output_points_cov.setVisible(output_type == 'Points')
        if output_type == 'Points':
            self._load_new_output_coverage_options(new_row_data)

        self._load_new_output_block_options(new_row_data)
        # Update the groupbox text for the current row's options to give the user a better indication of the selection.
        self.ui.grp_output_options.setTitle(f'Output block options - Row {new_row + 1}')

    def on_output_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the output blocks table view has changed.

        Args:
            top_left_index (QModelIndex): The row whose data has changed
            bottom_right_index (QModelIndex): Unused, but required for the function signature
        """
        # Make sure the index is valid and the row is in the table. When the last row is deleted, this function is
        # called before the index is invalidated, but after the item has been removed from the table, so make sure
        # to check that the index row isn't greater than the table contents.
        if not (top_left_index.isValid() and self.output_blocks_table.model.rowCount() > top_left_index.row()):
            return
        # If a flux or mass block, only show the output interval widgets.
        row = top_left_index.row()
        output_type = self.output_blocks_table.model.data_frame['format'].iat[row]
        flux_mass = output_type in ['Flux', 'Mass', 'Transport']
        self._enable_widgets_for_flux_mass_transport_output(flux_mass)
        # Hide the compression group if not applicable
        show_compression = output_type in ['XMDF', 'NetCDF']
        self.ui.grp_define_compression.setVisible(show_compression)
        # Only show the output points feature coverage selector if points type
        self.ui.grp_output_points_cov.setVisible(output_type == 'Points')
        if output_type == 'Points':
            row_data = self.output_blocks_table.model.data_frame.iloc[row]
            # if row_data['row_index'] < 0:
            #     self._data['row_index'].iloc[row] = self._data.output.attrs['next_row_index']
            self._load_new_output_coverage_options(row_data)

    def _load_new_output_coverage_options(self, row_data):
        """Load output coverage options when the current row changes.

        Args:
            row_data (pandas.Series): The output block model Series for the newly selected row
        """
        dset = self.data.output_points_coverages
        sliced_dset = dset.where(dset.row_index == row_data.row_index, drop=True)
        self.output_points_coverages_table.model.beginResetModel()
        self.output_points_coverages_table.model.data_frame = sliced_dset.to_dataframe()
        self.output_points_coverages_table.model.endResetModel()

    def _store_old_output_coverage_options(self):
        """Save the old output coverages when the row changes."""
        # Remove any coverages that the user deleted from the table.
        delete_uuids = self.output_points_coverages_table.remove_uuids
        delete_indices = self.output_points_coverages_table.remove_indices
        self.data.remove_output_coverages(delete_uuids, delete_indices)
        self.output_points_coverages_table.remove_uuids = []
        self.output_points_coverages_table.remove_indices = []
        # Add any new coverages that the user added to the table.
        new_rows = self.output_points_coverages_table.model.data_frame
        num_filtered = self.data.add_output_coverage(df=new_rows)
        if num_filtered > 0:
            pass
            # TODO: Get this working. Need to not include the rows we are adding if they already exist in the
            #       DataFrame (uuid + row) as rows we filtered.
            # msg = f'{num_filtered} duplicate or undefined coverage(s) were filtered out.'
            # message_with_ok(parent=self, message=msg, app_name='SMS', icon='Warning',
            #                 win_icon=self.windowIcon())

    def on_tog_use_salinity(self, state):
        """Called when use salinity toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.tog_couple_salinity.setVisible(state == Qt.Checked)

    def on_tog_use_temperature(self, state):
        """Called when use temperature toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.tog_couple_temp.setVisible(state == Qt.Checked)

    def on_tog_use_sediment(self, state):
        """Called when use sediment toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.tog_couple_sediment.setVisible(state == Qt.Checked)

    def on_tog_use_isodate(self, state):
        """Called when use ISODATE toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        use_isodate = state == Qt.Checked
        self.ui.date_reference.setVisible(use_isodate)
        self.ui.date_start_time.setVisible(use_isodate)
        self.ui.date_end_time.setVisible(use_isodate)
        self.ui.date_output_start_time.setVisible(use_isodate)
        self.ui.date_output_final_time.setVisible(use_isodate)
        self.ui.edt_ref_date.setVisible(not use_isodate)
        self.ui.edt_start_time.setVisible(not use_isodate)
        self.ui.edt_end_time.setVisible(not use_isodate)
        self.ui.edt_output_start_time.setVisible(not use_isodate)
        self.ui.edt_output_final_time.setVisible(not use_isodate)
        # Update label text.
        if use_isodate:
            self.ui.lbl_ref_date.setText('Reference date:')
            self.ui.lbl_start_time.setText('Starting time:')
            self.ui.label_end_time.setText('Ending time:')
            self.ui.lbl_output_start_time.setText('Starting time:')
            self.ui.lbl_output_final_time.setText('Ending time:')
        else:
            self.ui.lbl_ref_date.setText('Reference time (h):')
            self.ui.lbl_start_time.setText('Starting time (h):')
            self.ui.label_end_time.setText('Ending time (h):')
            self.ui.lbl_output_start_time.setText('Starting time (h):')
            self.ui.lbl_output_final_time.setText('Ending time (h):')

    def on_tog_is_3d(self, state):
        """Called when is 3D toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        enabled = state == Qt.Checked
        self.ui.grp_mixing_model_vert.setVisible(enabled)
        self.ui.grp_3d_geometry.setVisible(enabled)
        self.ui.lbl_spatial_order_vert.setVisible(enabled)
        self.ui.cbx_spatial_order_vert.setVisible(enabled)
        self.ui.tog_output_vertical_velocity.setVisible(enabled)

    def on_hardware_solver_changed(self, current_idx):
        """Slot called when combobox option for hardware solver changes.

        Args:
            current_idx (int): Current index of the combobox
        """
        enabled = current_idx == 1  # 0 == 'CPU', 1 == 'GPU'
        self.ui.lbl_device_id.setVisible(enabled)
        self.ui.edt_device_id.setVisible(enabled)

    def on_global_roughness_changed(self, current_idx):
        """Slot called when combobox option for global roughness method changes.

        Args:
            current_idx (int): Current index of the combobox
        """
        # 0 = Mannings N (default), 1 = KS
        self.ui.lbl_roughness_coefficient.setText(
            'Nikuradse roughness (m)'if current_idx == 1 else "Manning's n coefficient"
        )

    def on_horiz_mixing_model_changed(self, index):
        """Called when the horizontal mixing model combobox option is changed.

        Args:
            index (int): Current combobox index
        """
        self.ui.layout_global_viscosity_horiz.setVisible(index != const.HORIZ_MIXING_OPT_NONE)
        smagorinsky_or_wu = index in [const.HORIZ_MIXING_OPT_SMAGORINSKY, const.HORIZ_MIXING_OPT_WU]
        self.ui.grp_horiz_viscosity_limits.setVisible(smagorinsky_or_wu)
        if index == const.HORIZ_MIXING_OPT_CONSTANT:
            self.ui.lbl_global_viscosity_horiz.setText('Global horizontal eddy viscosity (m²/s):')
            if self._imperial:
                UnitsLabelSwitcher.switch_widget_to_imperial(self.ui.lbl_global_viscosity_horiz)
        elif smagorinsky_or_wu:
            self.ui.lbl_global_viscosity_horiz.setText('Global horizontal eddy viscosity coefficient:')

    def on_vert_mixing_model_changed(self, index):
        """Called when the vertical mixing model combobox option is changed.

        Args:
            index (int): Current combobox index
        """
        if index == 0:  # 0 = Constant
            self.ui.layout_global_viscosity_vert.setVisible(True)
            self.ui.layout_parametric_coefficients.setVisible(False)
            self.ui.layout_external_turbulence.setVisible(False)
            self.ui.grp_vertical_viscosity_limits.setVisible(False)
            self.ui.grp_global_vert_diffusivity_limits.setVisible(False)
            self.ui.layout_global_diffusivity_vert.setVisible(True)
            self.ui.layout_global_diffusivity_vert_coef.setVisible(False)
        elif index == 1:  # 1 = Parametric
            self.ui.layout_global_viscosity_vert.setVisible(False)
            self.ui.layout_external_turbulence.setVisible(False)
            self.ui.grp_vertical_viscosity_limits.setVisible(True)
            self.ui.grp_global_vert_diffusivity_limits.setVisible(True)
            self.ui.layout_global_diffusivity_vert.setVisible(False)
            self.ui.layout_global_diffusivity_vert_coef.setVisible(True)
        elif index == 2:  # 2 = External
            self.ui.layout_global_viscosity_vert.setVisible(False)
            self.ui.layout_parametric_coefficients.setVisible(False)
            self.ui.layout_external_turbulence.setVisible(True)
            self.ui.grp_vertical_viscosity_limits.setVisible(True)
            self.ui.grp_global_vert_diffusivity_limits.setVisible(True)
            self.ui.layout_global_diffusivity_vert.setVisible(False)
            self.ui.layout_global_diffusivity_vert_coef.setVisible(False)

    def on_horiz_diffusivity_model_changed(self, index):
        """Called when the horizontal diffusivity model combobox option is changed.

        Args:
            index (int): Current combobox index
        """
        self.ui.layout_global_diffusivity_horiz.setVisible(
            index not in [const.HORIZ_DIFFUSIVITY_OPT_NONE, const.HORIZ_DIFFUSIVITY_OPT_ELDER]
        )
        self.ui.layout_global_diffusivity_horiz_coef.setVisible(index == const.HORIZ_DIFFUSIVITY_OPT_ELDER)
        self.ui.grp_global_horiz_diffusivity_limits.setVisible(
            index not in [const.HORIZ_DIFFUSIVITY_OPT_NONE, const.HORIZ_DIFFUSIVITY_OPT_ELDER]
        )

    def on_tog_turbulence_update(self, state):
        """Called when is define turbulence update toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        enabled = state == Qt.Checked
        self.ui.edt_turbulence_update.setEnabled(enabled)

    def on_tog_initial_wse(self, state):
        """Called when the initial WSE toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.edt_initial_water_level.setEnabled(state == Qt.Checked)

    def on_vertical_mesh_type_changed(self, index):
        """Called when the vertical mesh type combobox option is changed.

        Args:
            index (int): Current combobox index
        """
        self.ui.grp_layer_faces.setVisible(index == 1)  # 1 = Z

    def on_layer_faces_changed(self, index):
        """Called when the layer faces combobox option is changed.

        Args:
            index (int): Current combobox index
        """
        if index == 0:  # 0 = CSV
            self.ui.layout_layer_faces_csv.setVisible(True)
            self.layer_faces_table.setVisible(False)
        else:  # Specified
            self.ui.layout_layer_faces_csv.setVisible(False)
            self.layer_faces_table.setVisible(True)

    def on_wind_stress_changed(self, index):
        """Called when the wind stress model type combobox option changes.

        Args:
            index (int): Current combobox index
        """
        show_wu = index + 1 == const.WIND_STRESS_OPT_WU
        show_constant = index + 1 == const.WIND_STRESS_OPT_CONSTANT
        show_kondo = index + 1 == const.WIND_STRESS_OPT_KONDO
        self.ui.lbl_wind_stress_wa.setVisible(show_wu)
        self.ui.edt_wind_stress_wa.setVisible(show_wu)
        self.ui.lbl_wind_stress_ca.setVisible(show_wu)
        self.ui.edt_wind_stress_ca.setVisible(show_wu)
        self.ui.lbl_wind_stress_wb.setVisible(show_wu)
        self.ui.edt_wind_stress_wb.setVisible(show_wu)
        self.ui.lbl_wind_stress_cb.setVisible(show_wu)
        self.ui.edt_wind_stress_cb.setVisible(show_wu)
        self.ui.lbl_wind_stress_bulk.setVisible(show_constant)
        self.ui.edt_wind_stress_bulk.setVisible(show_constant)
        self.ui.lbl_wind_stress_scale.setVisible(show_kondo)
        self.ui.edt_wind_stress_scale.setVisible(show_kondo)

    def on_btn_external_turbulence(self):
        """Called when the external turbulence directory selector button is clicked."""
        gui_util.select_file(self, self.ui.lbl_external_turbulence_selection, 'Select a turbulence model folder',
                             '', self.data.info.attrs['proj_dir'], True)

    def on_btn_layer_faces_csv(self):
        """Called when the layer faces CSV file selector button is clicked."""
        gui_util.select_file(self, self.ui.lbl_layer_faces_csv_selection, 'Select a layer faces CSV file',
                             'CSV files (*.csv);;All Files (*.*)', self.data.info.attrs['proj_dir'], False)

    def on_tog_statistics_dt(self, state):
        """Called when define statistics dt toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.edt_statistics_dt.setEnabled(state == Qt.Checked)

    def on_btn_restart_file(self):
        """Called when the hot start file selector button is clicked."""
        gui_util.select_file(self, self.ui.lbl_restart_file_selection, 'Select a restart file',
                             'Restart files (*.rst);;All Files (*.*)', self.data.info.attrs['proj_dir'], False)

    def on_tog_use_restart_time(self, _):
        """Hide/show note label in time tab about restart file start time being use when options change."""
        # The options for using a restart file and using the restart file time must both be enabled to show the note.
        show_note = self.ui.grp_restart.isChecked() and self.ui.tog_use_restart_time.isChecked()
        self.ui.lbl_restart_start_time_note.setVisible(show_note)
        self.ui.lbl_start_time.setEnabled(not show_note)
        self.ui.edt_start_time.setEnabled(not show_note)
        self.ui.date_start_time.setEnabled(not show_note)

    def on_btn_transport_file(self):
        """Called when the transport BC file selector button is clicked."""
        gui_util.select_file(self, self.ui.lbl_transport_file_selection, 'Select a transport BC file',
                             'NetCDF files (*.nc);;All Files (*.*)', self.data.info.attrs['proj_dir'], False)

    def on_toggle_changed(self, index):
        """Called when the use default checkbox is toggled for a row.

        Args:
            index (QModelIndex): Model index of the toggle whose state changed
        """
        if not index.isValid():
            return
        row = index.row()
        col = index.column()

        update_indices = []
        table_view = self.set_mat_table
        if col == MaterialData.COL_OVERRIDE_BOTTOM_ROUGHNESS:
            update_indices.append(table_view.filter_model.index(row, MaterialData.COL_BOTTOM_ROUGHNESS))
        for update_idx in update_indices:
            table_view.update(update_idx)

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        restore_splitter_geometry(splitter=self.output_splitter, package_name='xms.tuflowfv',
                                  dialog_name=f'{self._dlg_name}_output_splitter')
        restore_splitter_geometry(splitter=self.ui.splitter_bc, package_name='xms.tuflowfv',
                                  dialog_name=f'{self._dlg_name}_splitter_bc')
        restore_splitter_geometry(splitter=self.ui.splitter_bc_vert, package_name='xms.tuflowfv',
                                  dialog_name=f'{self._dlg_name}_splitter_bc_vert')

    def accept(self):
        """Save dialog data on accepted."""
        self._save_data()
        save_splitter_geometry(splitter=self.output_splitter, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_output_splitter')
        save_splitter_geometry(splitter=self.ui.splitter_bc, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_splitter_bc')
        save_splitter_geometry(splitter=self.ui.splitter_bc_vert, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_splitter_bc_vert')
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        # We always save the splitter position, even if user rejects.
        save_splitter_geometry(splitter=self.output_splitter, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_output_splitter')
        save_splitter_geometry(splitter=self.ui.splitter_bc, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_splitter_bc')
        save_splitter_geometry(splitter=self.ui.splitter_bc_vert, package_name='xms.tuflowfv',
                               dialog_name=f'{self._dlg_name}_splitter_bc_vert')
        super().reject()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)
