"""Bc class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
from enum import Enum, IntEnum

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.wash.tools import types


class BcTypeEnum(str, Enum):  # StrEnum is not available in Python 3.10
    """Rate type enum."""
    PS1_well_flow = 'PS1'  # At points
    PS2_well_conc = 'PS2'  # At points
    DB1_spec_head = 'DB1'  # At points
    DB2_spec_conc = 'DB2'  # At points
    CB1_spec_flux = 'CB1'  # At faces
    CB2_spec_mass_flux = 'CB2'  # At faces
    RS1_var_flux = 'RS1'  # At faces
    RS2_var_conc = 'RS2'  # At faces
    RW1_relief_well = 'RW1'  # at points
    BcNone = 'None'


class RateTypeEnum(IntEnum):
    """Rate type enum."""
    CONSTANT = 0
    TRANSIENT = 1


@dataclass
class Bc:
    """A boundary condition."""
    bc_type: BcTypeEnum = BcTypeEnum.BcNone  # PS1_well_flow, PS2_well_conc, DB1_spec_head...
    rate_type: RateTypeEnum = RateTypeEnum.CONSTANT  # CONSTANT or TRANSIENT
    constant: float = 0.0  # Constant value
    xy_series_id: int = types.XM_NONE  # XySeries ID
    factor: float = 1.0  # Scaling factor


@dataclass
class PointBc:
    """Class to hold boundary conditions at points (boundarynoderec in GMS)."""
    head: Bc = None  # DB1
    conc: Bc = None  # DB2


@dataclass
class FaceBc:
    """Class to hold boundary conditions at faces (boundaryfacerec in GMS)."""
    flux: Bc = None  # CB1 or RS1
    conc: Bc = None  # CB2 or RS2


@dataclass
class Well:
    """Class to hold wells at points (wellrec in GMS)."""
    pump: Bc = None  # PS1
    conc: Bc = None  # PS2


@dataclass
class ReliefWell2:
    """Class to hold relief wells at points."""
    nodes: list[int] = None
    screen_flags: list[bool] = None
    time_type: int = 1  # 1 if transient, 0 if constant
    constant: float = 0.0  # Constant value
    xy_series_id: int = types.XM_NONE  # XySeries ID
    profile_type: int = 1  # 1 for total head, 0 for pressure head
    ref_elev: float = 0.0
    sat_k: float = 0.0
    well_diameter: float = 0.0
    well_resistance: float = 0.0


@dataclass
class ReliefWell:
    """Class to hold relief wells at points."""
    head: Bc = None  # this gives us the BC type, rate type, constant, xyseries and a factor
    nodes: list[int] = None
    screen_flags: list[bool] = None
    profile_type: int = 1  # 1 for total head, 0 for pressure head
    ref_elev: float = 0.0
    sat_k: float = 0.0
    well_diameter: float = 0.0
    well_resistance: float = 0.0
