"""Worker thread for the ww3_shel.nml importer."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.wavewatch3.file_io.shell_nml_reader import ShellNmlReader


class ShellImportWorkerThread(QThread):
    """Worker thread for importing an WW3 shell input file."""
    processing_finished = Signal()

    def __init__(self):
        """Construct the worker."""
        super().__init__()
        self.reader = None

    def run(self):
        """Thread runner that imports the ww3_shel.nml file."""
        try:
            self.reader = ShellNmlReader()
            self.reader.read()
        except Exception:
            logging.getLogger('xms.wavewatch3').exception('Error(s) encountered while reading WaveWatch3 shell file.')
        finally:
            self.processing_finished.emit()


def import_shell_with_feedback():
    """Driver for the shell import script with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':  # pragma: no cover
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope
    else:
        win_cont = None
        main_hwnd = None

    worker = ShellImportWorkerThread()
    display_text = {
        'title': 'Reading WaveWatch3 Simulation',
        'working_prompt': 'Reading WaveWatch3 control file. Please wait...',
        'error_prompt': 'Error(s) encountered reading simulation file. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading simulation file. Review log output for more details.',
        'success_prompt': 'Successfully read WaveWatch3 simulation.',
        'note': '',
        'auto_load': 'Auto load data into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.wavewatch3', worker, win_cont)
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send()  # Send data back to XMS
    if win_cont is not None:  # pragma: no cover
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
