"""Class to write a WaveWatch3 namelists nml file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.wavewatch3.file_io.namelists_defaults_util import WW3NamelistsDefaultsUtil


class WW3NamelistsNmlWriter:
    """Class to write a WaveWatch3 namelists nml file (related to grid namelist)."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.wavewatch3')
        self._bound_process = None
        self._grid_process = None
        self._xms_data = xms_data
        self._defaults = WW3NamelistsDefaultsUtil()
        self._parameter_names = {}

        # Get parameter names and values
        for group in self._xms_data.sim_data_model_control.groups:
            group_name = group[0]  # group[0] is the group name
            group_obj = self._xms_data.sim_data_model_control.group(group_name)
            for param_name in group_obj.parameter_names:
                self._parameter_names[param_name] = group_obj.parameter(param_name).value
        self._set_dependent_parameter_values()

    def _set_dependent_parameter_values(self):
        """
        Some namelist variables depend solely on others.

        These dependent variables have been removed and are created here for export.
        """
        params = self._parameter_names
        self._parameter_names['UNSTEXPFSN'] = 1 if params['method'] == 'Explicit' else 0  # True if Explicit
        self._parameter_names['UNSTEXPFSPSI'] = 0  # False for both Explicit and Implicit
        self._parameter_names['UNSTEXPFSFCT'] = 0  # False for both Explicit and Implicit
        self._parameter_names['UNSTIMPFSN'] = 0  # False for both Explicit and Implicit
        self._parameter_names['UNSTIMPTOTAL'] = 1 if params['method'] == 'Implicit' else 0  # True if Implicit
        self._parameter_names['UNSTEXPTOTAL'] = 0  # False for both Explicit and Implicit
        self._parameter_names['UNSTIMPREFRACTION'] = 1 if params['method'] == 'Implicit' else 0  # True if Implicit
        self._parameter_names['UNSTIMPFREQSHIFT'] = 1 if params['method'] == 'Implicit' else 0  # True if Implicit
        self._parameter_names['UNSTIMPSOURCE'] = 1 if params['method'] == 'Implicit' else 0  # True if Implicit

    def _write_namelists_nml_file(self):
        """Writes the namelist file."""
        file_w_path = "namelists.nml"
        self._write_constants_in_source_terms_data()
        self._write_propagation_schemes_data()
        self._write_unstructured_grids_data()
        self._write_output_data()
        self._write_miscellaneous_data()
        self._write_sea_state_stress_data()
        self._flush(file_w_path)

    def _write_constants_in_source_terms_data(self):
        """Writes out the various constants in source terms namelists."""
        self._write_stresses_data()
        self._write_linear_input_data()
        self._write_exponential_input_data()
        self._write_nonlinear_interactions_data()
        self._write_nonlinear_filter_data()
        self._write_whitecapping_dissipation_data()
        self._write_bottom_friction_data()
        self._write_surf_breaking_data()
        self._write_dissipation_in_the_ice_data()
        self._write_scattering_in_the_ice_data()
        self._write_triad_nonlinear_interactions_data()
        self._write_shoreline_reflections_data()
        self._write_bound_second_order_spectrum_data()

    def _write_stresses_data(self):
        """Writes out the stresses namelists."""
        consts = self._xms_data.sim_data_model_control.group('consts')
        if consts.parameter('stresses').value == 0:
            self._write_stresses_flx3_namelist()
        else:
            self._write_stresses_flx4_namelist()

    def _write_stresses_flx3_namelist(self):
        """Writes out the FLX3 namelist."""
        # $   TC 1996 with cap    : Namelist FLX3
        # $                           CDMAX  : Maximum allowed CD (cap)
        # $                           CTYPE  : Cap type :
        # $                                     0: Discontinuous (default).
        # $                                     1: Hyperbolic tangent.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='FLX3')

    def _write_stresses_flx4_namelist(self):
        """Writes out the FLX4 namelist."""
        # $   Hwang 2011          : Namelist FLX4
        # $                           CDFAC  : re-scaling of drag
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='FLX4')

    def _write_linear_input_data(self):
        """Writes out the linear input namelists."""
        self._write_linear_input_sln1_namelist()

    def _write_linear_input_sln1_namelist(self):
        """Writes out the SLN1 namelist."""
        # $   Cavaleri and M-R    : Namelist SLN1
        # $                           CLIN   : Proportionality constant.
        # $                           RFPM   : Factor for fPM in filter.
        # $                           RFHF   : Factor for fh in filter.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SLN1')

    def _write_exponential_input_data(self):
        """Writes out the namelists under exponential input."""
        consts = self._xms_data.sim_data_model_control.group('consts')
        if consts.parameter('exponential_input').value == 'WAM-3 [SIN1]':
            self._write_exponential_data_sin1_namelist()
        elif consts.parameter('exponential_input').value == 'Tolman and Chalikov [SIN2]':
            self._write_exponential_data_sin2_namelist()
        elif consts.parameter('exponential_input').value == 'WAM4 and Variants [SIN3]':
            self._write_exponential_data_sin3_namelist()
        elif consts.parameter('exponential_input').value == 'BYDRZ input [SIN6]':
            self._write_exponential_data_sin6_namelist()

    def _write_exponential_data_sin1_namelist(self):
        """Writes out the exponential data SIN1 namelist."""
        # $   WAM-3               : Namelist SIN1
        # $                           CINP   : Proportionality constant.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SIN1')

    def _write_exponential_data_sin2_namelist(self):
        """Writes out the exponential data SIN2 namelist."""
        # $   Tolman and Chalikov : Namelist SIN2
        # $                           ZWND   : Height of wind (m).
        # $                           SWELLF : swell factor in (n.nn).
        # $                           STABSH, STABOF, CNEG, CPOS, FNEG :
        # $                                    c0, ST0, c1, c2 and f1 in . (n.nn)
        # $                                    through (2.65) for definition of
        # $                                    effective wind speed (!/STAB2).
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SIN2')

    def _write_exponential_data_sin3_namelist(self):
        """Writes out the exponential data SIN3 namelist."""
        # $   WAM4 and variants  : Namelist SIN3
        # $                           ZWND    : Height of wind (m).
        # $                           ALPHA0  : minimum value of Charnock coefficient
        # $                           Z0MAX   : maximum value of air-side roughness z0
        # $                           BETAMAX : maximum value of wind-wave coupling
        # $                           SINTHP  : power of cosine in wind input
        # $                           ZALP    : wave age shift to account for gustiness
        # $                       TAUWSHELTER : sheltering of short waves to reduce u_star
        # $                         SWELLFPAR : choice of swell attenuation formulation
        # $                                    (1: TC 1996, 3: ACC 2008)
        # $                            SWELLF : swell attenuation factor
        # $     Extra parameters for SWELLFPAR=3 only
        # $                  SWELLF2, SWELLF3 : swell attenuation factors
        # $                           SWELLF4 : Threshold Reynolds number for ACC2008
        # $                           SWELLF5 : Relative viscous decay below threshold
        # $                             Z0RAT : roughness for oscil. flow / mean flow
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SIN3', attrs=attrs)
        if len(diff_keys) == 0:
            return
        do_extra = True if 'SWELLFPAR' in diff_keys else False
        extras = ['SWELLF2', 'SWELLF3', 'SWELLF4', 'SWELLF5', 'Z0RAT']
        self._ss.write('&SIN3\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SWELLFPAR':
                self._ss.write("    SWELLFPAR = 3\n")
            elif key in extras:
                if do_extra:
                    self._ss.write(f"    {var} = {attrs[key]}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_exponential_data_sin6_namelist(self):
        """Writes out the exponential data SIN6 namelist."""
        # $   BYDRZ input         : Namelist SIN6
        # $                          SINA0    : factor for negative input
        # $                          SINWS    : wind speed scaling option
        # $                          SINFC    : high-frequency extent of the
        # $                                     prognostic frequency region
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SIN6')

    def _write_nonlinear_interactions_data(self):
        """Writes out the namelists under Nonlinear interactions."""
        consts = self._xms_data.sim_data_model_control.group('consts')
        if consts.parameter('non_linear_interactions').value == 0:
            self._write_nonlinear_interactions_snl1_namelist()
        elif consts.parameter('non_linear_interactions').value == 1:
            self._write_nonlinear_interactions_snl2_namelist()
        elif consts.parameter('non_linear_interactions').value == 2:
            self._write_nonlinear_interactions_snl3_namelist()
        elif consts.parameter('non_linear_interactions').value == 3:
            self._write_nonlinear_interactions_snl4_namelist()

    def _write_nonlinear_interactions_snl1_namelist(self):
        """Writes out the nonlinear interactions SNL1 namelist."""
        # $   Discrete I.A.       : Namelist SNL1
        # $                           LAMBDA : Lambda in source term.
        # $                           NLPROP : C in sourc term. NOTE : default
        # $                                    value depends on other source
        # $                                    terms selected.
        # $                           KDCONV : Factor before kd in Eq. (n.nn).
        # $                           KDMIN, SNLCS1, SNLCS2, SNLCS3 :
        # $                                    Minimum kd, and constants c1-3
        # $                                    in depth scaling function.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SNL1')

    def _write_nonlinear_interactions_snl2_namelist(self):
        """Writes out the nonlinear interactions SNL2 namelist.  Also writes out ANL2 namelist if available."""
        # $   Exact interactions  : Namelist SNL2
        # $                           IQTYPE : Type of depth treatment
        # $                                     1 : Deep water
        # $                                     2 : Deep water / WAM scaling
        # $                                     3 : Shallow water
        # $                           TAILNL : Parametric tail power.
        # $                           NDEPTH : Number of depths in for which
        # $                                    integration space is established.
        # $                                    Used for IQTYPE = 3 only
        # $                         Namelist ANL2
        # $                           DEPTHS : Array with depths for NDEPTH = 3
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SNL2', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SNL2\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SNL2IQTYPE':
                # Special case for IQTYPE, which starts at 1
                self._ss.write(f"    {var} = {attrs[key] + 1}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')
        self._write_nonlinear_interactions_anl2_namelist()

    def _write_nonlinear_interactions_anl2_namelist(self):
        """Writes out the special nonlinear interactions ANL2 namelist."""
        # $                         Namelist ANL2
        # $                           DEPTHS : Array with depths for NDEPTH = 3
        anl2 = self._xms_data.sim_data.anl2depths
        if anl2 is not None:
            if 'Depth' in anl2 and anl2.Depth.size > 0:
                # Write the namelist only if data is available
                self._ss.write('&ANL2 DEPTHS = ')
                for i in range(anl2.Depth.size):
                    if i != 0:
                        # Fill in spaces on any additional lines after the first
                        self._ss.write('               ')
                    self._ss.write(f'{float(anl2.Depth[i])}')
                    if (i + 1) < anl2.Depth.size:
                        # We have more data, end the line with a space and comma
                        self._ss.write(' ,\n')
                    else:
                        # End the namelist
                        self._ss.write(' /\n\n\n\n')

    def _write_nonlinear_interactions_snl3_namelist(self):
        """Writes out the nonlinear interactions SNL3 namelist.  Also writes out ANL3 namelist if available."""
        # $   Gen. Multiple DIA   : Namelist SNL3
        # $                           NQDEF  : Number of quadruplets.
        # $                           MSC    : Scaling constant 'm'.
        # $                           NSC    : Scaling constant 'N'.
        # $                           KDFD   : Deep water relative filter depth,
        # $                           KDFS   : Shallow water relative filter depth,
        # $                         Namelist ANL3
        # $                           QPARMS : 5 x NQDEF paramaters describing the
        # $                                    quadruplets, repeating LAMBDA, MU, DT12.
        # $                                    Cdeep and Cshal. See examples below.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SNL3')
        self._write_nonlinear_interactions_anl3_namelist()

    def _write_nonlinear_interactions_anl3_namelist(self):
        """Writes out the special nonlinear interactions ANL3 namelist."""
        # $                         Namelist ANL3
        # $                           QPARMS : 5 x NQDEF paramaters describing the
        # $                                    quadruplets, repeating LAMBDA, MU, DT12.
        # $                                    Cdeep and Cshal. See examples below.
        anl3 = self._xms_data.sim_data.anl3qparms
        if anl3 is not None:
            if 'LAMBDA' in anl3 and anl3.LAMBDA.size > 0:
                # Write the namelist only if data is available
                self._ss.write('&ANL3 QPARAMS = ')
                for i in range(anl3.LAMBDA.size):
                    if i != 0:
                        # Fill in spaces on any additional lines after the first
                        self._ss.write('                ')
                    self._ss.write(
                        f'{float(anl3.LAMBDA[i])}, {float(anl3.MU[i])}, {float(anl3.DT12[i])}, '
                        f'{float(anl3.Cdeep[i])}, {float(anl3.Cshal[i])}'
                    )
                    if (i + 1) < anl3.LAMBDA.size:
                        # We have more data, end the line with a space and comma
                        self._ss.write(' ,\n')
                    else:
                        # End the namelist
                        self._ss.write(' /\n\n\n\n')

    def _write_nonlinear_interactions_snl4_namelist(self):
        """Writes out the nonlinear interactions SNL4 namelist."""
        # $   Two Scale Approx.   : Namelist SNL4
        # $                           INDTSA : Index for TSA/FBI computations
        # $                                    (0 = FBI ; 1 = TSA)
        # $                           ALTLP  : Index for alternate looping
        # $                                    (1 = no ; 2 = yes)
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SNL4', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SNL4\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SNL4ALTLP':
                # Special case for ALTLP, which is 1 (off) or 2 (on), not 0 or 1
                self._ss.write(f"    {var} = {attrs[key]}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_nonlinear_filter_data(self):
        """Writes out the various nonlinear filter based on DIA namelists."""
        self._write_nonlinear_filter_snls_namelist()

    def _write_nonlinear_filter_snls_namelist(self):
        """Writes out the nonlinear filter based on DIA SNLS namelist."""
        # $ Nonlinear filter based on DIA  - - - - - - - - - - - - - - - - - - -
        # $                         Namelist SNLS
        # $                           A34    : Relative offset in quadruplet
        # $                           FHFC   : Proportionality constants.
        # $                           DMN    : Maximum relative change.
        # $                           FC1-3  : Constants in frequency filter.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SNLS')

    def _write_whitecapping_dissipation_data(self):
        """Writes out the various nonlinear filter based on DIA namelists."""
        consts = self._xms_data.sim_data_model_control.group('consts')
        if consts.parameter('whitecapping_dissipation').value == 0:
            self._write_whitecapping_dissipation_sds1_namelist()
        elif consts.parameter('whitecapping_dissipation').value == 1:
            self._write_whitecapping_dissipation_sds2_namelist()
        elif consts.parameter('whitecapping_dissipation').value == 2:
            self._write_whitecapping_dissipation_sds3_namelist()
        elif consts.parameter('whitecapping_dissipation').value == 3:
            self._write_whitecapping_dissipation_sds6_namelist()
        elif consts.parameter('whitecapping_dissipation').value == 4:
            self._write_whitecapping_dissipation_swl6_namelist()

    def _write_whitecapping_dissipation_sds1_namelist(self):
        """Writes out the whitecapping disipation SDS1 namelist."""
        # $   WAM-3               : Namelist SDS1
        # $                           CDIS, APM : As in source term.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SDS1')

    def _write_whitecapping_dissipation_sds2_namelist(self):
        """Writes out the whitecapping disipation SDS2 namelist."""
        # $   Tolman and Chalikov : Namelist SDS2
        # $                           SDSA0, SDSA1, SDSA2, SDSB0, SDSB1, PHIMIN :
        # $                                    Constants a0, a1, a2, b0, b1 and
        # $                                    PHImin.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SDS2')

    def _write_whitecapping_dissipation_sds3_namelist(self):
        """Writes out the whitecapping disipation SDS3 namelist."""
        # $   WAM4 and variants   : Namelist SDS3
        # $                           SDSC1    : WAM4 Cds coeffient
        # $                           MNMEANP, WNMEANPTAIL : power of wavenumber
        # $                                    for mean definitions in Sds and tail
        # $                           SDSDELTA1, SDSDELTA2 : relative weights
        # $                                    of k and k^2 parts of WAM4 dissipation
        # $                           SDSLF, SDSHF : coefficient for activation of
        # $                              WAM4 dissipation for unsaturated (SDSLF) and
        # $                               saturated (SDSHF) parts of the spectrum
        # $                           SDSC2    : Saturation dissipation coefficient
        # $                           SDSC4    : Value of B0=B/Br for wich Sds is zero
        # $                           SDSBR    : Threshold Br for saturation
        # $                           SDSP     : power of (B/Br-B0) in Sds
        # $                           SDSBR2   : Threshold Br2 for the separation of
        # $                             WAM4 dissipation in saturated and non-saturated
        # $                           SDSC5 : coefficient for turbulence dissipation
        # $                           SDSC6 : Weight for the istropic part of Sds_SAT
        # $                           SDSDTH: Angular half-width for integration of B
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SDS3')

    def _write_whitecapping_dissipation_sds6_namelist(self):
        """Writes out the whitecapping disipation SDS6 namelist."""
        # $   BYDRZ               : Namelist SDS6
        # $                          SDSET    : Select threshold normalization spectra
        # $                          SDSA1, SDSA2, SDSP1, SDSP2  :
        # $                               Coefficients for dissipation terms T1 and T2
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SDS6')

    def _write_whitecapping_dissipation_swl6_namelist(self):
        """Writes out the whitecapping disipation SWL6 namelist."""
        # $                       : Namelist SWL6
        # $                          SWLB1    : Coefficient for swell dissipation
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SWL6')

    def _write_bottom_friction_data(self):
        """Writes out the various bottom friction namelists."""
        self._write_bottom_friction_sbt1_namelist()

    def _write_bottom_friction_sbt1_namelist(self):
        """Writes out the bottom friction SBT1 namelist."""
        # $   JONSWAP             : Namelist SBT1
        # $                           GAMMA   : Bottom friction emprical constant
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='SBT1')

    def _write_surf_breaking_data(self):
        """Writes out the various surf breaking namelists."""
        self._write_surf_breaking_sdb1_namelist()

    def _write_surf_breaking_sdb1_namelist(self):
        """Writes out the surf breaking SDB1 namelist."""
        # $   Battjes and Janssen : Namelist SDB1
        # $                           BJALFA  : Dissipation constant (default = 1)
        # $                           BJGAM   : Breaking threshold (default = 0.73)
        # $                           BJFLAG  : TRUE  - Use Hmax/d ratio only (default)
        # $                                     FALSE - Use Hmax/d in Miche formulation
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SDB1', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SDB1\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'BJFLAG':
                # Special case for BJFLAG, which is TRUE then FALSE on the combo box
                val = ".true." if attrs[key] == 0 else ".false."
                self._ss.write(f"    {var} = {val}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_dissipation_in_the_ice_data(self):
        """Writes out the various dissipation in the ice namelists."""
        self._write_dissipation_in_the_ice_sic2_namelist()

    def _write_dissipation_in_the_ice_sic2_namelist(self):
        """Writes out the dissipation in the ice SIC2 namelist."""
        # $   Generalization of Liu et al. : Namelist SIC2
        # $                           IC2DISPER  : If true uses Liu formulation with eddy viscosity
        # $                                        If false, uses the generalization with turbulent
        # $                                        to laminar transition
        # $                           IC2TURB    : empirical factor for the turbulent part
        # $                           IC2ROUGH   : under-ice roughness length
        # $                           IC2REYNOLDS: Re number for laminar to turbulent transition
        # $                           IC2SMOOTH  : smoothing of transition reprensenting random waves
        # $                           IC2VISC    : empirical factor for viscous part
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SIC2', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SIC2\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SIC2IC2DISPER':
                # Special case for IC2DISPER, which is TRUE then FALSE on the combo box
                val = ".true." if attrs[key] == 0 else ".false."
                self._ss.write(f"    {var} = {val}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_scattering_in_the_ice_data(self):
        """Writes out the various scattering in the ice & creep dissipations namelists."""
        self._write_scattering_in_the_ice_sis2_namelist()

    def _write_scattering_in_the_ice_sis2_namelist(self):
        """Writes out the scattering in the ice SIS2 namelist."""
        # $   Generalization of Wiliams et al. : Namelist SIS2
        # $                           ISC1          : scattering coefficient (default = 1)
        # $                           IS2BACKSCAT   : fraction of energy back-scattered (default = 1 )
        # $                           IS2BREAK      : TRUE  - changes floe max diameter
        # $                                         : FALSE - does not change floe max diameter
        # $                           IS2C1         : scattering in pack ice
        # $                           IS2C2         : frequency dependance of scattering in pack ice
        # $                           IS2C3         : frequency dependance of scattering in pack ice
        # $                           ISBACKSCAT    : fraction of scattered energy actualy redistributed
        # $                           IS2DISP       : use of ice-specific dispersion relation (T/F)
        # $                           FRAGILITY     : parameter between 0 and 1 that gives the shape of FSD
        # $                           IS2DMIN       : minimum floe diameter in meters
        # $                           IS2DAMP       : multiplicative coefficient for dissipation term from RP
        # $                           IS2UPDATE     : TRUE  - updates the max floe diameter with forcing only
        # $                                         : FALSE - updates the max floe diameter at each time step
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SIS2', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SIS2\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SIS2':
                # Special case for IS2UPDATE, which is TRUE then FALSE on the combo box
                val = ".true." if attrs[key] == 0 else ".false."
                self._ss.write(f"    {var} = {val}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_triad_nonlinear_interactions_data(self):
        """Writes out the various triad nonlinear interactions namelists."""
        self._write_triad_nonlinear_interactions_str1_namelist()

    def _write_triad_nonlinear_interactions_str1_namelist(self):
        """Writes out the triad nonlinear interactions STR1 namelist."""
        # $   Lumped Triad Interaction (LTA) : Namelist STR1 (To be implemented)
        # $                           PTRIAD1 : Proportionality coefficient (default 0.05)
        # $                           PTRIAD2 : Multiple of Tm01 up to which interaction
        # $                                     is computed (2.5)
        # $                           PTRIAD3 : Ursell upper limit for computing
        # $                                     interactions (not used, default 10.)
        # $                           PTRIAD4 : Shape parameter for biphase
        # $                                     computation (0.2)
        # $                           PTRIAD5 : Ursell number treshold for computing
        # $                                     interactions (0.01)
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='STR1')

    def _write_shoreline_reflections_data(self):
        """Writes out the various shoreline reflections namelists."""
        self._write_shoreline_reflections_ref1_namelist()

    def _write_shoreline_reflections_ref1_namelist(self):
        """Writes out the shoreline reflections REF1 namelist."""
        # $   ref. parameters       : Namelist REF1
        # $                           REFCOAST  : Reflection coefficient at shoreline
        # $                           REFFREQ   : Activation of freq-dependent ref.
        # $                           REFMAP    : Scale factor for bottom slope map
        # $                           REFRMAX   : maximum ref. coeffient (default 0.8)
        # $                           REFFREQPOW: power of frequency
        # $                           REFICEBERG: Reflection coefficient for icebergs
        # $                           REFSUBGRID: Reflection coefficient for islands
        # $                           REFCOSP_STRAIGHT: power of cosine used for
        # $                                       straight shoreline
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='REF1')

    def _write_bound_second_order_spectrum_data(self):
        """Writes out the various bound 2nd order spectrum and free IG namelists."""
        self._write_bound_second_order_spectrumd_sig1_namelist()

    def _write_bound_second_order_spectrumd_sig1_namelist(self):
        """Writes out the various bound 2nd order spectrum and free IG SIG1 namelist."""
        # $   IG1 parameters       : Namelist SIG1
        # $                           IGMETHOD  : 1: Hasselmann, 2: Krasitskii-Janssen
        # $                           IGADDOUTP : activation of bound wave correction
        # $                                       in ww3_outp / ww3_ounp
        # $                           IGSOURCE  : 1: uses bound waves, 2: empirical
        # $                           IGSTERMS  :  > 0 : no source term in IG band
        # $                           IGMAXFREQ : maximum frequency of IG band
        # $                           IGEMPIRICAL: constant in empirical free IG source
        # $                           IGBCOVERWRITE: T: Replaces IG spectrum, does not add
        # $                           IGSWELLMAX: T: activates free IG sources for all freq.
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='SIG1', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&SIG1\n')
        for key, var in zip(diff_keys, diff_vars):
            if key == 'SIG1IGMETHOD':
                # Special case for IGMETHOD, which is 1 (Hasselmann) or 2 (Krasitskii-Janssen), not 0 or 1
                self._ss.write(f"    {var} = {attrs[key] + 1}\n")
            elif key == 'SIG1IGSOURCE':
                # Special case for IGSOURCE, which is 1 (bound waves) or 2 (emperical), not 0 or 1
                self._ss.write(f"    {var} = {attrs[key] + 1}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_propagation_schemes_data(self):
        """Writes out the various propagation schemes namelists."""
        consts = self._xms_data.sim_data_model_control.group('consts')
        if consts.parameter('propagation_schemes').value == 0:
            self._write_propagation_schemes_pro1_namelist()
        elif consts.parameter('propagation_schemes').value == 1:
            self._write_propagation_schemes_pro2_namelist()
        else:
            self._write_propagation_schemes_pro3_namelist()

    def _write_propagation_schemes_pro1_namelist(self):
        """Writes out the propagation schemes PRO1 namelist."""
        # $   First order         : Namelist PRO1
        # $                           CFLTM  : Maximum CFL number for refraction.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='PRO1')

    def _write_propagation_schemes_pro2_namelist(self):
        """Writes out the propagation schemes PRO2 namelist."""
        # $   UQ/UNO with diffusion : Namelist PRO2
        # $                           CFLTM  : Maximum CFL number for refraction.
        # $                           DTIME  : Swell age (s) in garden sprinkler
        # $                                    correction. If 0., all diffusion
        # $                                    switched off. If small non-zero
        # $                                    (DEFAULT !!!) only wave growth
        # $                                    diffusion.
        # $                           LATMIN : Maximum latitude used in calc. of
        # $                                    strength of diffusion for prop
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='PRO2')

    def _write_propagation_schemes_pro3_namelist(self):
        """Writes out the propagation schemes PRO2 namelist."""
        # $   UQ/UNO with averaging : Namelist PRO3
        # $                           CFLTM  : Maximum CFL number for refraction.
        # $                           WDTHCG : Tuning factor propag. direction.
        # $                           WDTHTH : Tuning factor normal direction.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='PRO3')

    def _write_unstructured_grids_data(self):
        """Writes out the various unstructured grids namelists."""
        grids = self._xms_data.sim_data_model_control.group('grids')
        if grids.parameter('UNST').value == 'UNSPT parameters [UNST]':
            self._write_unstructured_grids_unst_namelist()
        else:
            self._write_unstructured_grids_psmc_namelist()

    def _write_unstructured_grids_unst_namelist(self):
        """Writes the UNST unstructured grids namelist."""
        # $ Unstructured grids ------------------------------------------------ $
        # $   UNST parameters       : Namelist UNST
        # $                           UGOBCAUTO : TRUE: OBC points are taken from  type 15 elements
        # $                                       FALSE: OBC points must be listed in ww3_grid.nml
        # $                           UGOBCDEPTH: Threshold ( < 0) depth for OBC points if UGOBCAUTO is TRUE
        # $                           EXPFSN    : Activation of N scheme
        # $                           EXPFSPSI  : Activation of PSI scheme
        # $                           EXPFSFCT  : Activation of FCT scheme
        # $                           IMPFSN    : Activation of N implicit scheme
        # $                           IMPTOTAL  : Activation of fully implicit scheme | Non splitting
        # $                           EXPTOTAL  : Turn on implicit refraction (only with imptotal)
        # $                           IMPREFRACTION  : Turn on implicit freq. shift (only with imptotal)
        # $                           IMPFREQSHIFT  : Turn on implicit freq. shift terms (only with imptotal)
        # $                           IMPSOURCE  : Turn on implicit source terms (only with imptotal)
        # $                           JGS_TERMINATE_MAXITER  : max. Number of iterations
        # $                           JGS_TERMINATE_DIFFERENCE  : terminate based on the total change of wave action
        # $                           JGS_TERMINATE_NORM  : terminate based on the norm of the solution
        # $                           JGS_USE_JACOBI  : Use Jacobi solver for imptotal
        # $                           JGS_BLOCK_GAUSS_SEIDEL  : Use Block Gauss Seidel method for imptotal
        # $                           JGS_MAXITER  : max. Number of solver iterations
        # $                           JGS_PMIN  : % of grid points that do not need to converge during solver iteration.
        # $                           JGS_DIFF_THR  : implicit solver threshold for JGS_TERMINATE_DIFFERENCE
        # $                           JGS_NORM_THR  : terminate based on the norm of the solution
        # $                           SETUP_APPLY_WLV  : Compute wave setup (experimental)
        # $                           SOLVERTHR_SETUP  : Solver threshold for setup computations
        # $                           CRIT_DEP_SETUP  : Critical depths for setup computations
        attrs = self._parameter_names
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name='UNST', attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write('&UNST\n')
        for key, var in zip(diff_keys, diff_vars):
            if key in [
                'UGOBCAUTO',
                'UNSTEXPFSN',
                'UNSTEXPFSPSI',
                'UNSTEXPFSFCT',
                'UNSTIMPFSN',
                'IMPTOTAL',
                'UNSTEXPTOTAL',
                'UNSTIMPREFRACTION',
                'UNSTIMPFREQSHIFT',
                'UNSTIMPSOURCE',
                'UNSTSETUP_APPLY_WLV',
                'JGS_TERMINATE_MAXITER',
                'JGS_TERMINATE_DIFFERENCE',
                'JGS_TERMINATE_NORM',
                'UNSTJGS_BLOCK_GAUSS_SEIDEL',
                'UNSTJGS_USE_JACOBI',
            ]:
                # Special boolean fields -- need to write out as .true. or .false., not 1 or 0
                if key == 'UGOBCAUTO':
                    val = ".true." if attrs[key] == 'Taken From Type 15 Elements' else ".false."
                    self._ss.write(f"    {var} = {val}\n")
                else:
                    val = ".true." if attrs[key] == 1 else ".false."
                    self._ss.write(f"    {var} = {val}\n")
            else:
                self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _write_unstructured_grids_psmc_namelist(self):
        """Writes out the PSMC namelist for unstructured grids."""
        # $ SMC grid propagation    : Namelist PSMC and default values
        # $                           CFLTM  : Maximum CFL no. for propagation, 0.7
        # $                           DTIME  : Swell age for diffusion term (s), 0.0
        # $                           LATMIN : Maximum latitude (deg) for GCT,   86.0
        # $                           RFMAXD : Maximum refraction turning (deg), 80.0
        # $                           LvSMC  : No. of refinement level, default 1
        # $                           ISHFT  : Shift number of i-index, default 0
        # $                           JEQT   : Shift number of j-index, default 0
        # $                           NBISMC : No. of input boundary points,    0
        # $                           UNO3   : Use 3rd order advection scheme, .FALSE.
        # $                           AVERG  : Add extra spatial averaging,    .FALSE.
        # $                           SEAWND : Use sea-point only wind input.  .FALSE.
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='PSMC')

    def _write_output_data(self):
        """Writes out the output namelists."""
        self._write_output_outs_namelist()

    def _write_output_outs_namelist(self):
        """Writes out the output OUTS namelist."""
        # $ Parameters (integers)   : Namelist OUTS
        # $ For the frequency spectrum E(f)
        # $                          E3D     : <=0: not declared, > 0: declared
        # $                          I1E3D   : First frequency index of output (default is 1)
        # $                          I2E3D   : Last frequency index of output  (default is NK)
        # $ For the mean direction th1m(f), and spread sth1m(f)
        # $                   TH1MF, STH1MF  : <=0: not declared, > 0: declared
        # $                 I1TH1MF, I1STH1MF: First frequency index of output (default is 1)
        # $                 I2TH1MF, I2STH1MF: First frequency index of output (default is 1)
        # $ For the mean direction th2m(f), and spread sth2m(f)
        # $                   TH2MF, STH2MF  : <=0: not declared, > 0: declared
        # $                 I1TH2MF, I1STH2MF: First frequency index of output (default is 1)
        # $                 I2TH2MF, I2STH2MF: First frequency index of output (default is 1)
        # $ For 2nd order pressure at K=0 (source of microseisms & microbaroms)
        # $                           P2SF   : <=0: not declared, > 0: declared
        # $                           I1P2SF : First frequency index of output (default is 1)
        # $                           I2P2SF : Last frequency index of output  (default is NK)
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='OUTS')

    def _write_miscellaneous_data(self):
        """Writes out the namelists for the miscellaneous data."""
        self._write_miscellaneous_misc_namelist()

    def _write_miscellaneous_misc_namelist(self):
        """Writes the MISC namelist."""
        # $ Miscellaneous ------------------------------------------------------ $
        # $   Misc. parameters    : Namelist MISC
        # $                           CICE0  : Ice concentration cut-off.
        # $                           CICEN  : Ice concentration cut-off.
        # $                           PMOVE  : Power p in GSE aleviation for
        # $                                    moving grids in Eq. (D.4).
        # $                           XSEED  : Xseed in seeding alg. (!/SEED).
        # $                           FLAGTR : Indicating presence and type of
        # $                                    subgrid information :
        # $                                     0 : No subgrid information.
        # $                                     1 : Transparancies at cell boun-
        # $                                         daries between grid points.
        # $                                     2 : Transp. at cell centers.
        # $                                     3 : Like 1 with cont. ice.
        # $                                     4 : Like 2 with cont. ice.
        # $                           TRCKCMPR : Logical variable (T/F). Set to F to
        # $                                      disable "compression" of track output.
        # $                                      This simplifies post-processing.
        # $                                      Default is T and will create track
        # $                                      output in the traditional manner
        # $                                      (WW3 v3, v4, v5).
        # $                           XP, XR, XFILT
        # $                                    Xp, Xr and Xf for the dynamic
        # $                                    integration scheme.
        # $                           IHMAX  : Number of discrete levels in part.
        # $                           HSPMIN : Minimum Hs in partitioning.
        # $                           WSM    : Wind speed multiplier in part.
        # $                           WSC    : Cut of wind sea fraction for
        # $                                    identifying wind sea in part.
        # $                           FLC    : Flag for combining wind seas in
        # $                                    partitioning.
        # $                           NOSW   : Number of partitioned swell fields
        # $                                    in field output.
        # $                           PTM    : Partioning method:
        # $                                     1 : Default WW3
        # $                                     2 : Watershedding + wind cutoff
        # $                                     3 : Watershedding only
        # $                                     4 : Wind speed cutoff only
        # $                                     5 : High/Low band cutoff (see PTFC)
        # $                           PTFC   : Cutouf frequency for High/Low band
        # $                                    partioning (PTM=5). Default = 0.1Hz
        # $                           FMICHE : Constant in Miche limiter.
        # $                           STDX   : Space-Time Extremes X-Length
        # $                           STDY   : Space-Time Extremes Y-Length
        # $                           STDT   : Space-Time Extremes Duration
        # $                           P2SF   : ......
        # $                           CALTYPE: Calendar type. The only accepted
        # $                                    values are 'standard' (default),
        # $                                    '365_day', or '360_day'.
        # Note:  missing  TRCKCMPR  PTM  PTFC
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='MISC')

    def _write_sea_state_stress_data(self):
        """Writes out the various sea state stress namelists."""
        sea_state_stress = self._xms_data.sim_data_model_control.group('sea_state_stress')
        if sea_state_stress.parameter('FLDMethod').value == 0:
            self._write_sea_state_stress_fld1_namelist()
        elif sea_state_stress.parameter('FLDMethod').value == 1:
            self._write_sea_state_stress_fld2_namelist()

    def _write_sea_state_stress_fld1_namelist(self):
        """Writes out the sea state stress FLD1 namelist."""
        # $   Reichl et al. 2014  : Namelist FLD1
        # $                           TAILTYPE  : High Frequency Tail Method
        # $                                       0: Constant value (prescribed)
        # $                                       1: Wind speed dependent
        # $                                          (Based on GFDL Hurricane
        # $                                          Model Z0 relationship)
        # $                           TAILLEV   : Level of high frequency tail
        # $                                       (if TAILTYPE==0)
        # $                                       Valid choices:
        # $                                       Capped min: 0.001, max: 0.02
        # $                           TAILT1    : Tail transition ratio 1
        # $                                       TAILT1*peak input frequency
        # $                                       is the first transition point of
        # $                                       the saturation specturm
        # $                                       Default is 1.25
        # $                           TAILT1    : Tail transition ratio 2
        # $                                       TAILT2*peak input frequency
        # $                                       is the second transition point of
        # $                                       the saturation specturm
        # $                                       Default is 3.00
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='FLD1')

    def _write_sea_state_stress_fld2_namelist(self):
        """Writes out the sea state stress FLD2 namelist."""
        # $   Donelan et al. 2012 : Namelist FLD2
        # $                           TAILTYPE : See above (FLD1)
        # $                           TAILLEV  : See above (FLD1)
        # $                           TAILT1   : See above (FLD1)
        # $                           TAILT2   : See above (FLD1)
        self._write_simple_namelist(attrs=self._parameter_names, namelist_name='FLD2')

    def _write_simple_namelist(self, attrs, namelist_name):
        """Writes out a namelist for non-default values, where no special editing or handling is required.

        Use this where you simply want to write out all the non-default values.
        Do not use if you need to change combo box values to a different string, ignore certain values based on the
        value of another value, etc.

        Args:
            attrs (:obj:`xarray.Dataset`):  The xarray.Dataset containing the data.
            namelist_name (:obj:`str`):  The namelist name.
        """
        diff_keys, diff_vars = self._defaults.check_namelist_defaults(namelist_name=namelist_name, attrs=attrs)
        if len(diff_keys) == 0:
            return
        self._ss.write(f'&{namelist_name}\n')
        for key, var in zip(diff_keys, diff_vars):
            self._ss.write(f"    {var} = {attrs[key]}\n")
        self._ss.write('/\n\n\n')

    def _flush(self, file_w_path):
        """Writes the StringIO previously processed to a file.

        Args:
            file_w_path (:obj:`str`):  String of the filename to write to.
        """
        f = open(file_w_path, 'w')
        self._ss.seek(0)
        shutil.copyfileobj(self._ss, f, 100000)
        f.close()

    def write(self):
        """Top-level entry point for the WaveWatch3 namelists file writer."""
        self._write_namelists_nml_file()
