"""Reader for WaveWatch3 solution output files."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import uuid

# 2. Third party modules
import netCDF4
import numpy as np

# 3. Aquaveo modules
from xms.data_objects.parameters import Dataset, datetime_to_julian
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules

WAVEWATCH3_NULL_VALUE = -99999.0


class NetcdfDatasetReader:
    """The Wavewatch3 NetCDF reader."""
    def __init__(
        self, filename, reftime, query, temp_dir, output_types, geom_uuid='CCCCCCCC-CCCC-CCCC-CCCC-CCCCCCCCCCCC'
    ):
        """Construct the reader.

        Args:
            filename (:obj:`str`): The filename (with path) to read.
            reftime (:obj:`datetime.datetime`): Simulation start time.
            query (:obj:`Query`): Object for communicating with XMS.
            temp_dir (:obj:`str`): Path to the XMS temp directory. This is where we will write the converted
                XMDF solution files.
            output_types (:obj:`list`): List of output types to read, ex:  DPT, CUR, WND, etc.
            geom_uuid (:obj:`str`): UUID of the solution datasets' geometry, if known. Otherwise, XMS will try to
                match based on number of values but can go to the wrong geometry if it matches.
        """
        self._logger = logging.getLogger('xms.wavewatch3')
        self.query = query
        self.temp_dir = temp_dir
        self.output_types = [output_type.lower() for output_type in output_types]  # Store as lower case
        self.reftime = datetime_to_julian(reftime)  # Used for reference time on transient datasets, convert to Julian.
        self.geom_uuid = geom_uuid  # Used to link datasets to a known geometry. Otherwise, SMS will look for a match.
        self.filename = filename  # file to read
        self.num_ts = 0
        self.num_nodes = 0
        self.read_file = ""
        self.out_filename = None  # For testing
        self.dset_uuid = None  # For testing
        self.overwrite_files = True  # For testing

    def _get_next_h5_filename_and_uuid(self):
        """Either get a random UUID and filename of an H5 file to write from the Query or hardcoded list if testing.

        Returns:
            (:obj:`tuple(str,str)`): Filesystem path to use to write an H5 file, UUID for the dataset
        """
        if self.out_filename and self.dset_uuid:
            return self.out_filename, self.dset_uuid
        else:  # pragma: no cover
            dset_uuid = str(uuid.uuid4())
            return os.path.join(self.temp_dir, f'{dset_uuid}.h5'), dset_uuid

    def _write_xmdf_dataset(self, dset_name, times, data, num_components, activity=None):
        """Write a solution dataset to an XMDF formatted file that XMS can read.

        Args:
            dset_name (:obj:`str`): Tree item name of the dataset to create in SMS.
            times (:obj:`Sequence`): 1-D array of float time step offsets
            data (:obj:`Sequence`): The dataset values organized in XMDF structure. Rows are timesteps and columns are
                node/cell values. If a vector dataset, outer dimensions contain the additional components.
            num_components (:obj:`int`): 1 = scalar, 2 = 2D vector
            activity (Sequence, optional): The activity mask for the dataset.
        """
        xmdf_filename, dset_uuid = self._get_next_h5_filename_and_uuid()
        writer = DatasetWriter(
            h5_filename=xmdf_filename,
            name=dset_name,
            dset_uuid=dset_uuid,
            geom_uuid=self.geom_uuid,
            null_value=WAVEWATCH3_NULL_VALUE,
            ref_time=self.reftime,
            time_units='Days',
            num_components=num_components,
            overwrite=self.overwrite_files
        )
        writer.write_xmdf_dataset(times, data, activity=activity)
        self.add_xmdf_dataset(xmdf_filename, dset_name)
        return dset_uuid

    def read_netcdf_scalars(self, filename, scalar_path, dset_name):
        """Read a scalar solution dataset from a NetCDF formatted file.

        Args:
            filename (:obj:`str`): Filesystem path to the NetCDF solution file.
            scalar_path (:obj:`str`): Path in the NetCDF file to the solution dataset
            dset_name (:obj:`str`): Tree item name of the dataset to create in SMS.
        """
        root_grp = netCDF4.Dataset(filename, "r", format="NETCDF4_CLASSIC")
        if scalar_path.lstrip('/') not in root_grp.variables:
            # Make sure the variable exists before trying to read it
            return False
        scalar_data = root_grp[scalar_path][:]
        if scalar_data.size == 0:
            self._logger.warning(
                f'Empty solution data set encountered: {os.path.basename(filename) if self.out_filename else filename}'
            )
            return False  # No data to write in the file

        # Replace -99999.0 null values with NaN for numpy operations.
        scalar_data[scalar_data == WAVEWATCH3_NULL_VALUE] = np.nan

        # Convert numpy NaNs back to null value for XMDF file.
        scalar_data[np.isnan(scalar_data)] = WAVEWATCH3_NULL_VALUE

        # Write the XMDF file
        dataset_uuid = self._write_xmdf_dataset(dset_name, root_grp["/time"][:], scalar_data, 1)
        return dataset_uuid

    def read_netcdf_vectors(self, filename, x_path, y_path, dset_name):
        """Read a vector solution dataset from a NetCDF formatted file.

        Args:
            filename (str): Filesystem path to the NetCDF solution file.
            x_path (str): Path in the NetCDF file to the x-component solution dataset
            y_path (str): Path in the NetCDF file to the y-component solution dataset
            dset_name (str): Tree item name of the dataset to create in SMS.
        """
        # Read the x and y component values from the NetCDF solution file.
        root_grp = netCDF4.Dataset(filename, "r", format="NETCDF4_CLASSIC")
        x_data = root_grp[f"{x_path}"][:]
        y_data = root_grp[f"{y_path}"][:]

        if x_data.size == 0:
            self._logger.warning(
                'Empty solution data set encountered: '
                f'{os.path.basename(filename) if self.out_filename else filename}'
            )
            return  # No data to write in the file
        # Replace -99999.0 and 0.0 null values with NaN for numpy operations.
        x_data[x_data == WAVEWATCH3_NULL_VALUE] = np.nan
        y_data[y_data == WAVEWATCH3_NULL_VALUE] = np.nan

        # Create an XMDF-style data cube of the vector components. Shape=(num_times, num_vals, 2)
        vector_data = np.stack([x_data, y_data], axis=2)

        # Convert numpy NaNs back to null value for XMDF file.
        vector_data[np.isnan(vector_data)] = WAVEWATCH3_NULL_VALUE

        # Write the XMDF file
        dataset_uuid = self._write_xmdf_dataset(dset_name, root_grp['/time'][:], vector_data, 2)
        return dataset_uuid

    def add_xmdf_dataset(self, filename, dset_name):
        """Add an existing XMDF dataset file to the Query to read into XMS.

        Args:
            filename (:obj:`str`): Filepath to the XMDF dataset
            dset_name (:obj:`str`): Name of the dataset. Used to build the HDF5 path to the dataset.
        """
        dset = Dataset(filename, f'Datasets/{dset_name}', 'NODE', 'NODE')
        if self.query:
            self.query.add_dataset(dset)

    def get_readers(self, output_type):
        """Get the appropriate reader method based on filename since WaveWatch3 uses hard-coded filenames.

        Args:
            output_type (:obj:`str`): The dataset to read.

        Returns:
            (:obj:`list`):  List of readers for the output type (some vector datasets are split into 2 scalars).
        """
        if not os.path.isfile(self.read_file):
            return None

        filename = os.path.basename(self.read_file).lower()

        if 'ww3.' in filename and '.nc' in filename:
            # Make sure the output type passed in is valid, then grab the associated reader
            if output_type in self.output_types:
                # Forcing Fields:
                if output_type.lower() == 'dpt':
                    return [self.read_netcdf_dpt]
                elif output_type.lower() == 'cur':
                    # Vector or mag/dir format, two components each type
                    return [
                        self.read_netcdf_cur_ucir, self.read_netcdf_cur_vcir, self.read_netcdf_cur_cspd,
                        self.read_netcdf_cur_cdir
                    ]
                elif output_type.lower() == 'wnd':
                    # Vector or mag/dir format, two components each type
                    return [
                        self.read_netcdf_wnd_uwnd, self.read_netcdf_wnd_vwnd, self.read_netcdf_wnd_wdir,
                        self.read_netcdf_wnd_wspd
                    ]
                elif output_type.lower() == 'ast':
                    return [self.read_netcdf_ast]
                elif output_type.lower() == 'wlv':
                    return [self.read_netcdf_wlv]
                elif output_type.lower() == 'ice':
                    return [self.read_netcdf_ice]
                elif output_type.lower() == 'ibg':
                    return [self.read_netcdf_ibg]
                elif output_type.lower() == 'd50':
                    return [self.read_netcdf_d50]
                elif output_type.lower() == 'ic1':
                    return [self.read_netcdf_ic1]
                elif output_type.lower() == 'ic5':
                    return [self.read_netcdf_ic5]
                # Standard mean wave Parameters
                elif output_type.lower() == 'hs':
                    return [self.read_netcdf_hs]
                elif output_type.lower() == 'lm':
                    return [self.read_netcdf_lm]
                elif output_type.lower() == 't02':
                    return [self.read_netcdf_t02]
                elif output_type.lower() == 't0m1':
                    return [self.read_netcdf_t0m1]
                elif output_type.lower() == 't01':
                    return [self.read_netcdf_t01]
                elif output_type.lower() == 'fp':
                    return [self.read_netcdf_fp]
                elif output_type.lower() == 'dir':
                    return [self.read_netcdf_dir]
                elif output_type.lower() == 'spr':
                    return [self.read_netcdf_spr]
                elif output_type.lower() == 'dp':
                    return [self.read_netcdf_dp]
                elif output_type.lower() == 'hig':
                    return [self.read_netcdf_hig]
                elif output_type.lower() == 'mxe':
                    return [self.read_netcdf_mxe]
                elif output_type.lower() == 'mxes':
                    return [self.read_netcdf_mxes]
                elif output_type.lower() == 'mxh':
                    return [self.read_netcdf_mxh]
                elif output_type.lower() == 'mxhc':
                    return [self.read_netcdf_mxhc]
                elif output_type.lower() == 'sdmh':
                    return [self.read_netcdf_sdmh]
                elif output_type.lower() == 'sdmhc':
                    return [self.read_netcdf_sdmhc]
                elif output_type.lower() == 'wbt':
                    return [self.read_netcdf_wbt]
                elif output_type.lower() == 'tp':
                    return [self.read_netcdf_tp]
                # Spectral Parameters (first 5)
                elif output_type.lower() == 'ef':
                    return [self.read_netcdf_ef]
                elif output_type.lower() == 'th1m':
                    return [self.read_netcdf_th1m]
                elif output_type.lower() == 'sth1m':
                    return [self.read_netcdf_sth1m]
                elif output_type.lower() == 'th2m':
                    return [self.read_netcdf_th2m]
                elif output_type.lower() == 'sth2m':
                    return [self.read_netcdf_sth2m]
                elif output_type.lower() == 'wn':
                    return [self.read_netcdf_wn]
                # Spectral Partition Parameters
                elif output_type.lower() == 'phs':
                    return [self.read_netcdf_phs]
                elif output_type.lower() == 'ptp':
                    return [self.read_netcdf_ptp]
                elif output_type.lower() == 'plp':
                    return [self.read_netcdf_plp]
                elif output_type.lower() == 'pdir':
                    return [self.read_netcdf_pdir]
                elif output_type.lower() == 'pspr':
                    return [self.read_netcdf_pspr]
                elif output_type.lower() == 'pws':
                    return [self.read_netcdf_pws]
                elif output_type.lower() == 'pdp':
                    return [self.read_netcdf_pdp]
                elif output_type.lower() == 'pqp':
                    return [self.read_netcdf_pqp]
                elif output_type.lower() == 'ppe':
                    return [self.read_netcdf_ppe]
                elif output_type.lower() == 'pgw':
                    return [self.read_netcdf_pgw]
                elif output_type.lower() == 'psw':
                    return [self.read_netcdf_psw]
                elif output_type.lower() == 'ptm10':
                    return [self.read_netcdf_ptm10]
                elif output_type.lower() == 'pt01':
                    return [self.read_netcdf_pt01]
                elif output_type.lower() == 'pt02':
                    return [self.read_netcdf_pt02]
                elif output_type.lower() == 'pep':
                    return [self.read_netcdf_pep]
                elif output_type.lower() == 'tws':
                    return [self.read_netcdf_tws]
                elif output_type.lower() == 'pnr':
                    return [self.read_netcdf_pnr]
                # Atmosphere-waves layer
                elif output_type.lower() == 'ust':
                    # Two components
                    return [self.read_netcdf_ust_uust, self.read_netcdf_ust_vust]
                elif output_type.lower() == 'cha':
                    return [self.read_netcdf_cha]
                elif output_type.lower() == 'cge':
                    return [self.read_netcdf_cge]
                elif output_type.lower() == 'faw':
                    return [self.read_netcdf_faw]
                elif output_type.lower() == 'taw':
                    # Two components
                    return [self.read_netcdf_taw_utaw, self.read_netcdf_taw_vtaw]
                elif output_type.lower() == 'twa':
                    # Two components
                    return [self.read_netcdf_twa_utaw, self.read_netcdf_twa_vtaw]
                elif output_type.lower() == 'wcc':
                    return [self.read_netcdf_wcc]
                elif output_type.lower() == 'wcf':
                    return [self.read_netcdf_wcf]
                elif output_type.lower() == 'wch':
                    return [self.read_netcdf_wch]
                elif output_type.lower() == 'wcm':
                    return [self.read_netcdf_wcm]
                elif output_type.lower() == 'fws':
                    return [self.read_netcdf_fws]
                # Wave-ocean layer
                elif output_type.lower() == 'sxy':
                    # Three components
                    return [self.read_netcdf_sxy_sxx, self.read_netcdf_sxy_syy, self.read_netcdf_sxy_sxy]
                elif output_type.lower() == 'two':
                    return [self.read_netcdf_two_utwo, self.read_netcdf_two_vtwo]
                elif output_type.lower() == 'bhd':
                    return [self.read_netcdf_bhd]
                elif output_type.lower() == 'foc':
                    return [self.read_netcdf_foc]
                elif output_type.lower() == 'tus':
                    # Two components
                    return [self.read_netcdf_tus_utus, self.read_netcdf_tus_vtus]
                elif output_type.lower() == 'uss':
                    # Two components
                    return [self.read_netcdf_uss_uuss, self.read_netcdf_uss_vuss]
                elif output_type.lower() == 'p2s':
                    # Two components
                    return [self.read_netcdf_p2s_fp2s, self.read_netcdf_p2s_pp2s]
                elif output_type.lower() == 'usf':
                    # Two components
                    return [self.read_netcdf_usf_uusf, self.read_netcdf_usf_vusf]
                elif output_type.lower() == 'p2l':
                    return [self.read_netcdf_p2l]
                elif output_type.lower() == 'twi':
                    # Two components
                    return [self.read_netcdf_twi_utic, self.read_netcdf_twi_vtic]
                elif output_type.lower() == 'fic':
                    return [self.read_netcdf_fic]
                elif output_type.lower() == 'usp':
                    # Two components
                    return [self.read_netcdf_usp_ussp, self.read_netcdf_usp_vssp]
                # Wave-bottom layer
                elif output_type.lower() == 'abr':
                    # Two components
                    return [self.read_netcdf_abr_uabr, self.read_netcdf_abr_vabr]
                elif output_type.lower() == 'ubr':
                    # Two components
                    return [self.read_netcdf_ubr_uubr, self.read_netcdf_ubr_vubr]
                elif output_type.lower() == 'bed':
                    # Three components
                    return [self.read_netcdf_bed_bed, self.read_netcdf_bed_ripplex, self.read_netcdf_bed_rippley]
                elif output_type.lower() == 'fbb':
                    return [self.read_netcdf_fbb]
                elif output_type.lower() == 'tbb':
                    # Two components
                    return [self.read_netcdf_tbb_utbb, self.read_netcdf_tbb_vtbb]
                # Spectrum parameters
                elif output_type.lower() == 'mss':
                    # Two components
                    return [self.read_netcdf_mss_mssu, self.read_netcdf_mss_mssc]
                elif output_type.lower() == 'msc':
                    # Two components
                    return [self.read_netcdf_msc_mscx, self.read_netcdf_msc_mscy]
                elif output_type.lower() == 'wl02':
                    return [self.read_netcdf_wl02]
                elif output_type.lower() == 'axt':
                    return [self.read_netcdf_axt]
                elif output_type.lower() == 'ayt':
                    return [self.read_netcdf_ayt]
                elif output_type.lower() == 'axy':
                    return [self.read_netcdf_axy]
                # Numerical diagnostics
                elif output_type.lower() == 'dtd':
                    return [self.read_netcdf_dtd]
                elif output_type.lower() == 'fc':
                    return [self.read_netcdf_fc]
                elif output_type.lower() == 'cfx':
                    return [self.read_netcdf_cfx]
                elif output_type.lower() == 'cfd':
                    return [self.read_netcdf_cfd]
                elif output_type.lower() == 'cfk':
                    return [self.read_netcdf_cfk]
                # User defined
                elif output_type.lower() == 'u1':
                    return [self.read_netcdf_u1]
                elif output_type.lower() == 'u2':
                    return [self.read_netcdf_u2]

        return None

    def read_netcdf_dpt(self):
        """Reader for NetCDF dpt variable."""
        self.read_netcdf_scalars(self.read_file, '/dpt', 'DPT')

    def read_netcdf_cur_ucir(self):
        """Reader for NetCDF cur variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/ucir', 'UCIR')

    def read_netcdf_cur_vcir(self):
        """Reader for NetCDF cur variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/vcir', 'VCIR')

    def read_netcdf_cur_cspd(self):
        """Reader for NetCDF cur variable (split up for mag/dir)."""
        self.read_netcdf_scalars(self.read_file, '/cspd', 'CSPD')

    def read_netcdf_cur_cdir(self):
        """Reader for NetCDF cur variable (split up for mag/dir)."""
        self.read_netcdf_scalars(self.read_file, '/cdir', 'CDIR')

    def read_netcdf_wnd_uwnd(self):
        """Reader for NetCDF wnd variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/uwnd', 'UWND')

    def read_netcdf_wnd_vwnd(self):
        """Reader for NetCDF wnd variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/vwnd', 'VWND')

    def read_netcdf_wnd_wdir(self):
        """Reader for NetCDF wnd variable (split up for mag/dir)."""
        self.read_netcdf_scalars(self.read_file, '/wdir', 'WDIR')

    def read_netcdf_wnd_wspd(self):
        """Reader for NetCDF wnd variable (split up for mag/dir)."""
        self.read_netcdf_scalars(self.read_file, '/wspd', 'WSPD')

    def read_netcdf_ast(self):
        """Reader for NetCDF ast variable."""
        self.read_netcdf_scalars(self.read_file, '/ast', 'AST')

    def read_netcdf_wlv(self):
        """Reader for NetCDF wlv variable."""
        self.read_netcdf_scalars(self.read_file, '/wlv', 'WLV')

    def read_netcdf_ice(self):
        """Reader for NetCDF ice variable."""
        self.read_netcdf_scalars(self.read_file, '/ice', 'ICE')

    def read_netcdf_ibg(self):
        """Reader for NetCDF ibg variable."""
        self.read_netcdf_scalars(self.read_file, '/ibg', 'IBG')

    def read_netcdf_d50(self):
        """Reader for NetCDF d50 variable."""
        self.read_netcdf_scalars(self.read_file, '/d50', 'D50')

    def read_netcdf_ic1(self):
        """Reader for NetCDF ic1 variable."""
        self.read_netcdf_scalars(self.read_file, '/ic1', 'ICE')

    def read_netcdf_ic5(self):
        """Reader for NetCDF ic5 variable."""
        self.read_netcdf_scalars(self.read_file, '/ic5', 'IC5')

    def read_netcdf_hs(self):
        """Reader for NetCDF hs variable."""
        self.read_netcdf_scalars(self.read_file, '/hs', 'HS')

    def read_netcdf_lm(self):
        """Reader for NetCDF lm variable."""
        self.read_netcdf_scalars(self.read_file, '/lm', 'LM')

    def read_netcdf_t02(self):
        """Reader for NetCDF t02 variable."""
        self.read_netcdf_scalars(self.read_file, '/t02', 'T02')

    def read_netcdf_t0m1(self):
        """Reader for NetCDF t0m1 variable."""
        self.read_netcdf_scalars(self.read_file, '/t0m1', 'T0M1')

    def read_netcdf_t01(self):
        """Reader for NetCDF t01 variable."""
        self.read_netcdf_scalars(self.read_file, '/t01', 'T01')

    def read_netcdf_fp(self):
        """Reader for NetCDF fp variable."""
        self.read_netcdf_scalars(self.read_file, '/fp', 'FP')

    def read_netcdf_dir(self):
        """Reader for NetCDF dir variable."""
        self.read_netcdf_scalars(self.read_file, '/dir', 'DIR')

    def read_netcdf_spr(self):
        """Reader for NetCDF spr variable."""
        self.read_netcdf_scalars(self.read_file, '/spr', 'SPR')

    def read_netcdf_dp(self):
        """Reader for NetCDF dp variable."""
        self.read_netcdf_scalars(self.read_file, '/dp', 'DP')

    def read_netcdf_hig(self):
        """Reader for NetCDF hig variable."""
        self.read_netcdf_scalars(self.read_file, '/hig', 'HIG')

    def read_netcdf_mxe(self):
        """Reader for NetCDF mxe variable."""
        self.read_netcdf_scalars(self.read_file, '/mxe', 'MXE')

    def read_netcdf_mxes(self):
        """Reader for NetCDF mxes variable."""
        self.read_netcdf_scalars(self.read_file, '/mxes', 'MXES')

    def read_netcdf_mxh(self):
        """Reader for NetCDF mxh variable."""
        self.read_netcdf_scalars(self.read_file, '/mxh', 'MXH')

    def read_netcdf_mxhc(self):
        """Reader for NetCDF mxhc variable."""
        self.read_netcdf_scalars(self.read_file, '/mxhc', 'MXHC')

    def read_netcdf_sdmh(self):
        """Reader for NetCDF sdmh variable."""
        self.read_netcdf_scalars(self.read_file, '/sdmh', 'SDMH')

    def read_netcdf_sdmhc(self):
        """Reader for NetCDF sdmhc variable."""
        self.read_netcdf_scalars(self.read_file, '/sdmhc', 'SDMHC')

    def read_netcdf_wbt(self):
        """Reader for NetCDF wbt variable."""
        self.read_netcdf_scalars(self.read_file, '/wbt', 'WBT')

    def read_netcdf_tp(self):
        """Reader for NetCDF tp variable."""
        self.read_netcdf_scalars(self.read_file, '/tp', 'TP')

    def read_netcdf_ef(self):
        """Reader for NetCDF ef variable."""
        self.read_netcdf_scalars(self.read_file, '/ef', 'EF')

    def read_netcdf_th1m(self):
        """Reader for NetCDF th1m variable."""
        self.read_netcdf_scalars(self.read_file, '/th1m', 'TH1M')

    def read_netcdf_sth1m(self):
        """Reader for NetCDF sth1m variable."""
        self.read_netcdf_scalars(self.read_file, '/sth1m', 'STH1M')

    def read_netcdf_th2m(self):
        """Reader for NetCDF th2m variable."""
        self.read_netcdf_scalars(self.read_file, '/th2m', 'TH2M')

    def read_netcdf_sth2m(self):
        """Reader for NetCDF sth2m variable."""
        self.read_netcdf_scalars(self.read_file, '/sth2m', 'STH2M')

    def read_netcdf_wn(self):
        """Reader for NetCDF wn variable."""
        self.read_netcdf_scalars(self.read_file, '/wn', 'WN')

    def read_netcdf_phs(self):
        """Reader for NetCDF phs variable."""
        self.read_netcdf_scalars(self.read_file, '/phs', 'PHS')

    def read_netcdf_ptp(self):
        """Reader for NetCDF ptp variable."""
        self.read_netcdf_scalars(self.read_file, '/ptp', 'PTP')

    def read_netcdf_plp(self):
        """Reader for NetCDF plp variable."""
        self.read_netcdf_scalars(self.read_file, '/plp', 'PLP')

    def read_netcdf_pdir(self):
        """Reader for NetCDF pdir variable."""
        self.read_netcdf_scalars(self.read_file, '/pdir', 'PDIR')

    def read_netcdf_pspr(self):
        """Reader for NetCDF pspr variable."""
        self.read_netcdf_scalars(self.read_file, '/pspr', 'PSPR')

    def read_netcdf_pws(self):
        """Reader for NetCDF pws variable."""
        self.read_netcdf_scalars(self.read_file, '/pws', 'PWS')

    def read_netcdf_pdp(self):
        """Reader for NetCDF pdp variable."""
        self.read_netcdf_scalars(self.read_file, '/pdp', 'PDP')

    def read_netcdf_pqp(self):
        """Reader for NetCDF pqp variable."""
        self.read_netcdf_scalars(self.read_file, '/pqp', 'PQP')

    def read_netcdf_ppe(self):
        """Reader for NetCDF ppe variable."""
        self.read_netcdf_scalars(self.read_file, '/ppe', 'PPE')

    def read_netcdf_pgw(self):
        """Reader for NetCDF pgw variable."""
        self.read_netcdf_scalars(self.read_file, '/pgw', 'PGW')

    def read_netcdf_psw(self):
        """Reader for NetCDF psw variable."""
        self.read_netcdf_scalars(self.read_file, '/psw', 'PSW')

    def read_netcdf_ptm10(self):
        """Reader for NetCDF ptm10 variable."""
        self.read_netcdf_scalars(self.read_file, '/ptm10', 'PTM10')

    def read_netcdf_pt01(self):
        """Reader for NetCDF pt01 variable."""
        self.read_netcdf_scalars(self.read_file, '/pt01', 'PT01')

    def read_netcdf_pt02(self):
        """Reader for NetCDF pt02 variable."""
        self.read_netcdf_scalars(self.read_file, '/pt02', 'PT02')

    def read_netcdf_pep(self):
        """Reader for NetCDF pep variable."""
        self.read_netcdf_scalars(self.read_file, '/pep', 'PEP')

    def read_netcdf_tws(self):
        """Reader for NetCDF tws variable."""
        self.read_netcdf_scalars(self.read_file, '/tws', 'TWS')

    def read_netcdf_pnr(self):
        """Reader for NetCDF pnr variable."""
        self.read_netcdf_scalars(self.read_file, '/pnr', 'PNR')

    def read_netcdf_ust_uust(self):
        """Reader for NetCDF ust variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/uust', 'UUST')

    def read_netcdf_ust_vust(self):
        """Reader for NetCDF ust variable (split up for vector)."""
        self.read_netcdf_scalars(self.read_file, '/vust', 'VUST')

    def read_netcdf_cha(self):
        """Reader for NetCDF cha variable."""
        self.read_netcdf_scalars(self.read_file, '/cha', 'CHA')

    def read_netcdf_cge(self):
        """Reader for NetCDF cge variable."""
        self.read_netcdf_scalars(self.read_file, '/cge', 'CGE')

    def read_netcdf_faw(self):
        """Reader for NetCDF faw variable."""
        self.read_netcdf_scalars(self.read_file, '/faw', 'FAW')

    def read_netcdf_taw_utaw(self):
        """Reader for NetCDF taw variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/utaw', 'UTAW')

    def read_netcdf_taw_vtaw(self):
        """Reader for NetCDF taw variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vtaw', 'VTAW')

    def read_netcdf_twa_utaw(self):
        """Reader for NetCDF twa variable (two componenets)."""
        self.read_netcdf_scalars(self.read_file, '/utwa', 'UTWA')

    def read_netcdf_twa_vtaw(self):
        """Reader for NetCDF twa variable (two componenets)."""
        self.read_netcdf_scalars(self.read_file, '/vtwa', 'VTWA')

    def read_netcdf_wcc(self):
        """Reader for NetCDF wcc variable."""
        self.read_netcdf_scalars(self.read_file, '/wcc', 'WCC')

    def read_netcdf_wcf(self):
        """Reader for NetCDF wcf variable."""
        self.read_netcdf_scalars(self.read_file, '/wcf', 'WCF')

    def read_netcdf_wch(self):
        """Reader for NetCDF wch variable."""
        self.read_netcdf_scalars(self.read_file, '/wch', 'WCH')

    def read_netcdf_wcm(self):
        """Reader for NetCDF wcm variable."""
        self.read_netcdf_scalars(self.read_file, '/wcm', 'WCM')

    def read_netcdf_fws(self):
        """Reader for NetCDF fws variable."""
        self.read_netcdf_scalars(self.read_file, '/fws', 'FWS')

    def read_netcdf_sxy_sxx(self):
        """Reader for NetCDF sxy variable (three components)."""
        self.read_netcdf_scalars(self.read_file, '/sxx', 'Sxx')

    def read_netcdf_sxy_syy(self):
        """Reader for NetCDF sxy variable (three components)."""
        self.read_netcdf_scalars(self.read_file, '/syy', 'Syy')

    def read_netcdf_sxy_sxy(self):
        """Reader for NetCDF sxy variable (three components)."""
        self.read_netcdf_scalars(self.read_file, '/sxy', 'Sxy')

    def read_netcdf_two_utwo(self):
        """Reader for NetCDF two variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/utwo', 'UTWO')

    def read_netcdf_two_vtwo(self):
        """Reader for NetCDF two variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vtwo', 'VTWO')

    def read_netcdf_bhd(self):
        """Reader for NetCDF bhd variable."""
        self.read_netcdf_scalars(self.read_file, '/bhd', 'BHD')

    def read_netcdf_foc(self):
        """Reader for NetCDF foc variable."""
        self.read_netcdf_scalars(self.read_file, '/foc', 'FOC')

    def read_netcdf_tus_utus(self):
        """Reader for NetCDF tus variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/utus', 'UTUS')

    def read_netcdf_tus_vtus(self):
        """Reader for NetCDF tus variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vtus', 'VTUS')

    def read_netcdf_uss_uuss(self):
        """Reader for NetCDF uss variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/uuss', 'UUSS')

    def read_netcdf_uss_vuss(self):
        """Reader for NetCDF uss variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vuss', 'VUSS')

    def read_netcdf_p2s_fp2s(self):
        """Reader for NetCDF p2s variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/fp2s', 'FP2S')

    def read_netcdf_p2s_pp2s(self):
        """Reader for NetCDF p2s variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/pp2s', 'PP2S')

    def read_netcdf_usf_uusf(self):
        """Reader for NetCDF usf variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/uusf', 'UUSF')

    def read_netcdf_usf_vusf(self):
        """Reader for NetCDF usf variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vusf', 'VUSF')

    def read_netcdf_p2l(self):
        """Reader for NetCDF p2l variable."""
        self.read_netcdf_scalars(self.read_file, '/p2l', 'P2L')

    def read_netcdf_twi_utic(self):
        """Reader for NetCDF twi variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/utic', 'UTIC')

    def read_netcdf_twi_vtic(self):
        """Reader for NetCDF twi variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vtic', 'VTIC')

    def read_netcdf_fic(self):
        """Reader for NetCDF fic variable."""
        self.read_netcdf_scalars(self.read_file, '/fic', 'FIC')

    def read_netcdf_usp_ussp(self):
        """Reader for NetCDF usp variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/ussp', 'USSP')

    def read_netcdf_usp_vssp(self):
        """Reader for NetCDF usp variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vssp', 'VSSP')

    def read_netcdf_abr_uabr(self):
        """Reader for NetCDF abr variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/uabr', 'UABR')

    def read_netcdf_abr_vabr(self):
        """Reader for NetCDF abr variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vabr', 'VABR')

    def read_netcdf_ubr_uubr(self):
        """Reader for NetCDF ubr variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/uubr', 'UUBR')

    def read_netcdf_ubr_vubr(self):
        """Reader for NetCDF ubr variable (two components)."""
        self.read_netcdf_scalars(self.read_file, '/vubr', 'VUBR')

    def read_netcdf_bed_bed(self):
        """Reader for NetCDF bed variable."""
        self.read_netcdf_scalars(self.read_file, '/bed', 'BED')

    def read_netcdf_bed_ripplex(self):
        """Reader for NetCDF bed variable."""
        self.read_netcdf_scalars(self.read_file, '/ripplex', 'RIPPLEX')

    def read_netcdf_bed_rippley(self):
        """Reader for NetCDF bed variable."""
        self.read_netcdf_scalars(self.read_file, '/rippley', 'RIPPLEY')

    def read_netcdf_fbb(self):
        """Reader for NetCDF fbb variable."""
        self.read_netcdf_scalars(self.read_file, '/fbb', 'FBB')

    def read_netcdf_tbb_utbb(self):
        """Reader for NetCDF tbb variable."""
        self.read_netcdf_scalars(self.read_file, '/utbb', 'UTBB')

    def read_netcdf_tbb_vtbb(self):
        """Reader for NetCDF tbb variable."""
        self.read_netcdf_scalars(self.read_file, '/vtbb', 'VTBB')

    def read_netcdf_mss_mssu(self):
        """Reader for NetCDF mss variable."""
        self.read_netcdf_scalars(self.read_file, '/mssu', 'MSSU')

    def read_netcdf_mss_mssc(self):
        """Reader for NetCDF mss variable."""
        self.read_netcdf_scalars(self.read_file, '/mssc', 'MSSC')

    def read_netcdf_msc_mscx(self):
        """Reader for NetCDF msc variable."""
        self.read_netcdf_scalars(self.read_file, '/mscx', 'MSCX')

    def read_netcdf_msc_mscy(self):
        """Reader for NetCDF msc variable."""
        self.read_netcdf_scalars(self.read_file, '/mscy', 'MSCY')

    def read_netcdf_wl02(self):
        """Reader for NetCDF wl02 variable."""
        self.read_netcdf_scalars(self.read_file, '/wl02', 'WL02')

    def read_netcdf_axt(self):
        """Reader for NetCDF axt variable."""
        self.read_netcdf_scalars(self.read_file, '/axt', 'AXT')

    def read_netcdf_ayt(self):
        """Reader for NetCDF ayt variable."""
        self.read_netcdf_scalars(self.read_file, '/ayt', 'AYT')

    def read_netcdf_axy(self):
        """Reader for NetCDF axy variable."""
        self.read_netcdf_scalars(self.read_file, '/axy', 'AXY')

    def read_netcdf_dtd(self):
        """Reader for NetCDF dtd variable."""
        self.read_netcdf_scalars(self.read_file, '/dtd', 'DTD')

    def read_netcdf_fc(self):
        """Reader for NetCDF fc variable."""
        self.read_netcdf_scalars(self.read_file, '/fc', 'FC')

    def read_netcdf_cfx(self):
        """Reader for NetCDF cfx variable."""
        self.read_netcdf_scalars(self.read_file, '/cfx', 'CFX')

    def read_netcdf_cfd(self):
        """Reader for NetCDF cfd variable."""
        self.read_netcdf_scalars(self.read_file, '/cfd', 'CFD')

    def read_netcdf_cfk(self):
        """Reader for NetCDF cfk variable."""
        self.read_netcdf_scalars(self.read_file, '/cfk', 'CFK')

    def read_netcdf_u1(self):
        """Reader for NetCDF u1 variable."""
        self.read_netcdf_scalars(self.read_file, '/u1', 'U1')

    def read_netcdf_u2(self):
        """Reader for NetCDF u2 variable."""
        self.read_netcdf_scalars(self.read_file, '/u2', 'U2')

    def get_num_nodes(self):
        """Gets the number of values in the nodes dataset."""
        root_grp = netCDF4.Dataset(self.read_file, "r", format="NETCDF4_CLASSIC")
        try:
            if 'MAPSTA' not in root_grp.variables:
                return 0
            scalar_data = root_grp['MAPSTA'][:]
        except IndexError:  # pragma: no cover - hard to test
            self._logger.warning('Empty solution data set encountered for node dataset.')
            return 0
        return scalar_data.size

    def add_mesh_datasets(self):
        """Read all found solution files."""
        # binary formats not supported (format=2)
        self.read_file = self.filename
        for output_type in self.output_types:
            readers = self.get_readers(output_type)
            for reader in readers:
                reader()

    def read(self):
        """Start the solution reader."""
        self.add_mesh_datasets()
