"""Class to read a WaveWatch3 ounf (grid output post-processing) namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import shlex
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.wavewatch3.data.output_fields import get_output_field_model
from xms.wavewatch3.file_io.io_util import READ_BUFFER_SIZE
from xms.wavewatch3.gui.gui_util import get_date_time_string


def _get_true_false_as_int(value):
    """Gets the various formats of true/false in as an integer.

    Args:
        value (:obj:`str`):  The string from the file.

    Returns:
        (:obj:`int`):  1 if some form of true, 0 if false
    """
    if value.upper() == '.FALSE.' or value.upper() == 'F' or value.upper() == "'F'" or value == '0':
        return 0
    elif value.upper() == '.TRUE.' or value.upper() == 'T' or value.upper() == "'T'" or value == '1':
        return 1
    else:
        raise ValueError(f"Unable to determine value of {value}.")


class OunfNmlReader:
    """Class to read a WaveWatch3 ounf nml file."""
    def __init__(self, filename='', sim_data=None):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the nml file. If not provided (not testing or control file read),
                will retrieve from Query.
            sim_data(:obj:`xms.wavewatch3.data.SimData`):  The simulation data to edit.
        """
        self._filename = filename
        self._query = None
        self._sim_data = sim_data
        self._sim_comp_uuid = str(uuid.uuid4())
        self._setup_query()
        self._lines = []
        self._current_line = 0
        self._logger = logging.getLogger('xms.wavewatch3')
        self._output_fields_section = get_output_field_model().model_parameters
        self._output_fields_section.restore_values(self._sim_data.output_field_values)

    def _setup_query(self):
        """Setup the xmsapi Query for sending data to SMS and get the import filename."""
        if not self._filename:  # pragma: no cover - slow to setup Query for the filename
            self._query = Query()
            self._filename = self._query.read_file

    def _parse_next_line(self, shell=False):
        """Parse the next line of text from the file.

        Skips empty and comment lines.

        Args:
            shell (:obj:`bool`): If True will parse line using shlex. Slower but convenient for quoted tokens.

        Returns:
            (:obj:`list[str]`): The next line of text, split on whitespace
        """
        line = None
        while not line or line.startswith('!'):  # blank lines and control file identifier
            if self._current_line >= len(self._lines):
                # raise RuntimeError('Unexpected end of file.')
                return None
            line = self._lines[self._current_line].strip()
            self._current_line += 1
        if shell:
            return shlex.split(line, posix=False)
        return line.split()

    def _get_namelist_cards_and_values(self, firstline_data):
        """Reads the entire namelist until the closing / is found.  Stores cards and values.

        Handles cases where the data is on multiple lines, or on a single line.

        Args:
            firstline_data (:obj:`list[str]`):  List of data parsed on the opening line.

        Returns:
            (:obj:`dict`):  Dictionary with the keys being the cards read, and values of corresponding data.
        """
        namelist_list = firstline_data
        if '/' not in namelist_list:
            # We haven't found the end of the namelist yet.  Read more lines as necessary.
            end_not_found = True
            while end_not_found:
                # Grab the next line of data
                data = self._parse_next_line()
                # Extend the list of all data by the current line read
                namelist_list.extend(data)
                # Check if we've got the end of the namelist yet
                if '/' in data:
                    end_not_found = False

        # Now, we have the entire namelist as a list of values.  Parse into cards/commands and values
        cards = []
        values = []
        for i in range(len(namelist_list)):
            # Get the list elements on either side of each =
            if namelist_list[i] == '=':
                if 0 < i < len(namelist_list):
                    cards.append(namelist_list[i - 1].rstrip(','))
                    str = namelist_list[i + 1].rstrip(',')
                    if str[0] == "'" and str[-1] != "'":
                        # We have a string to read... get it until we find a closing single quote
                        # This could be something like a date:  '20150227 000000'
                        closing_found = False
                        while not closing_found:
                            i += 1
                            str += " " + namelist_list[i + 1].rstrip(',')
                            if str[-1] == "'":
                                closing_found = True
                    values.append(str)

        # Return the cards and values found throughout the namelist read
        return cards, values

    def _read_ounf_nml_file(self):
        """Read the OUNF nml file."""
        reading = True
        while reading:
            data = self._parse_next_line()
            if data:
                if '&FIELD_NML' in data[0].strip():
                    self._read_field_nml_namelist(data)
                elif '&FILE_NML' in data[0].strip():
                    self._read_file_nml_namelist(data)
                elif '&SMC_NML' in data[0].strip():
                    self._read_smc_nml_namelist(data)
                else:
                    raise ValueError(f'Unrecognized namelist {data}')
            else:
                reading = False

    def _read_field_nml_namelist(self, data):
        """Read the FIELD_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        output_fields = self._output_fields_section
        user_defined = output_fields.group('user_defined')
        self._logger.info('Reading FIELD_NML namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'FIELD%TIMESTART' in cards:
            time_str = values[cards.index('FIELD%TIMESTART')]
            time_vals = time_str.split(' ')
            user_defined.parameter('field_timestart').value = get_date_time_string(time_vals[0], time_vals[1])
        if 'FIELD%TIMESTRIDE' in cards:
            val = values[cards.index('FIELD%TIMESTRIDE')].lstrip("'").rstrip("'")
            user_defined.parameter('field_timestride').value = int(val)
        if 'FIELD%TIMECOUNT' in cards:
            val = values[cards.index('FIELD%TIMECOUNT')].lstrip("'").rstrip("'")
            user_defined.parameter('field_timecount').value = int(val)
        if 'FIELD%TIMESPLIT' in cards:
            # Due to combo box order:  0, 4, 6, 8, 10
            vals = [0, 4, 6, 8, 10]
            file_val = int(values[cards.index('FIELD%TIMESPLIT')])
            val = vals.index(file_val) if file_val in vals else 2
            user_defined.parameter('field_timesplit').value = int(val)
        if 'FIELD%LIST' in cards:
            user_defined.parameter('field_list').value = values[cards.index('FIELD%LIST')].lstrip("'").rstrip("'")
        if 'FIELD%SAMEFILE' in cards:
            val = _get_true_false_as_int(values[cards.index('FIELD%SAMEFILE')])
            user_defined.parameter('field_samefile').value = val
        if 'FIELD%VECTOR' in cards:
            val = _get_true_false_as_int(values[cards.index('FIELD%VECTOR')])
            user_defined.parameter('field_vector').value = val
        if 'FIELD%TYPE' in cards:
            # Due to combo box order:  2, 3, 4
            vals = [2, 3, 4]
            file_val = int(values[cards.index('FIELD%TYPE')])
            val = vals.index(file_val) if file_val in vals else 1
            user_defined.parameter('field_type').value = int(val)

    def _read_file_nml_namelist(self, data):
        """Read the FILE_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        output_fields = self._output_fields_section
        user_defined = output_fields.group('user_defined')
        self._logger.info('Reading FILE_NML namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'FILE%PREFIX' in cards:
            user_defined.parameter('file_prefix').value = values[cards.index('FILE%PREFIX')].lstrip("'").rstrip("'")
        if 'FILE%NETCDF' in cards:
            vals = [3, 4]
            file_val = values[cards.index('FILE%NETCDF')]
            val = int(vals.index(file_val) if file_val in vals else 1)
            user_defined.parameter('file_netcdf').value = int(val)
        if 'FILE%IX0' in cards:
            user_defined.parameter('file_ix0').value = int(values[cards.index('FILE%IX0')])
        if 'FILE%IXN' in cards:
            user_defined.parameter('file_ixn').value = int(values[cards.index('FILE%IXN')])
        if 'FILE%IY0' in cards:
            user_defined.parameter('file_iy0').value = int(values[cards.index('FILE%IY0')])
        if 'FILE%IYN' in cards:
            user_defined.parameter('file_iyn').value = int(values[cards.index('FILE%IYN')])

    def _read_smc_nml_namelist(self, data):
        """Read the SMC_NML namelist.

        Args:
            data(:obj:`list[str]`):  The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading SMC_NML namelist...')
        _, _ = self._get_namelist_cards_and_values(data)

    def read(self):
        """Top-level entry point for the WaveWatch3 ounf nml input file reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()

            self._read_ounf_nml_file()
            self._logger.info('Committing changes....')
            self._sim_data.output_field_values = self._output_fields_section.extract_values()
            self._sim_data.commit()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception(
                'Unexpected error in ounf nml preprocessor file '
                f'(line {self._current_line + 1}).'
            )
            raise
