"""For testing."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import read_grid_from_file
from xms.coverage.spectral import SpectralCoverage
from xms.data_objects._data_objects.parameters import FilterLocation
from xms.mesher.meshing import mesh_utils

# 4. Local modules
from xms.wavewatch3.dmi.xms_data import XmsData
from xms.wavewatch3.file_io.spec_list_writer import WW3SpecListWriter


def calc_timestep_values(xms_data: XmsData):
    """Calculates default timestep values for the model control given a grid and first frequency value.

    Args:
        xms_data: XmsData.
    """
    cogrid = read_grid_from_file(xms_data.do_ugrid.cogrid_file)
    size_func = mesh_utils.size_function_from_edge_lengths(cogrid.ugrid)
    dxy = min(size_func)
    units = xms_data.do_ugrid.projection.horizontal_units
    dxy = dxy * 0.3048 if 'FEET' in units else dxy * 60.0 * 1852.0 if units == 'ARC_DEGREES' else dxy

    global_values = xms_data.sim_data.model_control_data
    parameters = global_values.group('parameters')
    freq1 = parameters.parameter('FREQ1').value

    freq1 = freq1 if freq1 > 0.0001 else 0.038
    tcfl = dxy / (9.8 / (freq1 * 4 * math.pi))
    dtxy = round(0.90 * tcfl)
    dtmax = round(3.0 * dtxy)
    dtkth = round(dtmax / 2.0)
    dtmin = 10 if dtmax > 10 else min([dtxy, dtmax, dtkth])

    parameters.parameter('maxGlobalDt').value = float(dtmax)
    parameters.parameter('maxCFLXY').value = float(dtxy)
    parameters.parameter('maxCFLKTheta').value = float(dtkth)
    parameters.parameter('minSourceDt').value = float(dtmin)

    xms_data.sim_data.global_values = global_values.extract_values()
    xms_data.sim_data.commit()


def populate_spectral_values(xms_data: XmsData, spectral_cov: SpectralCoverage):
    """Gets spectral directions and frequencies for the model control given a spectral coverage.

    Args:
        xms_data: XmsData.
        spectral_cov: The spectral coverage.
    """
    spectral_points = spectral_cov.m_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
    spectral_coverages = [spectral_cov]
    spec_list_writer = WW3SpecListWriter(spectral_coverages)
    spectral_directions = spec_list_writer.get_spectral_directions(spectral_cov, spectral_points[0])
    spectral_frequencies = spec_list_writer.get_spectral_frequencies(spectral_cov, spectral_points[0])

    global_values = xms_data.sim_data.model_control_data
    run_control = global_values.group('parameters')
    if len(spectral_frequencies) > 0:
        run_control.parameter('FREQ1').value = float(spectral_frequencies[0])
    run_control.parameter('NK').value = len(spectral_frequencies)
    run_control.parameter('NTH').value = len(spectral_directions)
    if len(spectral_frequencies) >= 2:
        run_control.parameter('XFR').value = float(spectral_frequencies[1] / spectral_frequencies[0])

    xms_data.sim_data.global_values = global_values.extract_values()
    xms_data.sim_data.commit()
