"""Data Storage class for persistent storage across the program."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import date
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_settings import AppSettingsData
from xms.FhwaVariable.core_data.app_data.profile import ProfileSettings
from xms.FhwaVariable.core_data.variables.variable import Variable
from xms.FhwaVariable.interface_adapters.view_model.main.support_window import SupportWindow
from xms.FhwaVariable.interface_adapters.view_model.main.window import GraphicsWindow, ImageWindow, PlotWindow, \
    TableWindow
from xms.FhwaVariable.interface_structures.graphics_view import GraphicsViewBase
from xms.FhwaVariable.interface_structures.view import ViewBase


# @singleton
class AppData():
    """A class that defines persistent storage across the program."""
    # _instance = None

    _app_initialized = False

    app_name = None
    version = None
    agency = None
    developed_by = None

    geometry_enabled = False
    coverages_enabled = False
    image_folder_name_plural = 'GIS'
    image_folder_name_singular = 'GIS'

    graphics_view = None
    view = None
    doc_windows = []
    doc_windows_layout = 'maximize all'
    support_windows = []
    support_plot_index = None
    support_wiki_index = None

    app_settings = None
    profile_settings_list = []
    selected_profile_settings = None
    profile_var = Variable('Profiles', 'list', 0, [])
    profile_icon = None
    # client_list = []?
    # personel_list = []?

    model_dict = {}
    model_project_defaults = {}

    # Functors (Keep as short as possible)
    message_box_functor = None
    select_file_functor = None
    edit_item_functor = None
    about_functor = None
    exit_functor = None

    def __init__(self, app_name: str = None, version: str = None, agency: str = None, developed_by: str = None,
                 doc_folder: Path = None, graphics_view: GraphicsViewBase = None, view: ViewBase = None):
        """Initializes the Settings CalcData.

        Args:
            app_name(str): The name of the program.
            version(float): The version of the program.
            agency(str): The agency that developed the program.
            developed_by(str): The company that developed the program.
            doc_folder(Path): The path to the documentation folder.
            graphics_view(bool): graphics_view to provide graphical support.
            view(ViewBase): The view to provide graphical support.
            graphics_view(bool): graphics_view to provide graphical support.
            view(ViewBase): The view to provide graphical support.
        """
        if not AppData._app_initialized:
            super().__init__()
            AppData._app_initialized = True

            if app_name is None or version is None or agency is None or developed_by is None:
                raise ValueError("AppData must be initialized with name, version, agency, and developed_by.")

            AppData.app_name = app_name
            AppData.version = version
            AppData.agency = agency
            AppData.developed_by = developed_by

            AppData.app_settings = AppSettingsData(app_name, version, agency, developed_by, doc_folder)
            # AppData.profile_settings_list = [ProfileSettings(app_name, version, agency, developed_by)]
            profile = self.add_profile()
            profile.read_only = True
            profile.set_name('FHWA Default Profile (read only)')
            profile.uuid = uuid.UUID('a6e25b6c-8c7b-481c-b534-e2e36d12f075')
            profile.set_version(1.0)
            profile.set_modification_date(date(2025, 1, 22))
            AppData.selected_profile_settings = 0
            self.update_profile_names()

            AppData.model_dict = {}
            AppData.model_project_defaults = {}

            # Plug-in the view and graphics view
            AppData.graphics_view = graphics_view
            AppData.view = view

            # Plug-in the view and graphics view
            AppData.graphics_view = graphics_view
            AppData.view = view

            # These windows will be saved into the file and loaded with the file.
            AppData.doc_windows = {}
            # If we have a GraphicsView plugged in, we will create a GraphicsView window
            if graphics_view is not None:
                self.create_document_window(window_type='graphics')

            # These windows will be saved into the registry and re-opened automatically
            AppData.support_windows = []
            num_windows = AppData.app_settings.retrieve_persistent_app_data('number_of_support_windows')
            if num_windows is not None:
                if not isinstance(num_windows, int):
                    try:
                        num_windows = int(num_windows)
                    except ValueError:
                        num_windows = 0

                for _ in range(num_windows):
                    self.create_support_window()

    @staticmethod
    def create_document_window(window_type='graphics'):
        """Create a document window."""
        index = len(AppData.doc_windows)
        if window_type == 'graphics':
            window = GraphicsWindow()
            window.window_title = f'view {index + 1}'
            window.icon = 'icon.png'
            if index == 0:
                window.is_active = True
        elif window_type == 'image':
            window = ImageWindow(window_index=index, app_title=AppData.app_name, icon='icon.png')
        elif window_type == 'table':
            window = TableWindow(window_index=index, app_title=AppData.app_name, icon='icon.png')
        elif window_type == 'plot':
            window = PlotWindow(window_index=index, app_title=AppData.app_name, icon='icon.png')
        else:
            return False
        window.window_index = index
        AppData.doc_windows[window.window_uuid] = window

    @staticmethod
    def close_document_window(window_uuid=None):
        """Close a document window."""
        if window_uuid is None:
            for window in AppData.doc_windows.values():
                if isinstance(window, GraphicsWindow):
                    if window.is_active:
                        window_uuid = window.window_uuid
                        break
        if window_uuid in AppData.doc_windows:
            AppData.doc_windows.pop(window_uuid)
            return True
        return False

    @staticmethod
    def get_document_window(window_uuid=None):
        """Get a document window."""
        if window_uuid in AppData.doc_windows:
            return AppData.doc_windows[window_uuid]
        if window_uuid is None:
            for window in AppData.doc_windows.values():
                if isinstance(window, GraphicsWindow):
                    if window.is_active:
                        return window
        if len(AppData.doc_windows) > 0:
            return list(AppData.doc_windows.values())[0]
        return None

    @staticmethod
    def create_support_window():
        """Create a support window."""
        index = len(AppData.support_windows)
        window = SupportWindow(window_index=index, app_title=AppData.app_name, icon='icon.png')

        if index == 0:
            window.show_dialog_plots = False
            if AppData.support_wiki_index is None:
                window.use_only_this_wiki = True
            else:
                window.use_only_this_wiki = False
            window.show_wiki = True
        else:
            if AppData.support_plot_index is None:
                window.show_dialog_plots = True
            else:
                window.show_dialog_plots = False
            window.show_wiki = False

        show_dialog_plots = AppData.app_settings.retrieve_persistent_app_data('show_dialog_plots', index)
        if show_dialog_plots is not None:
            if show_dialog_plots.lower() == 'true':
                show_dialog_plots = True
            elif show_dialog_plots.lower() == 'false':
                show_dialog_plots = False
            else:
                try:
                    show_dialog_plots = bool(show_dialog_plots)
                except ValueError:
                    show_dialog_plots = window.show_dialog_plots
            window.show_dialog_plots = show_dialog_plots
        else:
            AppData.app_settings.save_persistant_app_data('show_dialog_plots', index, window.show_dialog_plots)

        plot_dock_area = AppData.app_settings.retrieve_persistent_app_data('plot_dock_area', index)
        if plot_dock_area is not None:
            window.plot_dock_area = plot_dock_area
        else:
            AppData.app_settings.save_persistant_app_data('plot_dock_area', index, window.plot_dock_area)

        show_wiki = AppData.app_settings.retrieve_persistent_app_data('show_wiki', index)
        if show_wiki is not None:
            if show_wiki.lower() == 'true':
                show_wiki = True
            elif show_wiki.lower() == 'false':
                show_wiki = False
            else:
                try:
                    show_wiki = bool(show_wiki)
                except ValueError:
                    show_wiki = window.show_wiki
            window.show_wiki = show_wiki
        else:
            AppData.app_settings.save_persistant_app_data('show_wiki', index, window.show_wiki)

        wiki_dock_area = AppData.app_settings.retrieve_persistent_app_data('wiki_dock_area', index)
        if wiki_dock_area is not None:
            window.wiki_dock_area = wiki_dock_area
        else:
            AppData.app_settings.save_persistant_app_data('wiki_dock_area', index, window.wiki_dock_area)

        use_only_this_wiki = AppData.app_settings.retrieve_persistent_app_data('use_only_this_wiki', index)
        if use_only_this_wiki is not None:
            if use_only_this_wiki.lower() == 'true':
                use_only_this_wiki = True
            elif use_only_this_wiki.lower() == 'false':
                use_only_this_wiki = False
            else:
                try:
                    use_only_this_wiki = bool(use_only_this_wiki)
                except ValueError:
                    use_only_this_wiki = window.use_only_this_wiki
            window.use_only_this_wiki = use_only_this_wiki
        else:
            AppData.app_settings.save_persistant_app_data('use_only_this_wiki', index, window.use_only_this_wiki)

        if window.show_dialog_plots:
            AppData.support_plot_index = index
        else:
            AppData.app_settings.save_persistant_app_data('show_dialog_plots', index, window.show_dialog_plots)

        if window.show_wiki and window.use_only_this_wiki:
            AppData.support_wiki_index = index

        AppData.support_windows.append(window)
        AppData.app_settings.save_persistant_app_data('number_of_support_windows', len(AppData.support_windows))

    @staticmethod
    def set_support_window_show_plot(index, show_plot):
        """Get the show plot flag for a support window.

        Args:
            index(int): The index of the support window.
            show_plot(bool): The show plot flag.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows[index].show_dialog_plots = show_plot
        if show_plot:
            # If another support window has plots turned on, turn it off
            if AppData.support_plot_index is not None and AppData.support_plot_index < len(AppData.support_windows):
                AppData.support_windows[AppData.support_plot_index].show_dialog_plots = False
            AppData.support_plot_index = index
        elif AppData.support_plot_index is not None and AppData.support_plot_index == index:
            AppData.support_plot_index = None
        AppData.app_settings.save_persistant_app_data('show_plot', index, show_plot)

    # @staticmethod
    # def set_support_window_plots_dict(index, plots_dict):
    #     """Get the plot dictionary for a support window.

    #     Args:
    #         index(int): The index of the support window.
    #         plots_dict(dict): The plot dictionary.
    #     """
    #     if index < 0 or index >= len(AppData.support_windows):
    #         return False
    #     AppData.support_windows[index].plots_dict = plots_dict

    @staticmethod
    def set_support_window_show_wiki(index, show_wiki):
        """Set the show wiki flag for a support window.

        Args:
            index(int): The index of the support window.
            show_wiki(bool): The show wiki flag.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows[index].show_wiki = show_wiki
        AppData.app_settings.save_persistant_app_data('show_wiki', index, show_wiki)

    @staticmethod
    def set_support_window_use_only_this_wiki(index, use_only_this_wiki):
        """Set the use only this wiki flag for a support window.

        Args:
            index(int): The index of the support window.
            use_only_this_wiki(bool): The use only this wiki flag.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        for window in AppData.support_windows:
            window.use_only_this_wiki = False
        if use_only_this_wiki is not None:
            AppData.support_wiki_index = index
            AppData.support_windows[index].use_only_this_wiki = use_only_this_wiki
        else:
            AppData.support_wiki_index = None
        AppData.app_settings.save_persistant_app_data('use_only_this_wiki', index, use_only_this_wiki)

    @staticmethod
    def set_support_window_wiki_url(index, wiki_url):
        """Set the wiki URL for a support window.

        Args:
            index(int): The index of the support window.
            wiki_url(str): The wiki URL.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows[index].wiki_url = wiki_url

    @staticmethod
    def set_support_window_plot_dock_area(index, plot_dock_area):
        """Set the plot dock area for a support window.

        Args:
            index(int): The index of the support window.
            plot_dock_area(str): The plot dock area.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows[index].plot_dock_area = plot_dock_area
        AppData.app_settings.save_persistant_app_data('plot_dock_area', index, plot_dock_area)

    @staticmethod
    def set_support_window_wiki_dock_area(index, wiki_dock_area):
        """Set the wiki dock area for a support window.

        Args:
            index(int): The index of the support window.
            wiki_dock_area(str): The wiki dock area.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows[index].wiki_dock_area = wiki_dock_area
        AppData.app_settings.save_persistant_app_data('wiki_dock_area', index, wiki_dock_area)

    @staticmethod
    def close_support_window(index, save_number_of_windows=True):
        """Close a support window.

        Args:
            index(int): The index of the support window.
            save_number_of_windows(bool): Save the number of support windows.
        """
        if index < 0 or index >= len(AppData.support_windows):
            return False
        AppData.support_windows.pop(index)
        if AppData.support_plot_index is not None and AppData.support_plot_index == index:
            AppData.support_plot_index = None
        if AppData.support_wiki_index is not None and AppData.support_wiki_index == index:
            AppData.support_wiki_index = None
        for i, window in enumerate(AppData.support_windows):
            window.support_window_index = i
        if save_number_of_windows:
            AppData.app_settings.save_persistant_app_data('number_of_support_windows', len(AppData.support_windows))

    def update_profile_names(self):
        """Update the profile names."""
        if AppData.profile_var is not None:
            AppData.profile_var.value_options = self.get_profile_names()
            AppData.profile_var.value = AppData.selected_profile_settings

    def get_profile_names(self, include_create_new=True):
        """Get the names of the profiles.

        Args:
            include_create_new(bool): Whether to include the create new profile option.

        Returns:
            list: The names of the profiles.
        """
        profile_names = [profile.name for profile in AppData.profile_settings_list]
        if include_create_new:
            profile_names.append('<Create new profile>')
        return profile_names

    def set_profile_val(self, current_text, model_base, add_undo=True):
        """Set the profile value.

        Args:
            current_text(str): The current text of the profile.
            model_base(str): The model base.
            add_undo(bool): Whether to add an undo action.
        """
        if AppData.profile_var is not None and current_text is not None:
            if current_text == '<Create new profile>':
                profile = ProfileSettings(self.app_name, self.version, self.agency, self.developed_by)
                profile.input['Profile name'].set_val(f'New Profile {len(AppData.profile_settings_list) + 1}')
                result, new_calc = self.edit_item_functor(profile, model_name=self.app_name, model_base=model_base,
                                                          icon=self.profile_icon)
                if result and not new_calc.read_only:
                    # TODO: Check if there was actaully any changes made to the profile
                    # TODO: Immediately save profile changes to the file
                    new_calc.set_name(new_calc.input['Profile name'].get_val())
                    # AppData.profile_settings_list[AppData.selected_profile_settings] = new_calc
                    version = new_calc.input['Version'].get_val()
                    # if version <= profile.input['Version'].get_val():
                    #     version = profile.input['Version'].get_val() + 0.1
                    new_calc.set_version(version)
                    new_calc.set_modification_date()
                    self.add_profile(new_calc)
                    AppData.selected_profile_settings = len(AppData.profile_settings_list) - 1
                    model_base.read_or_save_profiles_from_file(save=True)
                return True
            elif current_text in AppData.profile_var.value_options:
                AppData.selected_profile_settings = AppData.profile_var.value_options.index(current_text)
                return True
        return False

    def add_profile(self, profile=None):
        """Add a profile to the list of profiles.

        Args:
            profile(ProfileSettings): The profile to add.
        """
        if profile is None:
            profile = ProfileSettings(self.app_name, self.version, self.agency, self.developed_by)
            profile.input['Gradation settings'].value.get_input_group()  # Initialize the default gradations
            AppData.profile_settings_list.append(profile)
        else:
            profile.set_name_ver_agency_dev(app_name=self.app_name, version=self.version, agency=self.agency,
                                            developed_by=self.developed_by)
            profile.input['Gradation settings'].value.get_input_group()  # Initialize the default gradations
            # TODO: Add a check to see if the profile already exists (by name and UUID)
            # (profiles must be unique by name and UUID)
            # If exists, prompt user to overwrite (if same name), continue import (give a new UUID), or cancel
            # (provide the user the versions and modification dates)
            AppData.profile_settings_list.append(profile)
        return profile

    def get_profile(self, index=None):
        """Get a profile from the list of profiles.

        Args:
            index(int): The index of the profile to get.

        Returns:
            ProfileSettings: The profile at the given index.
        """
        if index is None:
            index = AppData.selected_profile_settings

        if index < 0 or index >= len(AppData.profile_settings_list):
            return None

        return AppData.profile_settings_list[index]

    def delete_profile(self, index=None):
        """Delete a profile from the list of profiles.

        Args:
            index(int): The index of the profile to delete.
        """
        if index is None:
            index = AppData.selected_profile_settings

        if AppData.profile_settings_list[index].read_only:
            return False

        if index < 0 or index >= len(AppData.profile_settings_list):
            return False

        AppData.profile_settings_list.pop(index)

        if AppData.selected_profile_settings >= len(AppData.profile_settings_list):
            AppData.selected_profile_settings = len(AppData.profile_settings_list) - 1

        return True

    def add_model_interactor(self, model_interactor):
        """Add a model interactor to the list of model interactors.

        Args:
            model_name(str): The name of the model.
            model_interactor(ProfileSettings): The model interactor to add.
        """
        model_name = model_interactor.model_name
        if model_name not in AppData.model_dict:
            AppData.model_dict[model_name] = model_interactor
            AppData.model_project_defaults[model_name] = model_interactor.project_settings
            model_interactor.reset_project_settings_to_defaults()
        else:
            print("Model already exists in the model dictionary.")
            # raise ValueError("Model already exists in the model dictionary.")

    def get_model_interactor(self, model_name):
        """Get a model interactor from the list of model interactors.

        Args:
            model_name(str): The name of the model.

        Returns:
            ProfileSettings: The model interactor with the given name.
        """
        if model_name in AppData.model_dict:
            return AppData.model_dict[model_name]
        else:
            return None

    def get_default_project_settings_by_name(self, model_name):
        """Get the project settings for a model.

        Args:
            model_name(str): The name of the model.

        Returns:
            ProfileSettings: The project settings for the model.
        """
        if AppData.model_project_defaults is not None:
            if model_name in AppData.model_project_defaults:
                return AppData.model_project_defaults[model_name]
        return None

    def get_project_settings(self, model_name, project_uuid):
        """Get the project settings for a model.

        Args:
            project_uuid(UUID): The UUID of the project settings.

        Returns:
            ProfileSettings: The project settings for the model.
        """
        if model_name in AppData.model_dict:
            return AppData.model_dict[model_name].get_project_settings_by_uuid(project_uuid)
        return False, None

    def get_setting_var(self, name, model_name=None, project_uuid=None, skip_locations=None):
        """Returns the variable of a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            model (string): model name

        Returns:
            if the setting was successfully found
            var (FHWAVariable): Value of the setting
            location_found
        """
        if skip_locations is None:
            skip_locations = []
        if AppData.app_settings is not None and 'app_settings' not in skip_locations:
            result, var = AppData.app_settings.get_setting_var(name)
            if result:
                return result, var, 'app_settings'
        if AppData.selected_profile_settings is not None and AppData.profile_settings_list and \
                -1 < AppData.selected_profile_settings < len(AppData.profile_settings_list) and \
                'profile_settings' not in skip_locations:
            result, var = AppData.profile_settings_list[AppData.selected_profile_settings].get_setting_var(name)
            if result:
                return result, var, 'profile_settings'
        if model_name is not None and project_uuid is not None and 'project_settings' not in skip_locations:
            result, settings = self.get_project_settings(model_name, project_uuid)
            if result:
                result2, var = settings.get_setting_var(name)
                if result2:
                    return result2, var, 'project_settings'

        return False, None, None

    def get_setting(self, name, default=None, model_name=None, project_uuid=None):
        """Returns a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            return_in_case_of_failure (varies): if the setting isn't found, this is returned

        Returns:
            if the setting was successfully found
            value (vaires): Value of the setting or return_in_case_of_failure
        """
        result, var, _ = self.get_setting_var(name, model_name, project_uuid)
        if result:
            return result, var.get_val()
        return result, default

    def set_setting(self, name, new_value, model_name=None, project_uuid=None):
        """Changes a setting with given name or displayed name.

        Args:
            name (string): the name or displayed name of the setting
            new_value (varies): new value to set
            model (string): model name

        Returns:
            if the setting was successfully found
        """
        set_results = False
        not_complete = True
        skip_locations = []
        while not_complete:
            result, var, location = self.get_setting_var(name, model_name, project_uuid, skip_locations)
            if result:
                set_results = True
                var.set_val(new_value)
                if location == 'project_settings':
                    not_complete = False
                else:
                    skip_locations.append(location)
            else:
                not_complete = False
        return set_results

    @staticmethod
    def get_theme():
        """Gets the selected theme."""
        return AppData.app_settings.get_theme()
