"""Presenter that creates a view-model of the main interface."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.interface_adapters.view_model.main.application import Application as vm_Application
from .data_tree_presenter import DataTreePresenter
from .tools_presenter import ToolsPresenter


class AppPresenter:
    """A class to define the model-view of the main interface."""
    def __init__(self):
        """Initialize the main interface class."""
        self.application = None

        self.tools = ToolsPresenter()
        self.data_tree = DataTreePresenter()

        self.app_name = ''
        self.version = ''

        self.view = None

        # TODO: Dockable widgets like metadata, properties, or allow the project explorer to be moved?

    # View wrapper methods
    def create_application(self, model_dict, view) -> None:
        """
        Create the application view from a view model.

        Args:
            model_dict (dict): The dict of models to create the interface
            view (View): The view plug-in to create the application
        """
        start_url = None
        if len(model_dict) > 0:
            start_url = model_dict[list(model_dict.keys())[0]].wiki_url
        # We don't have a clipboard before we create the application
        self.view = view
        self.application = self.build_app_viewmodel(model_dict, None, start_url)
        self.application.doc_windows_layout = ''  # This layout won't work yet
        view.create_application(self.application)

    def update_application(self, model_dict, view, wiki_url=None) -> None:
        """
        Update the application view from a view model.

        Args:
            model_dict (dict): The dict of models to update the interface
            view (View): The view plug-in to update the application
            wiki_url (str): The wiki url to load
        """
        clipboard = view.get_clipboard()
        self.application = self.build_app_viewmodel(model_dict, clipboard, wiki_url)
        view.update_application(self.application)

    def run(self, view):
        """Run the application.

        Args:
            view (View): The view plug-in to run the application
        """
        view.run()

    def show_context_menu(self, menu_dict, pos, item_uuid, view):
        """Show the menu.

        Args:
            menu_dict (dict): The menu dictionary
            pos (tuple): The position to show the menu
            item_uuid (str): The uuid of the item
            view (View): The view plug-in to show the menu
        """
        # TODO: Create a function that will provide an update to the menu (copy, paste, etc.)
        # clipboard = view.get_clipboard()
        # menu = self.tools.create_toolbars(menu_dict, clipboard)
        view.show_context_menu(menu_dict, pos, item_uuid)

    # view-model methods
    def build_app_viewmodel(self, model_dict, clipboard=None, wiki_url=None):
        """Build the main interface.

        Args:
            model_dict (dict): The list of models to build the interface
            clipboard (Clipboard): The clipboard to use
            wiki_url (str): The wiki url to load
        """
        self.application = vm_Application()
        app_name = ''
        first_time = True
        sys_complexity = None
        for model in model_dict:
            if first_time:
                first_time = False

                app_name = model

                _, sys_complexity_var, _ = model_dict[model].get_setting_var('Complexity')
                sys_complexity = sys_complexity_var.value if sys_complexity_var else None
                self.application.gui_tool = model_dict[model].get_selected_gui_tool()

                model_dict[model].verify_layout()

                self.get_main_icon(model_dict[model])
                self.update_title_and_status(model_dict[model])
                self.create_menu_bar(model_dict[model], clipboard, sys_complexity=sys_complexity)
                self.create_toolbars(model_dict[model], sys_complexity=sys_complexity)

                _, self.application.show_wiki = model_dict[model].get_setting('Show wiki panel')
                # _, self.application.use_only_dock_wiki = model_dict[model].get_setting('Use only dockable wiki')

        self.create_data_tree(model_dict)
        # self.create_windows(model_dict[model])
        # Size the main window
        self.size_window(self.application, app_name)

        # Set document windows:
        if AppData.graphics_view is not None:
            self.application.doc_windows_layout = AppData.doc_windows_layout
            for doc_window_uuid in AppData.doc_windows:
                doc_w = AppData.doc_windows[doc_window_uuid]
                if doc_w is None:
                    continue
                self.size_window(doc_w, app_name)
                doc_w.icon = self.application.icon
                doc_w.resulting_image = AppData.graphics_view.render(
                    doc_window_uuid, doc_w.camera_orientation,
                    width=doc_w.resulting_image.image_width,
                    height=doc_w.resulting_image.image_height)
                self.application.doc_windows[doc_window_uuid] = AppData.doc_windows[doc_window_uuid]

        for support_window in AppData.support_windows:
            self.size_window(support_window, app_name)
            support_window.icon = self.application.icon
            if support_window.show_wiki is False:
                support_window.use_only_this_wiki_enabled = False
            else:
                support_window.use_only_this_wiki_enabled = True
            self.application.support_windows.append(support_window)

        self.application.wiki_url = wiki_url

        return self.application

    def convert_string_to_bool(self, string):
        """Convert a string to a boolean.

        Args:
            string (str): The string to convert.

        Returns:
            bool: The converted boolean value.
        """
        if string.lower() == 'true':
            return True
        elif string.lower() == 'false':
            return False
        else:
            try:
                return bool(int(string))
            except ValueError:
                return None

    def size_window(self, window, app_name):
        """Size the window.

        Args:
            window (view model): The window to size
            app_name (str): The name of the application
        """
        app_settings = AppData.app_settings

        if self.view is not None:
            screens_info = self.view.get_screens_info()

        # window.screen_index = screens_info['primary_screen']['index']
        screen_setup_str = screens_info['screen_setup_str']

        x = None
        y = None
        width = 1200
        height = 800

        maximized = app_settings.retrieve_persistent_dialog_data(screen_setup_str, window.window_title, app_name,
                                                                 'maximized')
        if maximized:
            window.maximized = self.convert_string_to_bool(maximized)
        window.minimized = False

        retrieved_x = app_settings.retrieve_persistent_dialog_data(screen_setup_str, window.window_title, app_name,
                                                                   'x')
        if retrieved_x:
            try:
                x = int(retrieved_x)
            except ValueError:
                pass

        retrieved_y = app_settings.retrieve_persistent_dialog_data(screen_setup_str, window.window_title, app_name,
                                                                   'y')
        if retrieved_y:
            try:
                y = int(retrieved_y)
            except ValueError:
                pass

        retrieved_width = app_settings.retrieve_persistent_dialog_data(screen_setup_str, window.window_title,
                                                                       app_name, 'width')
        if retrieved_width:
            try:
                width = int(retrieved_width)
            except ValueError:
                pass

        retrieved_height = app_settings.retrieve_persistent_dialog_data(screen_setup_str, window.window_title,
                                                                        app_name, 'height')
        if retrieved_height:
            try:
                height = int(retrieved_height)
            except ValueError:
                pass

        screen_res_x, screen_res_y = self.view.get_screen_resolution()

        if x is None or y is None and screen_res_x is not None and screen_res_y is not None:
            x = (screen_res_x - width) / 2
            y = (screen_res_y - height) / 2
            if x < 0:
                x = 0
            if y < 20:  # Give enought to grab and move
                y = 20

        window.x = x
        window.y = y
        window.width = width
        window.height = height

        # Validate the window size
        if window.x < 0:
            window.x = 0
        if window.y < 20:  # Give enought to grab and move
            window.y = 20
        if window.x + window.width > screens_info['max_x']:
            if window.width > screens_info['max_x']:
                window.x = 0
                window.width = screens_info['max_x']
            else:
                window.x = screens_info['max_x'] - width
        if window.y + window.height > screens_info['max_y']:
            if window.height > screens_info['max_y']:
                window.y = 0
                window.height = screens_info['max_y']
            else:
                window.y = screens_info['max_y'] - height

    def get_main_icon(self, model_base):
        """Get the main icon.

        Args:
            model_base (ModelBase): The model to get the icon
        """
        self.application.icon = model_base.model_icon
        return self.application.icon

    def update_title_and_status(self, model_base):
        """Update the title and status bar.

        Args:
            model_base (ModelBase): The model to update the title and status bar
        """
        projects_list = model_base.get_projects_list()
        multiple_projects = len(projects_list) > 1

        self.app_name = model_base.model_name
        self.version = model_base.version
        self.application.window_title = f'{self.app_name} v{self.version}'

        current_file = model_base.file_manager.get_current_file()
        if current_file and not multiple_projects:
            self.application.window_title += f' - {current_file}'
        return self.application.window_title

    def create_menu_bar(self, model_base, clipboard=None, sys_complexity=None):
        """Create the menu bar.

        Args:
            model_base (ModelBase): The model to create the menu bar
            clipboard (Clipboard): The clipboard to use
            sys_complexity (int, optional): The system complexity level. Defaults to None.
        """
        self.application.menu_bar = self.tools.create_menu_bar(model_base, clipboard, sys_complexity=sys_complexity)
        return self.application.menu_bar

    def create_toolbars(self, model_base, sys_complexity=None):
        """Create the tool bars.

        Args:
            model_base (ModelBase): The model to create the tool bars
            sys_complexity (int): The system complexity
        """
        self.application.tool_bar, self.application.center_toolbar_names = self.tools.create_toolbars(
            model_base, sys_complexity=sys_complexity)
        return self.application.tool_bar

    def create_data_tree(self, model_dict):
        """Create the data tree.

        Args:
            model_dict (dict): The dict of models to create the data tree
        """
        self.application.data_tree = self.data_tree.build_tree(model_dict)
        return self.application.data_tree

    # def create_windows(self, model_base):
    #     """Create the windows.

    #     Args:
    #         model_base (ModelBase): The model to create the windows
    #     """
    #     pass
