"""CalcData for performing Abutment Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.contraction_scour_calc import ContractionScourCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_calc import ScourBaseCalc
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class AbutmentScourCalc(ScourBaseCalc):
    """A class that defines a contraction scour at a bridge contraction."""
    # Change in how we are using figures 8.9 - 8.12 per Scott Hogan's e-mail Sep 11, 2024 ("Follow-up")
    # The abutment scour references and application of Figures 8.9 through 8.12 is being significantly simplified in
    # the 6th ed of HEC-18.  We are dropping figures 8.9 and 8.11 and using only the amplification factors for the
    # vertical wall with wingwall abutments.   Figure 8.9 will apply for Scour Condition A and Figure 8.11 for
    # Scour condition B.

    # # Figure 8.9 from HEC-18
    # q_spillthrough_mainchannel = [
    #     1.005, 1.015, 1.028, 1.04, 1.061, 1.095, 1.12, 1.145, 1.17, 1.2, 1.225,
    #     1.25, 1.3, 1.355, 1.408, 1.461, 1.501, 1.535, 1.581, 1.648, 1.698,
    #     1.77, 1.86, 2.0, 2.1, 2.25, 2.5, 2.75, 3.0
    # ]
    # α_spillthrough_mainchannel = [
    #     1.2, 1.3, 1.4, 1.465, 1.53, 1.6, 1.628, 1.65, 1.663, 1.673, 1.676,
    #     1.677, 1.67, 1.648, 1.6, 1.52, 1.45, 1.402, 1.35, 1.3, 1.276, 1.243,
    #     1.213, 1.179, 1.1635, 1.147, 1.126, 1.113, 1.102
    # ]
    # α_spillthrough_mainchannel_interpolate = Interpolation(
    #     q_spillthrough_mainchannel, α_spillthrough_mainchannel)

    # 8.10 from HEC-18 (previously α_wingwall_mainchannel)
    α_mainchannel_x = [
        1.0006, 1.0185, 1.0345, 1.048, 1.0655, 1.091, 1.118, 1.142, 1.17, 1.2,
        1.225, 1.25, 1.3, 1.355, 1.406, 1.462, 1.501, 1.537, 1.584, 1.648,
        1.699, 1.772, 1.8615, 2.0, 2.1, 2.25, 2.5, 2.75, 3.0
    ]
    α_mainchannel_y = [
        1.2, 1.3, 1.4, 1.465, 1.53, 1.6, 1.653, 1.687, 1.716, 1.737, 1.749,
        1.755, 1.757, 1.741, 1.708, 1.659, 1.618, 1.58, 1.53, 1.467, 1.423,
        1.367, 1.31, 1.244, 1.209, 1.169, 1.129, 1.109, 1.102
    ]

    # 8.11 from HEC-18 (previously α_spillthrough_floodplain)
    α_floodplain_x = [
        1.0, 1.019, 1.051, 1.0713, 1.1059, 1.155, 1.2, 1.25, 1.3, 1.35, 1.4,
        1.45, 1.5, 1.55, 1.6, 1.65, 1.7, 1.75, 1.8, 1.85, 2.0, 2.25, 2.5, 2.75,
        3.0, 3.25, 3.5, 3.75, 4.0, 4.6, 5.0
    ]
    α_floodplain_y = [
        1.0, 1.25, 1.49, 1.6, 1.75, 1.9, 1.995, 2.071, 2.122, 2.156, 2.1756,
        2.184, 2.185, 2.179, 2.168, 2.153, 2.137, 2.116, 2.095, 2.073, 2.002,
        1.876, 1.749, 1.626, 1.509, 1.4108, 1.34, 1.282, 1.235, 1.159, 1.134
    ]

    # # 8.12 from HEC-18
    # α_wingwall_floodplain = [
    #     1.0025, 1.003, 1.0051, 1.0053, 1.009, 1.0145, 1.019, 1.0255, 1.034,
    #     1.045, 1.06, 1.0765, 1.099, 1.13, 1.17, 1.22, 1.29, 1.35, 1.42, 1.47,
    #     1.58, 1.7, 1.85, 2.0, 2.25, 2.5, 2.75, 3.0, 3.25, 3.5, 3.75, 4.0, 4.25,
    #     4.6, 5.0
    # ]
    # α_wingwall_floodplain = [
    #     1.0, 1.25, 1.49, 1.6, 1.75, 1.9, 1.995, 2.1, 2.2, 2.3, 2.4, 2.48,
    #     2.551, 2.602, 2.622, 2.604, 2.554, 2.503, 2.441, 2.396, 2.299, 2.198,
    #     2.082, 1.975, 1.818, 1.687, 1.574, 1.478, 1.397, 1.329, 1.274, 1.229,
    #     1.193, 1.157, 1.134
    # ]
    # α_wingwall_floodplain_interpolate = Interpolation(α_wingwall_floodplain,
    #                                                   α_wingwall_floodplain)

    def __init__(self, null_data, zero_tol):
        """Initializes the AbutmentScourCalc class.

        Args:
            null_data (float): The value to use for null data.
            zero_tol (float): The tolerance for zero values.
        """
        super().__init__()

        self.α_mainchannel_interpolate = Interpolation(AbutmentScourCalc.α_mainchannel_x,
                                                       AbutmentScourCalc.α_mainchannel_y,
                                                       null_data=null_data, zero_tol=zero_tol)
        self.α_floodplain_interpolate = Interpolation(AbutmentScourCalc.α_floodplain_x,
                                                      AbutmentScourCalc.α_floodplain_y,
                                                      null_data=null_data, zero_tol=zero_tol)

        self.d50 = None
        self.critical_shear_stress = None

    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        # _, zero_tol = self.get_data('Zero tolerance')

        if self.gradations:
            self.update_gradation_lists()
        else:
            _, self.d50 = self.get_data('Contracted D50')
            _, self.critical_shear_stress = self.get_data('Critical shear stress (τc)')
            _, self.upstream_d50 = self.get_data('Approach D50')

        result = self.check_float_vars_to_greater_zero(
            ['Unit discharge, upstream (q1)', 'Unit discharge, contracted (q2)', 'Average depth upstream (y1)',
             'Depth prior to scour in contracted section (y0)'])

        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            if not self.check_float_vars_to_greater_zero(["Manning's n (n)"]):
                result = False

        return result

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Document the source of the equation (if possible):
        #   Give publication, date, page number and link (if there's a bug, that it can be traced back to the source).
        # Assign new variables from the input_dict in similar names as the equation and document units

        # HEC-18, 6th Edition, 2024, Assorted pages; see chapter 8 for details
        _, self.d50 = self.get_data('Contracted D50')
        _, self.critical_shear_stress = self.get_data('Critical shear stress (τc)')
        self.compute_nchrp_data()

        if self.gradations is not None and self.wse_x is not None and self.wse_y is not None and \
                len(self.cross_section_x) > 0 and len(self.cross_section_y) > 0 and \
                self.abutment_toe_station is not None:
            #  Determine scour depth by subtracting the max flow depth from the WSE
            _, null_data = self.get_data('Null data')
            _, zero_tol = self.get_data('Zero tolerance')
            if len(self.wse_y) > 1:
                wse_interp = Interpolation(
                    self.wse_x, self.wse_y, null_data=null_data, zero_tol=zero_tol)
                wse_elevation = float(wse_interp.interpolate_y(self.abutment_toe_station)[0])
            elif len(self.wse_y) == 1:
                wse_elevation = float(self.wse_y[0])
            streambed_interp = Interpolation(
                self.cross_section_x, self.cross_section_y, null_data=null_data, zero_tol=zero_tol)
            self.streambed_elevation, _ = streambed_interp.interpolate_y(self.abutment_toe_station)
            bottom_elevation = wse_elevation - self.ymax
            self.scour_depth = self.streambed_elevation - bottom_elevation
            self.results['Scour hole depth (yc)'] = self.scour_depth

            if self.is_computing_shear_decay:
                self.compute_shear_decay(wse_elevation)

        self.compute_scour_hole_geometry()

        return self.scour_depth

    def compute_shear_decay(self, wse_elevation):
        """Compute the shear decay for contraction scour.

        Args:
            wse_elevation (float): Water surface elevation
        """
        if self.gradations is None:
            self.warnings['Gradations not defined'] = 'Gradations must be defined to compute shear decay.'
            return
        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')
        _, y1 = self.get_data('Average depth upstream (y1)')

        _, unit_discharge = self.get_data('Unit discharge, contracted (q2)')

        thalweg = float(self.streambed_elev_post_scour)
        original_streambed = self.streambed_elev_prior_to_scour
        cs_and_ltd = original_streambed - thalweg
        if cs_and_ltd < 0.0:
            cs_and_ltd = 0.0
        # wse = original_streambed + y1
        # # Use the WSE, if we have it
        # if self.wse_x is not None and max(self.wse_x) > self.centerline and min(self.wse_x) < self.centerline:
        #     wse = float(np.interp(self.centerline, self.wse_x, self.wse_y))

        decay_x = []  # Shear
        decay_y = []  # Elevation
        marker_x = []
        marker_y = []
        if len(self.con_decay_y_markers) > 0:
            lowest_marker = self.con_decay_y_markers[-1]

        _, include_contraction_shear = self.get_data('Include contraction shear')
        if include_contraction_shear:
            decay_x = copy.deepcopy(self.con_decay_x)
            decay_y = copy.deepcopy(self.con_decay_y)
            marker_x = copy.deepcopy(self.con_decay_x_markers)
            marker_y = copy.deepcopy(self.con_decay_y_markers)

        if len(marker_y) > 0:
            thalweg = min(thalweg, lowest_marker)

        # Enter decay function parameters (Later provide a lookup table based on Pier Shape)
        shear_alpha, shear_beta = self.compute_alpha_beta_for_pier_shape()

        # _, g = self.get_data('Gravity')

        # Calculate flow conditions after contraction scour
        # y2 = y0 + cs_and_ltd  # New depth after contraction scour
        # v2 = unit_discharge / y2
        # fr2 = (v2) / (g * y2) ** 0.5

        if 'Shear decay results' not in self.results:
            self.results['Shear decay results'] = {}
        self.results['Shear decay results']['Unit discharge'] = unit_discharge
        self.results['Shear decay results']['Shear decay alpha'] = shear_alpha
        self.results['Shear decay results']['Shear decay beta'] = shear_beta
        # self.results['Shear decay results']['New depth after cs and ltd (yp)'] = y2)
        # self.results['Shear decay results']['New velocity after cs and ltd (v2)'] = v2)

        nchrp_alpha = self.results['Amplification factor (α)']
        q2_over_q1 = self.results['q2 / q1']

        self.compute_local_shear_decay(
            decay_x, decay_y, marker_x, marker_y, shear_alpha, shear_beta, wse_elevation,
            thalweg, unit_discharge, nchrp_alpha, q2_over_q1, y0, y1)

    def compute_local_shear_decay(self, decay_x, decay_y, marker_x, marker_y, shear_alpha, shear_beta, wse,
                                  thalweg, unit_q, nchrp_alpha, q2_over_q1, y0, y1):
        """Compute the shear decay for contraction scour.

        Args:
            decay_x (list): The x values for the decay curve.
            decay_y (list): The y values for the decay curve.
            marker_x (list): The x values for the markers.
            marker_y (list): The y values for the markers.
            shear_alpha (float): The shear alpha.
            shear_beta (float): The shear beta.
            wse (float): The water surface elevation.
            thalweg (float): The thalweg elevation.
            unit_q (float): The maximum unit discharge.
            nchrp_alpha (float): The NCHRP alpha.
            q2_over_q1 (float): The q2 over q1.
            y0 (float): The depth prior to scour.
            y1 (float): The average depth upstream.
        """
        # Now taken care of with scour base functionality
        # bh_uuid = self.get_data('Selected borehole')
        # bh = self.bh_dict[bh_uuid]
        # self.bh_layers = bh.get_data('Layers').item_list

        # Get starting soil layer
        self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = self.get_layer_index_from_elevation(
            thalweg)
        mannings_n = self.bh_layers[self.layer_index]['calculator'].manning_n
        cur_mannings_n = mannings_n
        lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(
            self.layer_index, wse, unit_q)
        if len(self.bh_layers) == 1:
            cur_layer_top = thalweg
            cur_layer_bottom = -sys.float_info.max

        cur_decay_y = thalweg
        cur_shear = self.compute_shear(wse, cur_decay_y, cur_mannings_n, unit_q)
        _, decay_inc = self.get_data('Shear decay curve increment', 0.5)

        _, min_thickness = self.get_data('Min layer thickness', 2.0)

        # Add Hager Equation as alternative y_max (g=9.81 m/s^2, Sg=2.65 ) --- Noncohesive soils
        # d50 = self.bh_layers[self.layer_index].get_d50()
        # sigma = self.bh_layers[self.layer_index].results['Shear decay parameters'][
        #     'Sediment gradation coefficient (σ)')
        # gamma_s = self.bh_layers[self.layer_index].get_data('Unit weight of soil (γs)')
        # ymax_depth = nchrp_alpha * y1 * (q2_over_q1) ** (0.8571428571428571) - y0  # 6/7 = 0.8571

        # contraction_method = self.results['Governing scour method')
        ymax_depth = 0.0
        cur_layer_index = self.layer_index - 1
        cur_depth = thalweg - cur_layer_bottom
        layer_top = thalweg
        first_time = True
        _, gamma_w = self.get_data('Unit weight of water (γw)')
        _, m_k = self.get_data('Manning constant', 1.486)
        alpha = self.results['Amplification factor (α)']

        three7 = 0.428571428571  # 3/7
        six7 = 0.8571428571428571  # 6/7
        lower_layer_weaker = False
        # Determine the max depth for the shear decay curve
        while cur_layer_index + 1 < len(self.bh_layers) and ((first_time or ymax_depth > cur_depth)
                                                             or lower_layer_weaker):
            cur_layer_index += 1
            cur_layer = self.bh_layers[cur_layer_index]
            n = float(cur_layer['calculator'].manning_n)
            critical_shear = float(cur_layer['calculator'].tau)
            if critical_shear is None or critical_shear <= 0.0:
                # This is an issue.... report, skip layer, and continue
                self.warnings[f'Layer {str(self.layer_index)}'] = \
                    (f'Layer {str(self.layer_index)} has a critical shear stress of 0.0 or less; '
                     'continuing the shear decay curve.')
                continue
            cur_layer_top = cur_layer['Top elevation']
            cur_layer_bottom = cur_layer['Bottom elevation']
            if layer_top > cur_layer_top:
                layer_top = cur_layer_top  # Don't get a layer top value higher than thalweg

            new_depth = (gamma_w / critical_shear) ** (three7) * (n * unit_q / m_k) ** (six7) * (alpha - 1)
            if new_depth > thalweg - layer_top:
                ymax_depth = new_depth
            else:
                ymax_depth = thalweg - layer_top
            cur_depth = thalweg - cur_layer_bottom

            cur_thickness = min(cur_layer_top - cur_layer_bottom, cur_depth - ymax_depth)
            lower_critical = None
            if cur_layer_index + 1 < len(self.bh_layers):
                lower_critical = float(self.bh_layers[cur_layer_index + 1]['calculator'].tau)
            if lower_critical is not None:
                lower_layer_weaker = lower_critical < cur_critical_shear and cur_thickness < min_thickness and \
                    cur_thickness > 0.0

            first_time = False

        ymax_depth = max(ymax_depth, 0.0)
        ymax = thalweg - ymax_depth

        # # Next line is future reference - if we want to use clear water max flow depth
        # # max_flow_depth = self.results['Max flow depth including scour (ymax)')
        # if contraction_method == 'Live bed':
        #     amplification_factor = self.results['Amplification factor (α)')
        #     _, ymax_live_bed = self.determine_flow_depth_including_scour_for_live_bed(y1, q2_over_q1,
        #                                                                               amplification_factor)

        #     # ymax_depth = max_flow_depth - (wse - thalweg)

        #     ymax_depth = ymax_live_bed
        #     if ymax_depth < 0:
        #         ymax_depth = 0.0

        #     self.results['Shear decay results']['Max computed scour depth'] = ymax_depth)

        #     # ymax_depth = ymax_hager
        #     # use_csu = self.get_data('Use CSU equation')
        #     # if use_csu:
        #     #     ymax_depth = ymax_csu
        #     ymax = thalweg - ymax_depth

        self.results['Shear decay results']['Unadjusted computed local shear'] = cur_shear

        # Calculate amplified tau at the pier.
        self.tau_local = shear_alpha * cur_shear

        # Enter zero depth data (adjustment blows up with zero increment)
        scour_depth = 0.0
        decay_x.append(self.tau_local)
        decay_y.append(cur_decay_y)
        marker_x.append(self.tau_local)
        marker_y.append(cur_decay_y)

        # Increment data
        scour_depth += decay_inc
        cur_decay_y -= decay_inc
        cur_thickness = cur_decay_y - cur_layer_bottom

        # Check if current layer is getting too thin to trust
        cur_thickness = cur_decay_y - cur_layer_bottom
        lower_layer_weaker = False
        if lower_layer_n is not None:
            lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness and \
                lower_layer_shear > lower_layer_critical

        layers_too_thin = []
        k_roughness = 1.0

        cur_shear = self.tau_local
        if ymax_depth and ymax_depth > 0:
            cur_shear = float(self.tau_local * np.exp(shear_beta * (0.0 / ymax_depth)))

        # Determine shear stress until criteria is met (converge on critical shear, or layer is too thin)
        while cur_shear > cur_critical_shear and cur_decay_y > ymax or \
                (cur_shear <= cur_critical_shear and lower_layer_weaker):
            if not cur_shear > cur_critical_shear and cur_decay_y > ymax and self.layer_index not in \
                    layers_too_thin:
                layers_too_thin.append(self.layer_index)
                self.warnings[f'Layer {str(self.layer_index)} is too thin'] = \
                    f'Layer {str(self.layer_index)} is too thin to trust; continuing the shear decay curve.'
            if not cur_layer_top >= cur_decay_y >= cur_layer_bottom:
                # Determine the shear stress at the bottom of the layer
                # step = thalweg - cur_layer_bottom
                cur_shear = float(self.tau_local * np.exp(shear_beta * (scour_depth / ymax_depth)))
                decay_x.append(cur_shear)
                decay_y.append(cur_layer_bottom)
                marker_x.append(cur_shear)
                marker_y.append(cur_layer_bottom)
                # Update data from the layers
                self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = \
                    self.get_layer_index_from_elevation(cur_decay_y)
                cur_mannings_n = self.bh_layers[self.layer_index]['calculator'].results['Shear decay parameters'][
                    'Manning n']
                lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(
                    self.layer_index, wse, unit_q)

                # K_roughness=(self.soil_summary.loc[i,'Manning n']/self.soil_summary.loc[idx,'Manning n'])**2
                # k_roughness = (cur_mannings_n / mannings_n) ** 2

                # # Determine the shear stress at the top of the new layer
                # step = thalweg - cur_layer_top
                # cur_shear = float(self.tau_local * np.exp(shear_beta * (step / ymax_depth)))
                # decay_x.append(cur_shear)
                # decay_y.append(cur_layer_top)
                # marker_x.append(cur_shear)
                # marker_y.append(cur_layer_top)

            cur_shear = float(self.tau_local * k_roughness * np.exp(-1 * shear_beta * (scour_depth / ymax_depth)))

            decay_x.append(cur_shear)
            decay_y.append(cur_decay_y)
            # If we overshot, correct it:
            if cur_shear < cur_critical_shear:
                _, null_data = self.get_data('Null data')
                _, zero_tol = self.get_data('Zero tolerance')
                shear_interp = Interpolation(decay_x, decay_y, null_data=null_data, zero_tol=zero_tol)
                decay_y[-1], _ = shear_interp.interpolate_y(cur_critical_shear)
                decay_x[-1] = cur_critical_shear

            # Update the current elevation
            scour_depth += decay_inc
            cur_decay_y -= decay_inc
            cur_thickness = cur_decay_y - cur_layer_bottom
            if lower_layer_n is not None:  # Check if current layer is getting too thin to trust
                lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness and \
                    lower_layer_shear > lower_layer_critical

        marker_x.append(decay_x[-1])
        marker_y.append(decay_y[-1])

        self.results['Shear decay results']['Shear decay curve shear'] = decay_x
        self.results['Shear decay results']['Shear decay curve elevation'] = decay_y
        self.results['Shear decay results']['Shear decay curve shear markers'] = marker_x
        self.results['Shear decay results']['Shear decay curve elevation markers'] = marker_y

        # Set the determined scour depth
        self.scour_depth = thalweg - decay_y[-1]

        self.set_layer_plot_data(decay_x, decay_y, marker_x, marker_y, ymax)

    def determine_flow_depth_including_scour_for_clear_water(self, q2, d50):
        """Determine the flow depth including scour for the Clear water method.

        Args:
            q2 (float): Unit discharge at contracted section
            d50 (float): Contracted D50 value
        """
        # result, manning_k = self.get_data('Manning constant', 1.486)
        if d50 > 0.0:
            ku = 11.17

            # HEC 18, page 8.12 (pdf page 226)
            yc = (q2 / (ku * d50**0.333333333333))**0.857142857143  # 6/7 = 0.857...
        else:
            # If critical stress is known, you can use an alternate equation
            # HEC-18, Equation 8.7 (pdf page 227)
            _, gamma_w = self.get_data('Unit weight of water (γw)')
            _, tau_c = self.get_data('Critical shear stress (τc)')
            _, n = self.get_data("Manning's n (n)")
            _, ku = self.get_data('Manning constant', 1.486)
            yc = 0.0
            if n > 0.0 and q2 > 0.0 and tau_c > 0.0:
                yc = (gamma_w / tau_c) ** (3.0 / 7.0) * (n * q2 / ku) ** 0.857142857143  # 6/7 = 0.857...
        return yc

    def determine_flow_depth_including_scour_for_live_bed(self, y1, q2_over_q1, amplification_factor):
        """Determine the flow depth including scour for the Live-bed method."""
        yc = y1 * q2_over_q1**0.857142857143  # 6/7 = 0.9868...
        self.ymax = amplification_factor * yc

        return yc, self.ymax

    def compute_nchrp_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.scour_depth = 0.0

        amplification_factor, q2_over_q1 = self.compute_alpha()

        _, scour_condition = self.get_data('Scour condition')
        contraction_method = scour_condition
        _, q1 = self.get_data('Unit discharge, upstream (q1)')
        _, q2 = self.get_data('Unit discharge, contracted (q2)')
        _, y1 = self.get_data('Average depth upstream (y1)')
        v1 = 0.0
        if y1 > 0.0:
            v1 = q1 / y1

        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            contraction_method = 'Cohesive'

        else:
            if scour_condition == 'Compute':
                _, zero_tol = self.get_data('Zero tolerance')
                _, null_data = self.get_data('Null data')
                contraction_calc = ContractionScourCalc(null_data=null_data, zero_tol=zero_tol,)

                self.input_dict['calc_data']['Average depth upstream (y1)'] = y1
                self.input_dict['calc_data']['Average velocity upstream (v1)'] = v1

                contraction_calc.input_dict = self.input_dict
                contraction_calc.gradations = self.gradations
                contraction_calc.upstream_d50 = self.upstream_d50
                contraction_calc.d50 = self.d50
                contraction_calc.critical_shear_stress = self.critical_shear_stress

                is_upstream_transporting_sediment = contraction_calc.determine_if_upstream_is_transporting_sediment()

                if is_upstream_transporting_sediment:
                    contraction_method = 'Live bed'
                else:
                    contraction_method = 'Clear water'

            self.results['Governing scour method'] = contraction_method

        if contraction_method == 'Live bed':
            yc, self.ymax = self.determine_flow_depth_including_scour_for_live_bed(y1, q2_over_q1,
                                                                                   amplification_factor)
            self.results['Flow depth including scour (yc)'] = yc
            self.results['Max flow depth including scour (ymax)'] = self.ymax

            yc_clear = self.determine_flow_depth_including_scour_for_clear_water(q2, self.d50)
            self.results['Flow depth including scour using clear water method (yc)'] = yc_clear
            ymax_clear = amplification_factor * yc_clear
            self.results['Max flow depth including scour using clear water method (ymax)'] = yc_clear
            if ymax_clear < self.ymax:
                self.warnings['Armor warning'] = \
                    'Scour MAY be limited by clear-water (armoring) even though the transport condition is live-bed'
        else:
            yc = self.determine_flow_depth_including_scour_for_clear_water(q2, self.d50)
            self.results['Flow depth including scour (yc)'] = yc
            self.ymax = amplification_factor * yc
            self.results['Max flow depth including scour (ymax)'] = self.ymax

        self.ymax += self.ltd
        self.results['longterm degradation'] = self.ltd

        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')
        ys = self.ymax - y0
        self.results['Average scour depth (y2)'] = ys
        self.results['Scour hole depth (yc)'] = ys
        self.scour_depth = ys

        return True

    def compute_alpha(self):
        """Computes the amplification factor."""
        _, scour_location = self.get_data('Scour location')

        _, q1 = self.get_data('Unit discharge, upstream (q1)')
        _, q2 = self.get_data('Unit discharge, contracted (q2)')

        q2_over_q1 = 0.0
        if q1 > 0.0:
            q2_over_q1 = q2 / q1

        # # Interpolate from figures 8.9, 8.10, 9.11, 8.12 from HEC-18, page 8.11 (pdf page 225)
        # if abutment_type == 'Spill-through abutment':
        #     if scour_location == 'Type a (main channel)':
        #         alpha, _ = AbutmentScourData.α_mainchannel_interpolate.interpolate_y(
        #             q2_over_q1, False)
        #     else:
        #         alpha, _ = AbutmentScourData.α_floodplain_interpolate.interpolate_y(
        #             q2_over_q1, False)
        # else:
        #     if scour_location == 'Type a (main channel)':
        #         alpha, _ = AbutmentScourData.α_mainchannel_interpolate.interpolate_y(
        #             q2_over_q1, False)
        #     else:
        #         alpha, _ = AbutmentScourData.α_floodplain_interpolate.interpolate_y(
        #             q2_over_q1, False)
        if scour_location == 'Type a (main channel)':
            alpha, _ = self.α_mainchannel_interpolate.interpolate_y(q2_over_q1, False)
        else:
            alpha, _ = self.α_floodplain_interpolate.interpolate_y(q2_over_q1, False)

        self.results['Amplification factor (α)'] = alpha
        self.results['q2 / q1'] = q2_over_q1

        return alpha, q2_over_q1
