"""A Python3 module to find if 2 given line segments intersect or not."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.HydraulicToolboxCalc.util.distance import Point, xy_distance


def on_segment(p: Point, q: Point, r: Point) -> bool:
    """Given three collinear points p, q, r, the function checks if point q lies on line segment 'pr'.

    Args:
        p (Point): first point of triplet
        q (Point): second point of triplet
        r (Point): third point of triplet
    """
    if (q.x <= max(p.x, r.x)) and (q.x >= min(p.x, r.x)):
        if (q.y <= max(p.y, r.y)) and (q.y >= min(p.y, r.y)):
            return True
    return False


def orientation(p: Point, q: Point, r: Point) -> int:
    """Determine the orientation of of an ordered triplet (p, q, r).

    Args:
        p (Point) first point of triplet
        q (Point): second point of triplet
        r (Point): third point of triplet

    Returns:
        0 if collinear points, 1 if clockwise points, 2 if counterclockwise points
    """
    # See https://www.geeksforgeeks.org/orientation-3-ordered-points/amp/
    # for details of below formula.

    val = (float(q.y - p.y) * (r.x - q.x)) - (float(q.x - p.x) * (r.y - q.y))
    if (val > 0):
        # Clockwise orientation
        return 1
    elif (val < 0):
        # Counterclockwise orientation
        return 2
    else:
        # Collinear orientation
        return 0


def find_intersection_and_closest_index(
        poly_line1_x: list[float], poly_line1_y: list[float], poly_line2_x: list[float], poly_line2_y: list[float],
        poly_line1_start_index: int = 0, poly_line2_start_index: int = 0, ignore_int_list: list[list[float]] = None,
        tol: float = 0.00001) -> tuple[bool, float, float, int, int]:
    """Find intersection point and closest index.

    Args:
        poly_line1_x (float): x value of poly line 1
        poly_line1_y (float): y value of poly line 1
        poly_line2_x (float): x value of poly line 2
        poly_line2_y (float): y value of poly line 2
        poly_line1_start_index (int): index to start search on poly_line1
        poly_line2_start_index (int): index to start search on poly_line2
        ignore_int_list (list): list of list of x, y points of intersections to ignore

    Returns:
        intersection_found (bool): Whether the points intersect
        intersect_x (float): x value of the point of intersection
        intersect_y (float): y value of the point of intersection
        poly_line1_closest_index (int): index of closest point to the intersection on poly line 1
        poly_line2_closest_index (int): index of closest point to the intersection on poly line 2
    """
    poly_line1_closest_index = -1
    poly_line2_closest_index = -1
    closest_dist = sys.float_info.max

    if not isinstance(poly_line1_start_index, int) or poly_line1_start_index >= len(poly_line1_x) - 1:
        raise ValueError("poly_line1_start_index must be an integer less than the length of poly_line1_x - 1")
    if not isinstance(poly_line2_start_index, int) or poly_line2_start_index >= len(poly_line2_x) - 1:
        raise ValueError("poly_line2_start_index must be an integer less than the length of poly_line2_x - 1")

    for poly_line1_index in range(poly_line1_start_index,
                                  len(poly_line1_x) - 1):
        for poly_line2_index in range(poly_line2_start_index, len(poly_line2_x) - 1):
            if do_intersect_xy(poly_line1_x[poly_line1_index],
                               poly_line1_y[poly_line1_index],
                               poly_line1_x[poly_line1_index + 1],
                               poly_line1_y[poly_line1_index + 1],
                               poly_line2_x[poly_line2_index],
                               poly_line2_y[poly_line2_index],
                               poly_line2_x[poly_line2_index + 1],
                               poly_line2_y[poly_line2_index + 1]):
                intersect_x, intersect_y, found = find_intersection_xy(
                    poly_line1_x[poly_line1_index],
                    poly_line1_y[poly_line1_index],
                    poly_line1_x[poly_line1_index + 1],
                    poly_line1_y[poly_line1_index + 1],
                    poly_line2_x[poly_line2_index],
                    poly_line2_y[poly_line2_index],
                    poly_line2_x[poly_line2_index + 1],
                    poly_line2_y[poly_line2_index + 1])
                if found:
                    ignore_intersection = False
                    if ignore_int_list is not None:
                        for xy_list in ignore_int_list:
                            if intersect_x + tol > xy_list[0] and intersect_x - tol < xy_list[0]:
                                if intersect_y + tol > xy_list[1] and intersect_y - tol < xy_list[1]:
                                    ignore_intersection = True
                                    break
                    if not ignore_intersection:
                        return True, intersect_x, intersect_y, poly_line1_index, poly_line2_index
            distance = xy_distance(poly_line1_x[poly_line1_index],
                                   poly_line1_y[poly_line1_index],
                                   poly_line2_x[poly_line2_index],
                                   poly_line2_y[poly_line2_index])
            if closest_dist > distance:
                closest_dist = distance
                poly_line1_closest_index = poly_line1_index
                poly_line2_closest_index = poly_line2_index

    intersect_x = poly_line1_x[poly_line1_closest_index]
    intersect_y = poly_line1_y[poly_line1_closest_index]

    return False, intersect_x, intersect_y, poly_line1_closest_index, poly_line2_closest_index


def do_intersect_xy(line1_pt1_x: float, line1_pt1_y: float, line1_pt2_x: float, line1_pt2_y: float,
                    line2_pt1_x: float, line2_pt1_y: float, line2_pt2_x: float, line2_pt2_y: float) -> bool:
    """Determines if line 1 and line 2 intersect (with XY values, not points).

    Args:
        line1_pt1_x (float): x value of poly line 1, point 1
        line1_pt1_y (float): y value of poly line 1, point 1
        line1_pt2_x (float): x value of poly line 1, point 2
        line1_pt2_y (float): y value of poly line 1, point 2
        line2_pt1_x (float): x value of poly line 2, point 1
        line2_pt1_y (float): y value of poly line 2, point 1
        line2_pt2_x (float): x value of poly line 2, point 2
        line2_pt2_y (float): y value of poly line 2, point 2

    Returns:
        True if line 1 and line 2 intersect
    """
    line1_pt1 = Point(line1_pt1_x, line1_pt1_y)
    line1_pt2 = Point(line1_pt2_x, line1_pt2_y)
    line2_pt1 = Point(line2_pt1_x, line2_pt1_y)
    line2_pt2 = Point(line2_pt2_x, line2_pt2_y)

    return do_intersect(line1_pt1, line1_pt2, line2_pt1, line2_pt2)


def do_intersect(line1_pt1: Point, line1_pt2: Point, line2_pt1: Point, line2_pt2: Point) -> bool:
    """Determines if line 1 and line 2 intersect.

    Args:
        line1_pt1 (Point): point 1 of line segment 1
        line1_pt2 (Point): point 1 of line segment 2
        line2_pt1 (Point): point 2 of line segment 1
        line2_pt2 (Point): point 2 of line segment 2

    Returns:
        True if line 1 and line 2 intersect
    """
    # Find the 4 orientations required for
    # the general and special cases
    o1 = orientation(line1_pt1, line1_pt2, line2_pt1)
    o2 = orientation(line1_pt1, line1_pt2, line2_pt2)
    o3 = orientation(line2_pt1, line2_pt2, line1_pt1)
    o4 = orientation(line2_pt1, line2_pt2, line1_pt2)

    # General case
    if (o1 != o2) and (o3 != o4):
        return True

    # Special Cases

    # line1_pt1 , line1_pt2 and line2_pt1 are collinear and line2_pt1 lies on segment line1_pt1-line1_pt2
    if (o1 == 0) and on_segment(line1_pt1, line2_pt1, line1_pt2):
        return True

    # line1_pt1 , line1_pt2 and line2_pt2 are collinear and line2_pt2 lies on segment line1_pt1-line1_pt2
    if (o2 == 0) and on_segment(line1_pt1, line2_pt2, line1_pt2):
        return True

    # line2_pt1 , line2_pt2 and line1_pt1 are collinear and line1_pt1 lies on segment line2_pt1-line2_pt2
    if (o3 == 0) and on_segment(line2_pt1, line1_pt1, line2_pt2):
        return True

    # line2_pt1 , line2_pt2 and line1_pt2 are collinear and line1_pt2 lies on segment line2_pt1-line2_pt2
    if (o4 == 0) and on_segment(line2_pt1, line1_pt2, line2_pt2):
        return True

    # If none of the cases
    return False


def find_intersection_xy(line1_pt1_x: float, line1_pt1_y: float, line1_pt2_x: float, line1_pt2_y: float,
                         line2_pt1_x: float, line2_pt1_y: float, line2_pt2_x: float, line2_pt2_y: float
                         ) -> tuple[float, float, bool]:
    """Finds the intersection of two lines.

    Args:
        line1_pt1_x (float): x value of poly line 1, point 1
        line1_pt1_y (float): y value of poly line 1, point 1
        line1_pt2_x (float): x value of poly line 1, point 2
        line1_pt2_y (float): y value of poly line 1, point 2
        line2_pt1_x (float): x value of poly line 2, point 1
        line2_pt1_y (float): y value of poly line 2, point 1
        line2_pt2_x (float): x value of poly line 2, point 2
        line2_pt2_y (float): y value of poly line 2, point 2
    """
    l1 = line([line1_pt1_x, line1_pt1_y], [line1_pt2_x, line1_pt2_y])
    l2 = line([line2_pt1_x, line2_pt1_y], [line2_pt2_x, line2_pt2_y])

    return intersection(l1, l2)


def line(p1: tuple[float, float], p2: tuple[float, float]) -> tuple[float, float, float]:
    """Line calculation for two points needed to determine intersection.

    Args:
        p1 (tuple): tuple of x, y data for point 1
        p2 (tuple): tuple of x, y data for point 1

    Returns:
        a, b, -c: calculated values
    """
    a = (p1[1] - p2[1])
    b = (p2[0] - p1[0])
    c = (p1[0] * p2[1] - p2[0] * p1[1])
    return a, b, -c


def intersection(line1: tuple[float, float, float], line2: tuple[float, float, float]) -> tuple[float, float, bool]:
    """Determines the intersection of line 1 and line 2.

    Args:
        line1 (line): The first line segment
        line2 (line): The second line segment

    Returns:
        x value of the intersection point
        y value of the intersection point
        True if intersection found, False if not
    """
    x = 0.0
    y = 0.0
    d = line1[0] * line2[1] - line1[1] * line2[0]
    dx = line1[2] * line2[1] - line1[1] * line2[2]
    dy = line1[0] * line2[2] - line1[2] * line2[0]
    if d != 0:
        x = dx / d
        y = dy / d
        return x, y, True
    else:
        return x, y, False
