"""A component for boundary condition data and commands."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from collections import defaultdict
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.api.tree import tree_util
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.components.display import (
    BC_ARC_INITIAL_ATT_ID_FILE, BC_ARC_INITIAL_COMP_ID_FILE, BC_POINT_INITIAL_ATT_ID_FILE,
    BC_POINT_INITIAL_COMP_ID_FILE, fix_duplicated_display_opts
)
from xms.adh.data.bc_io import BcIO
from xms.adh.data.card_info import CardInfo
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.file_io import io_util
from xms.adh.gui.bc_arc_attributes_dialog import BcArcDialog
from xms.adh.gui.bc_point_attributes_dialog import BcPointDialog
from xms.adh.gui.flux_dialog import FluxDialog
from xms.adh.gui.friction_dialog import FrictionDialog
from xms.adh.gui.populate_flow_data import PopulateFlowData
from xms.adh.gui.sediment_diversion_dialog import SedimentDiversionDialog
from xms.adh.gui.transport_constituent_assignment_dialog import TransportConstituentAssignmentDialog


class BcConceptualComponent(AdHComponent):
    """A hidden Dynamic Model Interface (DMI) component for the AdH model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Assign Transport', 'open_transport_assign'),
            ('Display Options...', 'open_display_options'),
        ]
        self.arc_commands = [
            ('Assign Arc Attributes', 'open_arc_attributes'), ('Assign Friction', 'open_friction_attributes'),
            ('Assign Flux', 'open_flux_attributes'), ('Assign Arc Transport', 'open_arc_transport_attributes')
        ]
        #                    ('Assign Sediment Diversion', 'open_sediment_diversion')]
        self.point_commands = [
            ('Assign Point Attributes', 'open_point_attributes'),
            ('Assign Point Transport', 'open_point_transport_attributes')
        ]
        self.data = BcIO(self.main_file)
        if os.path.exists(self.main_file):
            self.cov_uuid = self.data.cov_uuid

        comp_dir = os.path.dirname(self.main_file)
        self.arc_disp_opts_file = os.path.join(comp_dir, 'bc_arc_display_options.json')
        self.point_disp_opts_file = os.path.join(comp_dir, 'bc_point_display_options.json')

        self._initialize_display_options()

    def _initialize_display_options(self):
        """Initializes or updates the display options and saves the UUID to the main file."""
        if not os.path.exists(self.arc_disp_opts_file):
            self._create_and_save_display_options()
        else:
            self._update_display_options()

    def _create_and_save_display_options(self):
        """Creates and saves display options for both arc and point categories with randomized UUIDs."""
        # Arc display options
        arc_categories = self._create_display_options(arc_default_display_options())
        write_display_options_to_json(self.arc_disp_opts_file, arc_categories)
        self.data.arc_display_uuid = arc_categories.uuid

        # Point display options
        point_categories = self._create_display_options(point_default_display_options())
        write_display_options_to_json(self.point_disp_opts_file, point_categories)
        self.data.point_display_uuid = point_categories.uuid

        self.data.commit()

    def _create_display_options(self, default_options: dict):
        """Creates a display option list from default options.

        Args:
            default_options: The default display options.
        """
        categories = CategoryDisplayOptionList()
        categories.from_dict(default_options)
        categories.comp_uuid = self.uuid
        return categories

    def _update_display_options(self):
        """Updates specific parts of the existing display options."""
        arc_json_dict = read_display_options_from_json(self.arc_disp_opts_file)
        arc_categories = CategoryDisplayOptionList()
        arc_categories.from_dict(arc_json_dict)
        for category in arc_categories.categories:
            if category.description == "Flow (NB VEL)":
                category.description = "Flow (NB OVL)"
        write_display_options_to_json(self.arc_disp_opts_file, arc_categories)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            new_bc_comp = BcConceptualComponent(new_main_file)

            # Correct the component UUID and change the display UUID to something new
            json_dict = fix_duplicated_display_opts(new_path, new_bc_comp.arc_disp_opts_file)
            new_bc_comp.data.info.attrs['arc_display_uuid'] = json_dict['uuid']
            json_dict = fix_duplicated_display_opts(new_path, new_bc_comp.point_disp_opts_file)
            new_bc_comp.data.info.attrs['point_display_uuid'] = json_dict['uuid']
            new_bc_comp.data.info.attrs['cov_uuid'] = ""

            new_bc_comp.data.commit()
        return new_main_file, messages, action_requests

    def project_open_event(self, _new_path):
        """Called when the XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            _new_path (str): Path to the new save location.

        """
        return  # This was a legacy thing, but we didn't have a release until this became unnecessary.

    def create_event(self, _lock_state):
        """This will be called when the component is created from nothing.

        Args:
            _lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        messages = []
        action_requests = [self.get_display_refresh_action()]
        return messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        if not self.cov_uuid:
            self.cov_uuid = query.parent_item_uuid()
            self.data.cov_uuid = self.cov_uuid

        is_initial = False
        point_comp_ids = []

        initial_att_file = str(os.path.join(os.path.dirname(self.main_file), BC_ARC_INITIAL_ATT_ID_FILE))
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            is_initial = True
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = str(os.path.join(os.path.dirname(self.main_file), BC_ARC_INITIAL_COMP_ID_FILE))
            comp_ids = read_display_option_ids(initial_comp_file)
            io_util.remove(initial_att_file)
            io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.arc, att_id, comp_id)

        initial_att_file = os.path.join(os.path.dirname(self.main_file), BC_POINT_INITIAL_ATT_ID_FILE)

        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            is_initial = True
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), BC_POINT_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            point_comp_ids = comp_ids
            io_util.remove(initial_att_file)
            io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.point, att_id, comp_id)

        if is_initial:
            self._update_display_id_files(point_comp_ids)

        self.data.commit()
        # Send the default display list to XMS.
        self.display_option_list.append(
            XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        self.display_option_list.append(
            XmsDisplayMessage(file=self.point_disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        return [], []

    def _update_display_id_files(self, point_comp_ids):
        """Updates all of the display id files.

        Args:
            point_comp_ids (list): A list of all point component ids.
        """
        all_comp_ids = self.data.comp_id_to_ids.COMP_ID.tolist()
        all_bc_ids = self.data.comp_id_to_ids.BC_ID.tolist()
        bc_to_comp_ids = defaultdict(list)
        for comp_id, bc_id in zip(all_comp_ids, all_bc_ids):
            if bc_id > 0:
                bc_to_comp_ids[bc_id].append(comp_id)
        arc_card_to_comp_ids = {card: [] for card in CardInfo.arc_card_to_id_file.keys()}
        point_card_to_comp_ids = {card: [] for card in CardInfo.point_card_to_id_file.keys()}
        for bc_id, bc_comp_ids in bc_to_comp_ids.items():
            if not bc_comp_ids:
                continue
            nb_out = self.data.nb_out.loc[self.data.nb_out['BC_ID'] == bc_id]
            if not nb_out.empty:
                arc_card_to_comp_ids['OUT outflow'].append(nb_out['OUT_COMP_ID'])
                arc_card_to_comp_ids['OUT inflow'].append(nb_out['IN_COMP_ID'])
            else:
                sdr_df = self.data.bc.stage_discharge_boundary
                sdr = sdr_df.loc[sdr_df['S_ID'] == bc_id]
                if not sdr.empty:
                    arc_card_to_comp_ids['SDR'].extend(bc_comp_ids)
                else:
                    bc_df = self.data.bc.solution_controls
                    bc = bc_df.loc[bc_df['STRING_ID'] == bc_id]
                    if not bc.empty:
                        card_1 = bc.CARD.iloc[0].upper()
                        card_2 = bc.CARD_2.iloc[0].upper()
                        if card_1 == 'OFF':
                            display_card = card_1
                        else:
                            display_card = card_2
                        if bc_comp_ids[0] in point_comp_ids:
                            if display_card not in CardInfo.point_card_to_id_file:
                                continue
                            point_card_to_comp_ids[display_card].extend(bc_comp_ids)
                        else:
                            if display_card not in CardInfo.arc_card_to_id_file:
                                continue
                            display_card = 'VEL' if card_1 == 'NB' and card_2 == 'OVL' else display_card
                            arc_card_to_comp_ids[display_card].extend(bc_comp_ids)
        for card, id_file in CardInfo.arc_card_to_id_file.items():
            write_display_option_ids(id_file, arc_card_to_comp_ids[card])
        for card, id_file in CardInfo.point_card_to_id_file.items():
            write_display_option_ids(id_file, point_card_to_comp_ids[card])

    def open_arc_attributes(self, query, params, win_cont):
        """Opens the arc attribute dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)
        populate_flow_data = None

        num_selected = len(att_ids)
        allow_nb_out = num_selected == 2
        dlg = BcArcDialog(win_cont, 'Arc Attributes')

        if num_selected == 1:
            populate_flow_data = PopulateFlowData()
            populate_flow_data.pe_tree = query.project_tree
            populate_flow_data.set_query(query)
            populate_flow_data.set_arc_id(att_ids[0])
        dlg.set_populate_flow_data(populate_flow_data)

        bc_comp_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
        bc_ids = bc_comp_id_result.loc[bc_comp_id_result['BC_ID'] > 0].BC_ID.tolist()

        bc_id_result = self.data.bc.solution_controls.loc[self.data.bc.solution_controls['STRING_ID'].isin(bc_ids)]
        nb_out_id_result = self.data.nb_out.loc[self.data.nb_out['BC_ID'].isin(bc_ids)]
        nb_sdr_result = \
            self.data.bc.stage_discharge_boundary.loc[self.data.bc.stage_discharge_boundary['S_ID'].isin(bc_ids)]

        nb_sdr_result_rows = len(nb_sdr_result.index)
        bc_id_result_rows = len(bc_id_result.index)
        nb_out_id_result_rows = len(nb_out_id_result.index)
        is_mixed = (nb_sdr_result_rows + bc_id_result_rows + nb_out_id_result_rows) > 1

        # check if in SDR table
        is_sdr = nb_sdr_result_rows > 0
        if is_sdr:
            card_type = 'SDR'
        elif bc_id_result_rows > 0:
            card_type = bc_id_result.iloc[0].CARD_2
        else:
            card_type = ''

        inflow_nb_out_comp_id = 0
        outflow_nb_out_comp_id = 0
        if nb_out_id_result_rows == 1:
            inflow_nb_out_comp_id = nb_out_id_result.iloc[0].IN_COMP_ID
            outflow_nb_out_comp_id = nb_out_id_result.iloc[0].OUT_COMP_ID
        inflow_arc_id = 0
        outflow_arc_id = 0
        comp_to_xms = None
        if num_selected == 1 and inflow_nb_out_comp_id > 0 and outflow_nb_out_comp_id > 0:
            # find the other arc id
            comp_to_xms = query.load_component_ids(self, arcs=True, delete_files=False)
            if 'ARC' in comp_to_xms:
                arc_ids = read_display_option_ids(comp_to_xms['ARC'][0])
                old_comp_ids = read_display_option_ids(comp_to_xms['ARC'][1])
                inflow_idx = old_comp_ids.index(inflow_nb_out_comp_id)
                outflow_idx = old_comp_ids.index(outflow_nb_out_comp_id)
                inflow_arc_id = arc_ids[inflow_idx]
                outflow_arc_id = arc_ids[outflow_idx]
                if inflow_nb_out_comp_id not in comp_ids:
                    att_ids.append(inflow_arc_id)
                    comp_ids.append(inflow_nb_out_comp_id)
                else:
                    att_ids.append(outflow_arc_id)
                    comp_ids.append(outflow_nb_out_comp_id)
            card_type = 'OUT'
            allow_nb_out = True
        elif num_selected == 2 and inflow_nb_out_comp_id > 0 and outflow_nb_out_comp_id > 0 and not is_mixed:
            card_type = 'OUT'
            if inflow_nb_out_comp_id == comp_ids[0]:
                inflow_arc_id = att_ids[0]
                outflow_arc_id = att_ids[1]
            else:
                inflow_arc_id = att_ids[1]
                outflow_arc_id = att_ids[0]
        elif num_selected == 2 and inflow_nb_out_comp_id == 0 and outflow_nb_out_comp_id == 0:
            inflow_arc_id = att_ids[1]
            outflow_arc_id = att_ids[0]
        dlg.set_arc_type(card_type, is_mixed, allow_nb_out, outflow_arc_id, inflow_arc_id)
        if is_sdr:
            coef_a = nb_sdr_result.iloc[0][2]
            coef_b = nb_sdr_result.iloc[0][3]
            coef_c = nb_sdr_result.iloc[0][4]
            coef_d = nb_sdr_result.iloc[0][5]
            coef_e = nb_sdr_result.iloc[0][6]
            dlg.set_sdr_data(coef_a, coef_b, coef_c, coef_d, coef_e)
        else:
            self._set_series_and_snap_for_dialog(
                dlg, bc_id_result, bc_id_result_rows, nb_out_id_result, nb_out_id_result_rows
            )
        if dlg.exec():
            new_card_type = dlg.get_arc_card_type()
            swap_out = dlg.get_out_assignment_changed()
            new_bc_id = self.data.get_next_bc_id()
            att_id_to_new_comp_id = self._update_component_ids(att_ids, comp_ids, 'BC_ID', new_bc_id, TargetType.arc)
            if new_card_type == 'SDR':
                self._save_sdr(dlg, new_bc_id, att_id_to_new_comp_id)
            elif new_card_type == 'OUT':
                self._save_nb_out(dlg, new_bc_id, inflow_arc_id, outflow_arc_id, swap_out, att_id_to_new_comp_id)
            else:  # elif new_card_type != 'OFF' and new_card_type is not None:
                self._save_other_arc_card(dlg, new_card_type, new_bc_id, att_id_to_new_comp_id)
            self.data.commit()
        if comp_to_xms:
            self.remove_id_files(comp_to_xms)

        return [], []

    def _set_series_and_snap_for_dialog(
        self, dialog, bc_id_result, bc_id_result_rows, nb_out_id_result, nb_out_id_result_rows
    ):
        """Sets the time series used by the card into the dialog. Also sets the current snapping if applicable.

        Args:
            dialog (BcArcDialog): The dialog that will be shown to the user.
            bc_id_result (DataFrame): The solution control table rows that are applicable.
            bc_id_result_rows (int): The number of solution control table rows that are applicable.
            nb_out_id_result (DataFrame): The nb out table rows that are applicable.
            nb_out_id_result_rows (int): The number of nb out table rows that are applicable.
        """
        if bc_id_result_rows > 0:
            old_comp_id = int(bc_id_result.iloc[0][2])  # 2 is the component id number for all DB cards
            snap_result = self.data.snap_types.loc[self.data.snap_types['ID'] == old_comp_id]['SNAP']
            if not snap_result.empty:
                snap_type = snap_result.iloc[0]
                dialog.set_snapping_type(snap_type)
        series_list = []
        data_col = 3
        for _ in range(dialog.get_number_of_series()):
            series_id = None
            if bc_id_result_rows > 0:
                series_id = bc_id_result.iloc[0][data_col]
            elif nb_out_id_result_rows > 0:
                series_id = nb_out_id_result.iloc[0].SERIES_ID

            if series_id and series_id in self.data.bc.time_series:
                data_for_dlg = self.data.bc.time_series[series_id]
                series_list.append(data_for_dlg)
            data_col += 1
        dialog.set_series(series_list)
        dialog.set_sdr_data(0.0, 0.0, 0.0, 0.0, 0.0)

    def _save_nb_out(self, dialog, new_bc_id, inflow_arc_id, outflow_arc_id, swap_out, att_id_to_new_comp_id):
        """Gets an NB OUT card ready to save on commit.

        Args:
            dialog (BcArcDialog): The dialog that has the data to save.
            new_bc_id (int): The new boundary condition id.
            inflow_arc_id (int): The arc feature id for the inflow.
            outflow_arc_id (int): The arc feature id for the outflow.
            swap_out (bool): True if the inflow and outflow arc ids need to be swapped.
            att_id_to_new_comp_id (dict): A dictionary of attribute ids to new component ids.
        """
        # create new row in nb out tables
        if swap_out:
            temp = inflow_arc_id
            inflow_arc_id = outflow_arc_id
            outflow_arc_id = temp
        new_comp_id_outflow = att_id_to_new_comp_id[outflow_arc_id]
        new_comp_id_inflow = att_id_to_new_comp_id[inflow_arc_id]
        if self.data.bc.time_series.keys():
            new_key = max(self.data.bc.time_series.keys()) + 1
        else:
            new_key = 1
        values = [new_bc_id, new_comp_id_outflow, new_comp_id_inflow, new_key]
        self.data.bc.time_series[new_key] = dialog.get_series(0)
        df2 = pd.DataFrame([values], columns=['BC_ID', 'OUT_COMP_ID', 'IN_COMP_ID', 'SERIES_ID'])
        self.data.nb_out = pd.concat([self.data.nb_out, df2])
        self.data.nb_out = self.data.nb_out.astype(dtype='Int64')
        out_id_file = CardInfo.arc_card_to_id_file['OUT outflow']
        in_id_file = CardInfo.arc_card_to_id_file['OUT inflow']
        self._append_comp_id_to_file(out_id_file, new_comp_id_outflow, self.arc_disp_opts_file)
        self._append_comp_id_to_file(in_id_file, new_comp_id_inflow, self.arc_disp_opts_file)

    def _determine_card_and_snap_type(self, new_card_type: str, new_bc_id: int, dialog: BcArcDialog) -> str:
        """Determines the first card in solution controls and adds snap type if applicable.

        Args:
            new_card_type: The card type the user selected in the dialog.
            new_bc_id: The new boundary condition id.
            dialog: The dialog that has the data to save.

        Returns:
            A string containing the first card to put in solution controls.
        """
        if new_card_type == 'OF':
            return 'OB'
        elif new_card_type in ['OVH', 'LDE', 'LDH', 'LID']:
            snap_type = dialog.get_snapping_type()
            snap_df = pd.DataFrame([[int(new_bc_id), snap_type]], columns=['ID', 'SNAP'])
            self.data.snap_types = pd.concat([self.data.snap_types, snap_df])
            return 'DB'
        elif not new_card_type:
            return ''
        return 'NB'

    def _generate_series_keys(self, dialog: BcArcDialog, series_count: int) -> list:
        """Generates xy series keys.

        Args:
            dialog: The dialog that has the data to save.
            series_count: The number of possible series for the current card index.

        Returns:
            A list containing the xy series keys to append to solution controls.
        """
        series_keys = []
        # there are only 3 xy series columns possible
        for series_idx in range(3):
            if series_idx >= series_count:
                series_keys.append(-1)
                continue

            new_series = dialog.get_series(series_idx)
            if new_series:
                new_key = max(self.data.bc.time_series.keys(), default=0) + 1
                self.data.bc.time_series[new_key] = new_series
                series_keys.append(new_key)
            else:
                series_keys.append(-1)
        return series_keys

    def _save_other_arc_card(self, dialog, new_card_type, new_bc_id, att_id_to_new_comp_id):
        """Gets an arc card ready to save on commit.

        Args:
            dialog (BcArcDialog): The dialog that has the data to save.
            new_card_type (str): The card type the user selected in the dialog.
            new_bc_id (int): The new boundary condition id.
            att_id_to_new_comp_id (dict): A dictionary of attribute ids to new component ids.
        """
        first_card = self._determine_card_and_snap_type(new_card_type, new_bc_id, dialog)
        series_count = dialog.get_number_of_series()
        series_keys = self._generate_series_keys(dialog, series_count)

        values = [first_card, new_card_type, new_bc_id] + series_keys
        # create the dataframe to append
        df2 = pd.DataFrame([values], columns=['CARD', 'CARD_2', 'STRING_ID', 'XY_ID_0', 'XY_ID_1', 'XY_ID_2'])
        self.data.bc.solution_controls = pd.concat([self.data.bc.solution_controls, df2])

        self._set_solution_control_type()

        id_file = CardInfo.arc_card_to_id_file[new_card_type]
        for new_comp_id in att_id_to_new_comp_id.values():
            self._append_comp_id_to_file(id_file, new_comp_id, self.arc_disp_opts_file)

    def _save_sdr(self, dialog, new_bc_id, att_id_to_new_comp_id):
        """Gets the SDR card ready to save on commit.

        Args:
            dialog (BcArcDialog): The dialog that has the data to save.
            new_bc_id (int): The new boundary condition id.
            att_id_to_new_comp_id (dict): A dictionary of attribute ids to new component ids.
        """
        # create new row in stage discharge boundary
        values = ['NB SDR', new_bc_id]
        values.extend(dialog.get_sdr_data())
        df2 = pd.DataFrame([values], columns=['CARD', 'S_ID', 'COEF_A', 'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E'])
        self.data.bc.stage_discharge_boundary = pd.concat([self.data.bc.stage_discharge_boundary, df2])
        self.data.bc.stage_discharge_boundary['S_ID'] = \
            self.data.bc.stage_discharge_boundary['S_ID'].astype(dtype='Int64')
        id_file = CardInfo.arc_card_to_id_file['SDR']
        for new_comp_id in att_id_to_new_comp_id.values():
            self._append_comp_id_to_file(id_file, new_comp_id, self.arc_disp_opts_file)

    def _update_component_ids(self, att_ids, comp_ids, id_label, new_id, target):
        """Updates the component to id dataframe and creates new component ids.

        Args:
            att_ids (list): A list of attribute ids.
            comp_ids (list): A list of component ids.
            id_label (str): The column label in the comp_id_to_ids dataframe to set with the new id.
            new_id (int): The new id.
            target (TargetType): The target type of the attribute ids.

        Returns:
            A dictionary of feature attribute ids to new component ids.
        """
        new_comp_id_rows = []
        id_column = self.data.comp_id_to_ids.columns.get_loc(id_label)
        empty_row = [new_id if col == id_column else -1 for col in range(1, len(self.data.comp_id_to_ids.columns))]
        att_id_to_new_comp_id = {}
        for att_id, comp_id in zip(att_ids, comp_ids):
            new_comp_id = self.data.get_next_comp_id()
            att_id_to_new_comp_id[att_id] = new_comp_id
            old_row = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'] == comp_id]
            if old_row.empty:
                append_row = [new_comp_id]
                append_row.extend(empty_row)
            else:
                # Make a copy of the old values and update the component id and the other id.
                old_values = old_row.iloc[0].tolist()
                old_values[0] = new_comp_id
                old_values[id_column] = new_id
                append_row = old_values
            new_comp_id_rows.append(append_row)
            self.update_component_id(target, att_id, int(new_comp_id))
        self.data.comp_id_to_ids = \
            pd.concat([self.data.comp_id_to_ids, pd.DataFrame(
                new_comp_id_rows, columns=self.data.comp_id_columns)], ignore_index=True)
        self.data.comp_id_to_ids = self.data.comp_id_to_ids.astype(dtype='Int64')
        return att_id_to_new_comp_id

    def _append_comp_id_to_file(self, id_file, new_comp_id, disp_file):
        """Appends component id to the display id file.

        Args:
            id_file (str): Relative path to the id file.
            new_comp_id (int): The new component id to add to the id file.
            disp_file (str): The display options file.
        """
        updated_comp_ids = read_display_option_ids(id_file)
        updated_comp_ids.append(new_comp_id)
        write_display_option_ids(id_file, updated_comp_ids)
        if disp_file:
            self.display_option_list.append(
                XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
            )

    def clean_attributes(self):
        """Removes unused attributes."""
        if self.cov_uuid not in self.comp_to_xms:
            return
        comp_ids = []
        if int(TargetType.arc) in self.comp_to_xms[self.cov_uuid]:
            comp_ids.extend(list(self.comp_to_xms[self.cov_uuid][int(TargetType.arc)].keys()))
        if int(TargetType.point) in self.comp_to_xms[self.cov_uuid]:
            comp_ids.extend(list(self.comp_to_xms[self.cov_uuid][int(TargetType.point)].keys()))
        comp_id_result = self.data.comp_id_to_ids.loc[~self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]

        drop_list = []
        for idx, _ in comp_id_result.iterrows():
            drop_list.append(idx)

        # remove duplicate ids
        drop_list = list(set(drop_list))
        self.data.comp_id_to_ids.drop(drop_list, axis=0, inplace=True)

        self._clean_bc()
        self._clean_friction()
        self._clean_flux()
        self._clean_transport()
        self._clean_sediment_diversions()

        used_time_series = self._get_used_series_ids()

        # Remove unused time series.
        series_ids = list(self.data.bc.time_series.keys())
        for series_id in series_ids:
            if series_id not in used_time_series:
                self.data.bc.time_series.pop(int(series_id))

    def _clean_bc(self):
        """Cleans the boundary condition tables."""
        bc_ids = self.data.comp_id_to_ids['BC_ID'].tolist()

        # Find items that have bc ids that are no longer used.
        self._drop_rows(self.data.bc.solution_controls, bc_ids, 'STRING_ID')
        self._drop_rows(self.data.bc.stage_discharge_boundary, bc_ids, 'S_ID')
        self._drop_rows(self.data.nb_out, bc_ids, 'BC_ID')

    def _get_used_series_ids(self):
        """Gets the times series ids that are currently used.

        Returns:
            A list of series ids that are currently in use.
        """
        used_time_series = self.data.nb_out.SERIES_ID.tolist()
        db_nb_rows = self.data.bc.solution_controls.loc[self.data.bc.solution_controls['CARD'].isin(['DB', 'NB'])]
        db_nb_rows = db_nb_rows.loc[db_nb_rows['CARD_2'] != 'TID']
        ovh_rows = db_nb_rows.loc[db_nb_rows['CARD_2'] == 'OVH']
        ovl_rows = db_nb_rows.loc[db_nb_rows['CARD_2'] == 'OVL']
        used_time_series.extend(db_nb_rows.XY_ID_0.tolist())
        used_time_series.extend(ovh_rows.XY_ID_1.tolist())
        used_time_series.extend(ovh_rows.XY_ID_2.tolist())
        used_time_series.extend(ovl_rows.XY_ID_1.tolist())
        used_time_series.extend(self.data.transport_assignments['SERIES_ID'].tolist())
        used_time_series.extend(self.data.sediment_assignments['SERIES_ID'].tolist())
        return used_time_series

    def _clean_friction(self):
        """Cleans the friction tables."""
        friction_ids = self.data.comp_id_to_ids['FRICTION_ID'].tolist()

        self._drop_rows(self.data.bc.friction_controls, friction_ids, 'STRING_ID')

    def _clean_flux(self):
        """Cleans the flux tables."""
        flux_ids = self.data.comp_id_to_ids['FLUX_ID'].tolist()

        self._drop_rows(self.data.flux, flux_ids, 'ID')

    def _clean_transport(self):
        """Cleans the transport tables."""
        transport_ids = self.data.comp_id_to_ids['TRANSPORT_ID'].tolist()

        # Find items that have transport ids that are no longer used.
        self._drop_rows(self.data.transport_assignments, transport_ids, 'TRAN_ID')
        self._drop_rows(self.data.sediment_assignments, transport_ids, 'TRAN_ID')
        self._drop_rows(self.data.uses_transport, transport_ids, 'TRAN_ID')
        self._drop_rows(self.data.uses_sediment, transport_ids, 'TRAN_ID')

    def _clean_sediment_diversions(self):
        """Cleans the sediment diversion tables."""
        diversion_ids = self.data.comp_id_to_ids['DIVERSION_ID'].tolist()

        # Find items that have sediment diversion ids that are no longer used.
        self._drop_rows(self.data.sediment_diversions, diversion_ids, 'DIV_ID')

    def _update_old_comp_id_display_list(self, att_id_to_new_comp_id, att_ids, comp_ids, for_arcs):
        """Update display list id file when assigning a new component id.

        Args:
            att_id_to_new_comp_id (dict): Mapping of att id to new comp id
            att_ids (list): List of att ids being updated
            comp_ids (list): List of the old comp ids, parallel with att_ids
            for_arcs (bool): True if BC being updated is an arc, False if a point
        """
        id_files = CardInfo.arc_card_to_id_file.values() if for_arcs else CardInfo.point_card_to_id_file.values()
        disp_opts_file = self.arc_disp_opts_file if for_arcs else self.point_disp_opts_file
        for att_id, new_comp_id in att_id_to_new_comp_id.items():
            try:
                old_comp_id_idx = att_ids.index(att_id)
            except ValueError:
                continue
            old_comp_id = comp_ids[old_comp_id_idx]
            for id_file in id_files:
                comp_ids_of_type = read_display_option_ids(os.path.join(os.path.dirname(self.main_file), id_file))
                if old_comp_id in comp_ids_of_type:
                    self._append_comp_id_to_file(id_file, new_comp_id, disp_opts_file)
                    break

    @staticmethod
    def _drop_rows(df, ids, id_label):
        """Gets the rows to drop in the dataframe.

        Args:
            df (Dataframe): The dataframe to look in.
            ids (list): The ids that are still valid. All other ids should be dropped.
            id_label (str): The column label in the dataframe to look in.
        """
        id_result = df.loc[~df[id_label].isin(ids)]
        drop_list = []
        for idx, _ in id_result.iterrows():
            drop_list.append(idx)

        # remove duplicate ids
        drop_list = list(set(drop_list))
        df.drop(drop_list, axis=0, inplace=True)

    def open_point_attributes(self, query, params, win_cont):
        """Opens the Assign Obstruction dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all
        att_ids, comp_ids = self._get_att_and_comp_ids(params)

        # Get the transport ids from the component ids.
        comp_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
        bc_ids = comp_id_result.loc[comp_id_result['BC_ID'] > 0].BC_ID.tolist()

        if 'node_id_files' in params and params['node_id_files']:
            att_ids.extend(read_display_option_ids(params['node_id_files'][0]))
            comp_ids.extend(read_display_option_ids(params['node_id_files'][1]))
        elif 'node_selection' in params and params['node_selection']:
            att_ids.extend(params['node_selection'])
            comp_ids.extend([-1 for _ in att_ids])
        if len(att_ids) != len(comp_ids) or len(comp_ids) != 1:
            return [('ERROR', 'Can only assign point attributes to only one point.')], []

        dlg = BcPointDialog(win_cont, 'Point Attributes')
        comp_id_result = \
            self.data.bc.solution_controls.loc[self.data.bc.solution_controls['CARD'].isin(['DB']) &  # noqa: W504
                                               self.data.bc.solution_controls['STRING_ID'].isin(bc_ids)]

        comp_id_result_rows = len(comp_id_result.index)
        card_type = ''
        if comp_id_result_rows > 0:
            card_type = comp_id_result['CARD_2'].iloc[0]

        is_mixed = comp_id_result_rows > 1

        dlg.set_point_type(card_type, is_mixed)

        series_list = []
        data_col = 3
        for _ in range(dlg.get_number_of_series()):
            series_id = None
            if comp_id_result_rows > 0:
                series_id = comp_id_result.iloc[0][data_col]

            data_for_dlg = None
            if series_id and series_id in self.data.bc.time_series:
                data_for_dlg = self.data.bc.time_series[series_id]
            series_list.append(data_for_dlg)
            data_col += 1
        dlg.set_series(series_list)
        dlg.set_units_for_series()
        if dlg.exec():
            # remove all old rows in tables, remove old time series
            new_card_type = dlg.get_point_card_type()
            series_count = dlg.get_number_of_series()

            # create a new row in solution controls
            new_bc_id = self.data.get_next_bc_id()
            first_card = 'DB'
            # Added to allow the user to set a point to be off
            if new_card_type is None:
                new_card_type = ''
                first_card = ''
            values = [first_card, new_card_type, new_bc_id]
            # there are only 3 more columns possible
            for series_idx in range(3):
                if series_idx >= series_count or new_card_type == '':
                    # values.append(None)
                    values.append(-1)
                    continue

                if self.data.bc.time_series.keys():
                    new_key = max(self.data.bc.time_series.keys()) + 1
                else:
                    new_key = 1
                new_series = dlg.get_series(series_idx)
                if new_series:
                    if new_card_type == 'WND':
                        new_series.series_type = 'SERIES WIND'
                    else:
                        new_series.series_type = 'SERIES BC'
                    self.data.bc.time_series[new_key] = new_series
                values.append(new_key)
            df2 = pd.DataFrame([values], columns=["CARD", "CARD_2", "STRING_ID", "XY_ID_0", "XY_ID_1", "XY_ID_2"])
            self.data.bc.solution_controls = pd.concat([self.data.bc.solution_controls, df2], ignore_index=True)
            self._set_solution_control_type()
            att_id_to_new_comp_id = self._update_component_ids(att_ids, comp_ids, 'BC_ID', new_bc_id, TargetType.point)

            for new_comp_id in att_id_to_new_comp_id.values():
                self._append_comp_id_to_file(
                    CardInfo.point_card_to_id_file[new_card_type], new_comp_id, self.point_disp_opts_file
                )
            self.data.commit()
        return [], []

    def _set_solution_control_type(self):
        """Sets the column types of the solution controls table to be integer."""
        self.data.bc.solution_controls['STRING_ID'] = \
            self.data.bc.solution_controls['STRING_ID'].astype(dtype='Int64')
        self.data.bc.solution_controls['XY_ID_0'] = \
            self.data.bc.solution_controls['XY_ID_0'].astype(dtype='Int64')
        self.data.bc.solution_controls['XY_ID_1'] = \
            self.data.bc.solution_controls['XY_ID_1'].astype(dtype='Int64')
        self.data.bc.solution_controls['XY_ID_2'] = \
            self.data.bc.solution_controls['XY_ID_2'].astype(dtype='Int64')

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for boundary conditions."""
        # Read the arc display options
        arc_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        json_dict = read_display_options_from_json(self.arc_disp_opts_file)
        arc_categories.from_dict(json_dict)

        # Read the default point display options, and save ourselves a copy with a randomized UUID.
        point_categories = CategoryDisplayOptionList()
        point_json_dict = read_display_options_from_json(self.point_disp_opts_file)
        point_categories.from_dict(point_json_dict)

        dlg = CategoryDisplayOptionsDialog([arc_categories, point_categories], win_cont)
        dlg.setWindowIcon(QIcon(get_xms_icon()))
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.arc_disp_opts_file
                if category_list.target_type == TargetType.point:
                    disp_file = self.point_disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    def add_selection_menu_item(self, command_text, command_method, param_dict, menu_list):
        """Adds a menu item to the list with stored selection data.

        Args:
            command_text (str): The menu item text the user will see.
            command_method (str): The method of this component for the action request.
            param_dict (dict): A dictionary of parameters for the function in the action request.
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        action = ActionRequest(
            modality='MODAL',
            method_name=command_method,
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            parameters=param_dict
        )
        menu_item = MenuItem(text=command_text, action=action)
        menu_list.append(menu_item)

    def clean_all(self, query, params):
        """Query XMS for a dump of all the current component ids of the specified entity type.

        Query needs to be at the Component Context level (or any
        other node with a "ComponentCoverageIds" ContextDefinition out edge).

        Args:
            query (xmsdmi.dmi.Query): Query for communicating with XMS.
            params (:obj:`dict'): Generic map of parameters.

        Returns:
            (dict): Key is stringified entity type and value is tuple of xms and component id files.
        """
        query.load_component_ids(self, arcs=True, points=True)
        self.clean_attributes()
        self.data.commit()
        return [], []

    def open_friction_attributes(self, query, params, win_cont):
        """Opens a dialog for editing friction options.

        Args:
            query (Query): A way to communicate with SMS.
            params (list): Arguments that this function was called with.
            win_cont (QWidget): A parent window container.

        Returns:
            A tuple of a list of messages and a list of action requests.
        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)
        fric_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
        fric_ids = fric_id_result.loc[fric_id_result['FRICTION_ID'] > 0]

        card_type = ''
        is_mid = False
        real_1 = None
        real_2 = None
        real_3 = None
        real_4 = None
        real_5 = None
        # fric_idx = []
        if not fric_ids.empty:
            old_fric_id = fric_ids['FRICTION_ID'].iloc[0]
            row_data = \
                self.data.bc.friction_controls.loc[self.data.bc.friction_controls['STRING_ID'] == old_fric_id].iloc[0]
            card_type = row_data.CARD_2
            is_mid = int(float(row_data.REAL_05) + 0.5) != 0
            real_1 = float(row_data.REAL_01)
            real_2 = float(row_data.REAL_02)
            real_3 = float(row_data.REAL_03)
            real_4 = float(row_data.REAL_04)
            real_5 = float(row_data.REAL_05)

        dlg = FrictionDialog(win_cont, 'Friction')
        dlg.friction.set_friction_type(card_type, is_mid, [real_1, real_2, real_3, real_4, real_5])
        if dlg.exec():
            fric_type = dlg.friction.get_friction_type()
            if fric_type is None:
                fric_type = ''
            fric_options = dlg.friction.get_friction_values()
            fric_is_mid = dlg.friction.get_friction_is_mid()
            # change from a bool to an int
            if fric_is_mid:
                fric_is_mid = 1
            else:
                fric_is_mid = 0

            # add the card
            new_fric_id = self.data.get_next_friction_id()
            values = ['FR', fric_type, new_fric_id]
            values.extend(fric_options)
            # 8 columns in the DataFrame, need to fill blank columns
            for _ in range(7 - len(values)):
                values.append(None)
            values.append(fric_is_mid)
            df2 = pd.DataFrame(
                [values],
                columns=["CARD", "CARD_2", "STRING_ID", "REAL_01", "REAL_02", "REAL_03", "REAL_04", "REAL_05"]
            )
            self.data.bc.friction_controls = pd.concat([self.data.bc.friction_controls, df2])

            att_id_to_new_comp_id = self._update_component_ids(
                att_ids, comp_ids, 'FRICTION_ID', new_fric_id, TargetType.arc
            )
            self._update_old_comp_id_display_list(att_id_to_new_comp_id, att_ids, comp_ids, True)
            self.data.commit()
        return [], []

    def open_flux_attributes(self, query, params, win_cont):
        """Opens a dialog to allow the user to assign this arc as a flux string and how it should snap.

        Args:
            query (Query): A way to communicate with SMS.
            params (list): Arguments that this function was called with.
            win_cont (QWidget): A parent window container.

        Returns:
            A tuple of a list of messages and a list of action requests.
        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)
        flux_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
        flux_ids = flux_id_result.loc[flux_id_result['FLUX_ID'] > 0]

        is_flux = False
        is_edge = False
        is_mid = False
        if not flux_ids.empty:
            old_flux_id = flux_ids['FLUX_ID'].iloc[0]
            flux_row = self.data.flux.loc[self.data.flux['ID'] == old_flux_id].iloc[0]
            is_flux = flux_row.IS_FLUX
            is_edge = flux_row.EDGESTRING
            is_mid = flux_row.MIDSTRING
        dialog = FluxDialog(win_cont)
        dialog.set_flux(is_flux, is_edge, is_mid)
        if dialog.exec_():
            is_flux, is_edge, is_mid = dialog.get_flux()
            new_flux_id = self.data.get_next_flux_id()
            values = [new_flux_id, is_flux, is_edge, is_mid]
            # Add the values to the dataframe and force the values to be integers.
            df2 = pd.DataFrame([values], columns=['ID', 'IS_FLUX', 'EDGESTRING', 'MIDSTRING'])
            self.data.flux = pd.concat([self.data.flux, df2])
            self.data.flux = self.data.flux.astype(dtype='Int64')
            att_id_to_new_comp_id = self._update_component_ids(
                att_ids, comp_ids, 'FLUX_ID', new_flux_id, TargetType.arc
            )
            self._update_old_comp_id_display_list(att_id_to_new_comp_id, att_ids, comp_ids, True)
            self.data.commit()
        return [], []

    def open_transport_assign(self, query, params, win_cont):
        """Opens the transport constituent assignment dialog.

        Args:
            query (Query): A way to communicate with XMS.
            params (list): A list that should be empty.
            win_cont (QWidget): A parent container for the dialog.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # get the project explorer
        pe_tree = query.project_tree

        transport_comp, transport_name, use_transport = self._get_transport(pe_tree, False)
        sediment_comp, sediment_name, use_sediment = self._get_transport(pe_tree, True)

        dialog = TransportConstituentAssignmentDialog(win_cont, 'Select Transport Constituents', pe_tree)
        dialog.transport.set_transport(self.data.transport_uuid, transport_comp, transport_name=transport_name)
        dialog.sediment.set_transport(self.data.sediment_uuid, sediment_comp, transport_name=sediment_name)
        if dialog.exec_():
            changed = False
            remove_time_series = []
            if dialog.transport.constituents_uuid != self.data.transport_uuid:
                self.data.transport_uuid = dialog.transport.constituents_uuid
                changed = True

                # Find time series to remove.
                remove_time_series.extend(self.data.transport_assignments['SERIES_ID'].tolist())

                # Clear all of the old transport assignments.
                self.data.transport_assignments = self.data.transport_assignments[0:0]

            if dialog.sediment.constituents_uuid != self.data.sediment_uuid:
                self.data.sediment_uuid = dialog.sediment.constituents_uuid
                changed = True

                # Find time series to remove.
                remove_time_series.extend(self.data.sediment_assignments['SERIES_ID'].tolist())

                # Clear all of the old sediment transport assignments.
                self.data.sediment_assignments = self.data.sediment_assignments[0:0]

            if changed:
                # Remove old time series.
                for series in remove_time_series:
                    del self.data.bc.time_series[series]

                # Save the data.
                self.data.commit()
        return [], []

    def open_arc_transport_attributes(self, query, params, win_cont):
        """Opens the transport attributes dialog for this arc/set of arcs.

        Args:
            query (Query): A way to communicate with XMS.
            params (list): A list that should only have one item in it. The item should contain the selection info.
            win_cont (QWidget): A parent container for the dialog.
        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)

        # get the project explorer
        pe_tree = query.project_tree

        assignments, transport_comp, transport_name, use_transport = self._get_transport_for_component_id(
            comp_ids, pe_tree, False
        )
        sed_assignments, sediment_comp, sediment_name, use_sediment = self._get_transport_for_component_id(
            comp_ids, pe_tree, True
        )

        dlg = TransportConstituentAssignmentDialog(
            win_cont, 'Arc Transport Constituents', pe_tree, use_transport, use_sediment, True, self.data.bc.time_series
        )
        dlg.transport.set_transport(self.data.transport_uuid, transport_comp, assignments, transport_name)
        dlg.sediment.set_transport(self.data.sediment_uuid, sediment_comp, sed_assignments, sediment_name)
        if dlg.exec_():
            self._add_assignment_from_dialog(comp_ids, att_ids, dlg, TargetType.arc)
            self.data.commit()
        return [], []

    def open_point_transport_attributes(self, query, params, win_cont):
        """Opens the transport attributes dialog for this point/set of points.

        Args:
            query (Query): A way to communicate with XMS.
            params (list): A list that should only have one item in it. The item should contain the selection info.
            win_cont (QWidget): A parent container for the dialog.
        """
        # make sure we have a coverage uuid
        if not self.cov_uuid:
            self.cov_uuid = query.parent_item_uuid()
            self.data.cov_uuid = self.cov_uuid

        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)

        # get the project explorer
        pe_tree = query.project_tree

        assignments, transport_comp, transport_name, use_transport = self._get_transport_for_component_id(
            comp_ids, pe_tree, False
        )
        sed_assignments, sediment_comp, sediment_name, use_sediment = self._get_transport_for_component_id(
            comp_ids, pe_tree, True
        )

        dlg = TransportConstituentAssignmentDialog(
            win_cont, 'Point Transport Constituents', pe_tree, use_transport, use_sediment, False,
            self.data.bc.time_series
        )
        dlg.transport.set_transport(self.data.transport_uuid, transport_comp, assignments, transport_name)
        dlg.sediment.set_transport(self.data.sediment_uuid, sediment_comp, sed_assignments, sediment_name)
        if dlg.exec_():
            # Points will always be considered a set.
            self._add_assignment_from_dialog(comp_ids, att_ids, dlg, TargetType.point)
            self.data.commit()
        return [], []

    def open_sediment_diversion(self, query, params, win_cont):
        """Opens the sediment diversion dialog for this arc/set of arcs.

        Args:
            query (Query): A way to communicate with XMS.
            params (list): A list that should only have one item in it. The item should contain the selection info.
            win_cont (QWidget): A parent container for the dialog.
        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        att_ids, comp_ids = self._get_att_and_comp_ids(params)

        dialog = SedimentDiversionDialog(win_cont)
        old_snap, old_top, old_bottom, old_bottom_main, is_diversion = self._get_diversion_for_component_id(comp_ids)
        dialog.set_diversion(old_snap, old_top, old_bottom, old_bottom_main, is_diversion)
        if dialog.exec_():
            self._save_sediment_diversion(att_ids, comp_ids, dialog)
        return [], []

    def _save_sediment_diversion(self, att_ids, comp_ids, dialog):
        """Saves the sediment diversion.

        Args:
            att_ids (list): A list of integer feature attribute ids.
            comp_ids (list): A list of previous integer component ids.
            dialog (SedimentDiversionDialog): The dialog to get sediment diversion data from.
        """
        snap, top, bottom, bottom_main, is_diversion = dialog.get_diversion()

        # Add a new sediment diversion
        if is_diversion:
            new_div_id = self.data.get_next_diversion_id()
            new_diversions = [[new_div_id, snap, top, bottom, bottom_main]]
            new_rows = pd.DataFrame(new_diversions, columns=['DIV_ID', 'SNAPPING', 'TOP', 'BOTTOM', 'BOTTOM_MAIN'])
            self.data.sediment_diversions = pd.concat([self.data.sediment_diversions, new_rows])
            self.data.sediment_diversions.DIV_ID = self.data.sediment_diversions.DIV_ID.astype(int)

            att_id_to_new_comp_id = self._update_component_ids(
                att_ids, comp_ids, 'DIVERSION_ID', new_div_id, TargetType.arc
            )
            self._update_old_comp_id_display_list(att_id_to_new_comp_id, att_ids, comp_ids, True)
        self.data.commit()

    def _get_transport(self, pe_tree, is_sediment):
        """Gets the transport constituents component data.

        Args:
            pe_tree (TreeNode): A root TreeNode of the Project Explorer.
            is_sediment (bool): True is getting a sediment transport constituent instead.

        Returns:
            A tuple with: (a TransportConstituentsIO or SedimentConstituentsIO of the transport constituent component,
            the transport constituent component name as a string, a bool that is True if transport constituents
            are currently used)
        """
        # check for the transport constituents component
        transport_comp = None
        use_transport = False
        transport_name = ''
        if is_sediment:
            attr_uuid = 'sediment_uuid'
        else:
            attr_uuid = 'transport_uuid'
        if self.data.info.attrs[attr_uuid]:
            # get the name of the transport constituent
            tree_node = tree_util.find_tree_node_by_uuid(pe_tree, self.data.info.attrs[attr_uuid])
            transport_name = tree_node.name
            main_file = tree_node.main_file

            # get the transport constituent
            if main_file:
                if is_sediment:
                    transport_comp = SedimentConstituentsIO(main_file)
                else:
                    transport_comp = TransportConstituentsIO(main_file)
        return transport_comp, transport_name, use_transport

    def _get_transport_for_component_id(self, comp_ids, pe_tree, is_sediment):
        """Gets the transport constituents component data.

        Args:
            comp_ids (list): A list of integer component ids.
            pe_tree (TreeNode): A root TreeNode of the Project Explorer.
            is_sediment (bool): True if getting sediment transport component ids.

        Returns:
            A tuple with: (the current transport constituent assignments as a pandas.Dataframe,
            a TransportConstituentsIO or SedimentConstituentsIO of the transport constituent component,
            the transport constituent component name as a string,
            a bool that is True if transport constituents are current used)
        """
        # check for the transport (or sediment transport) constituents component
        transport_comp = None
        assignments = None
        use_transport = False
        transport_name = ''
        if is_sediment:
            attr_uuid = 'sediment_uuid'
        else:
            attr_uuid = 'transport_uuid'
        if self.data.info.attrs[attr_uuid]:
            # get the name of the transport constituent
            tree_node = tree_util.find_tree_node_by_uuid(pe_tree, self.data.info.attrs[attr_uuid])
            transport_name = tree_node.name
            main_file = tree_node.main_file

            # get the transport constituent
            if is_sediment:
                transport_comp = SedimentConstituentsIO(main_file)
            else:
                transport_comp = TransportConstituentsIO(main_file)

            # Get the transport ids from the component ids.
            comp_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
            transport_ids = comp_id_result.loc[comp_id_result['TRANSPORT_ID'] > 0].TRANSPORT_ID.tolist()

            if is_sediment:
                id_result = \
                    self.data.sediment_assignments.loc[self.data.sediment_assignments['TRAN_ID'].isin(transport_ids)]
            else:
                id_result = \
                    self.data.transport_assignments.loc[self.data.transport_assignments['TRAN_ID'].isin(transport_ids)]
            if not id_result.empty:
                tran_id = id_result['TRAN_ID'].iloc[0]
                assignments = id_result.loc[id_result['TRAN_ID'] == tran_id]
                if is_sediment:
                    use_transport_result = self.data.uses_sediment.loc[self.data.uses_sediment['TRAN_ID'] == tran_id]
                    use_transport = use_transport_result['USES_SEDIMENT'] != 0
                else:
                    use_transport_result = self.data.uses_transport.loc[self.data.uses_transport['TRAN_ID'] == tran_id]
                    use_transport = use_transport_result['USES_TRANSPORT'] != 0
        return assignments, transport_comp, transport_name, use_transport

    def _get_diversion_for_component_id(self, comp_ids):
        """Gets the sediment diversion component data that was previously used, if any.

        Args:
            comp_ids (list): A list of integer component ids.

        Returns:
            Returns a tuple with the snapping type ('Edgestring' or 'Midstring'), the top elevation for zone of
            withdrawal, the bottom elevation for zone of withdrawal, the bottom elevation for the main channel, and
            True if values were found. If none is found, then default values are returned.
        """
        div_id_result = self.data.comp_id_to_ids.loc[self.data.comp_id_to_ids['COMP_ID'].isin(comp_ids)]
        div_ids = div_id_result.loc[div_id_result['DIVERSION_ID'] > 0]

        if not div_ids.empty:
            old_div_id = div_ids['DIVERSION_ID'].iloc[0]
            values = self.data.sediment_diversions.loc[self.data.sediment_diversions['DIV_ID'] == old_div_id].iloc[0]
            return values['SNAPPING'], values['TOP'], values['BOTTOM'], values['BOTTOM_MAIN'], True
        return 'Edgestring', 0.0, 0.0, 0.0, False

    def _add_assignment_from_dialog(self, comp_ids, att_ids, dialog, target):
        """Saves transport constituent assignment data from the dialog.

        Args:
            comp_ids (list): A list of component ids from the selection.
            att_ids (list): A list of feature attribute ids from the selection.
            dialog (TransportConstituentAssignmentDialog): The dialog to get data from.
            target (TargetType): The target attribute type.
        """
        new_transport_id = self.data.get_next_transport_id()

        self._remove_old_transport_assignments(dialog)

        new_use_transports = []

        # add in the data from the dialog
        tran_id_list = [new_transport_id for _ in range(len(dialog.transport.assignments.index))]
        assignments = dialog.transport.assignments.copy()
        assignments.insert(0, 'TRAN_ID', tran_id_list)
        self.data.transport_assignments = pd.concat([self.data.transport_assignments, assignments])

        # add the new use transport assignments
        new_use_transport = 1 if dialog.transport.ui.transport_constituents_group.isChecked() else 0
        new_use_transports.append([new_transport_id, new_use_transport])

        new_row = pd.DataFrame(new_use_transports, columns=['TRAN_ID', 'USES_TRANSPORT'])
        self.data.uses_transport = pd.concat([self.data.uses_transport, new_row])

        self.data.transport_assignments.TRAN_ID = self.data.transport_assignments.TRAN_ID.astype(int)
        self.data.transport_assignments.CONSTITUENT_ID = self.data.transport_assignments.CONSTITUENT_ID.astype(int)
        self.data.transport_assignments.SERIES_ID = self.data.transport_assignments.SERIES_ID.astype(int)
        self.data.uses_transport.TRAN_ID = self.data.uses_transport.TRAN_ID.astype(int)

        self._remove_old_sediment_assignments(dialog)

        new_use_sediments = []
        # add in the data from the dialog
        tran_id_list = [new_transport_id for _ in range(len(dialog.sediment.assignments.index))]
        assignments = dialog.sediment.assignments.copy()
        assignments.insert(0, 'TRAN_ID', tran_id_list)
        self.data.sediment_assignments = pd.concat([self.data.sediment_assignments, assignments])

        # add the new use sediment transport assignment
        new_use_sediment = 1 if dialog.sediment.ui.transport_constituents_group.isChecked() else 0
        new_use_sediments.append([new_transport_id, new_use_sediment])

        new_row = pd.DataFrame(new_use_sediments, columns=['TRAN_ID', 'USES_SEDIMENT'])
        self.data.uses_sediment = pd.concat([self.data.uses_sediment, new_row])

        self.data.sediment_assignments.TRAN_ID = self.data.sediment_assignments.TRAN_ID.astype(int)
        self.data.sediment_assignments.CONSTITUENT_ID = self.data.sediment_assignments.CONSTITUENT_ID.astype(int)
        self.data.sediment_assignments.SERIES_ID = self.data.sediment_assignments.SERIES_ID.astype(int)
        self.data.uses_sediment.TRAN_ID = self.data.uses_sediment.TRAN_ID.astype(int)

        att_id_to_new_comp_id = self._update_component_ids(att_ids, comp_ids, 'TRANSPORT_ID', new_transport_id, target)
        self._update_old_comp_id_display_list(
            att_id_to_new_comp_id, att_ids, comp_ids, True if target == TargetType.arc else False
        )

    def _remove_old_sediment_assignments(self, dialog):
        """Removes the sediment transport assignments associated with the component ids.

        This will remove all sediment transport assignments if the sediment transport constituents component changed.

        Args:
            dialog (TransportConstituentAssignmentDialog): The dialog to get data from.
        """
        # If the sediment transport component changed, wipe all previous sediment transport assignments and add in
        # only the new ones.
        # If the sediment transport component is unchanged, add the new sediment transport assignments.
        if dialog.sediment.constituents_uuid != self.data.sediment_uuid:
            self.data.sediment_uuid = dialog.sediment.constituents_uuid
            # clear all of the old transport assignments
            self.data.sediment_assignments = self.data.sediment_assignments[0:0]

    def _remove_old_transport_assignments(self, dialog):
        """Removes the transport assignments associated with the component ids.

        This will remove all transport assignments if the transport constituents component changed.

        Args:
            dialog (TransportConstituentAssignmentDialog): The dialog to get data from.
        """
        # If the transport component changed, wipe all previous transport assignments and add in only the new ones.
        # If the transport component is unchanged, add the new transport assignments.
        if dialog.transport.constituents_uuid != self.data.transport_uuid:
            self.data.transport_uuid = dialog.transport.constituents_uuid
            # clear all the old transport assignments
            self.data.transport_assignments = self.data.transport_assignments[0:0]


def arc_default_display_options() -> dict:
    """
    Retrieves default display options for arcs.

    Returns:
        dict: A dictionary containing the display options for arc visualization.
    """
    arc_default_file = os.path.join(
        os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
        'bc_arc_default_display_options.json'
    )
    json_dict = read_display_options_from_json(arc_default_file)
    return json_dict


def point_default_display_options() -> dict:
    """
    Retrieves default display options for points.

    Returns:
        dict: A dictionary containing the display options for arc visualization.
    """
    point_default_file = os.path.join(
        os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
        'bc_point_default_display_options.json'
    )
    point_json_dict = read_display_options_from_json(point_default_file)
    return point_json_dict
