"""Class for reading/writing a 3D Bridge definition to disk."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import pandas

# 3. Aquaveo modules

# 4. Local modules
from xms.bridge.bridge import Bridge


class BridgeIo:
    """Dialog to display and edit an XY Series.

    From https://doc-snapshots.qt.io/qtforpython/tutorials/datavisualize/plot_datapoints.html

    """
    def __init__(self):
        """Initializes the class."""

    def _read_csv(self, filename):
        """Reads the csv file using pandas and returns a dataframe.

        Args:
            filename (:obj:`str`): The filename.

        Returns:
            dataframe (:obj:`pandas.DataFrame`): The pandas dataframe.
        """
        columns = ['Distance', 'Elevation']
        types = {'Distance': np.float64, 'Elevation': np.float64}
        df = pandas.read_csv(filename, header=None, names=columns, index_col=False, usecols=columns, dtype=types)
        df.index += 1
        return df

    def _to_csv(self, data_frame, filename):
        """Writes the data_frame to the filename as a CSV file.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The DataFrame.
            filename (:obj:`str`): The filename.
        """
        data_frame.to_csv(filename, header=False, index=False)

    def read(self, bridge_main_filename):
        """Reads the bridge main file and returns a Bridge object.

        Args:
            bridge_main_filename (:obj:`str`): Path to the bridge main file.

        Returns:
            bridge (:obj:`Bridge`): The bridge object.
        """
        # Read the arc IDs
        bridge = Bridge()
        bridge.arc_id_upstream = bridge.arc_id_downstream = -1
        bridge.specify_downstream_profile = True
        with open(bridge_main_filename, 'r') as bridge_main_file:
            for line in bridge_main_file:
                words = line.split()
                if words and words[0].lower() == 'upstream_arc_id':
                    if len(words) > 1:
                        bridge.arc_id_upstream = int(words[1])
                elif words and words[0].lower() == 'downstream_arc_id':
                    if len(words) > 1:
                        bridge.arc_id_downstream = int(words[1])
                elif words and words[0].lower() == 'manning_n':
                    if len(words) > 1:
                        bridge.manning_n = float(words[1])
                elif words and words[0].lower() == 'specify_downstream_profile':
                    if len(words) > 1 and words[1] == 'False':
                        bridge.specify_downstream_profile = False

        if bridge.arc_id_upstream == -1 or bridge.arc_id_downstream == -1:
            raise RuntimeError('Error reading upstream or downstream arc IDs.')

        # Read the profiles
        top_filename = os.path.join(os.path.dirname(bridge_main_filename), 'top.csv')
        up_filename = os.path.join(os.path.dirname(bridge_main_filename), 'upstream.csv')
        down_filename = os.path.join(os.path.dirname(bridge_main_filename), 'downstream.csv')

        if os.path.isfile(top_filename):
            bridge.df_top = self._read_csv(top_filename)
        if os.path.isfile(up_filename):
            bridge.df_upstream = self._read_csv(up_filename)
        if os.path.isfile(down_filename):
            bridge.df_downstream = self._read_csv(down_filename)

        return bridge

    def write(self, bridge, bridge_main_filename):
        """Writes the bridge to files.

        Args:
            bridge (:obj:`Bridge`): The bridge.
            bridge_main_filename (:obj:`str`): Path to bridge main file.
        """
        with open(bridge_main_filename, 'w') as bridge_main_file:
            bridge_main_file.write('XMS_BRIDGE\n')
            bridge_main_file.write('FILE_VERSION 1.1\n')
            bridge_main_file.write(f'upstream_arc_id {bridge.arc_id_upstream}\n')
            bridge_main_file.write(f'downstream_arc_id {bridge.arc_id_downstream}\n')
            bridge_main_file.write(f'manning_n {bridge.manning_n}\n')
            bridge_main_file.write(f'specify_downstream_profile {bridge.specify_downstream_profile}\n')

        top_filename = os.path.join(os.path.dirname(bridge_main_filename), 'top.csv')
        up_filename = os.path.join(os.path.dirname(bridge_main_filename), 'upstream.csv')
        down_filename = os.path.join(os.path.dirname(bridge_main_filename), 'downstream.csv')

        self._to_csv(bridge.df_top, top_filename)
        self._to_csv(bridge.df_upstream, up_filename)
        self._to_csv(bridge.df_downstream, down_filename)
