"""This module tracks the ongoing progress of a CMS-Flow run."""

# 1. Standard Python modules

# 2. Third party modules
import sqlite3

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.cmsflow.data.simulation_data import SimulationData


class CmsflowTracker:
    """This class tracks the progress of a CMS-Flow run."""
    prog = None
    query = None
    conn = None
    cursor = None
    echo_file = None
    echo_pos = 0
    duration = 0.0
    current_time = 0.0
    plot_statements = []

    UNITS_SECONDS = 'seconds'
    UNITS_MINUTES = 'minutes'
    UNITS_HOURS = 'hours'
    UNITS_DAYS = 'days'
    UNITS_WEEKS = 'weeks'

    def __init__(self):
        """Does nothing except construct the class."""
        pass

    @staticmethod
    def calculate_progress(time):
        """Calculate progress given a timestep.

        Args:
            time (float): The current time in seconds

        Returns:
            (float): Progress as a percent of all timesteps.

        """
        # Cast to float to avoid truncation if int
        return (time / float(CmsflowTracker.duration)) * 100.0

    @staticmethod
    def progress_function():
        """Method called inside the progress loop for computing percent done."""
        if not CmsflowTracker.echo_file:
            CmsflowTracker.echo_file = CmsflowTracker.prog.command_line_output_file

        found_iter = False
        try:
            with open(CmsflowTracker.echo_file, 'r') as f:
                f.seek(CmsflowTracker.echo_pos)
                data_names = CmsflowTracker.prog.get_plot_data_names("plot1")
                prev_echo_line = ''
                echo_line = f.readline()
                while echo_line:
                    if echo_line.endswith('\n') or echo_line.endswith('\r'):
                        CmsflowTracker.echo_pos = f.tell()
                        if echo_line.strip().startswith('ntime='):
                            echo_vals = echo_line.split(',')
                            try:
                                # ntime=0, dt=0.0, time=0.0, Active Cells=0
                                CmsflowTracker.current_time = float(echo_vals[2].split('=')[1])
                            except ValueError:
                                pass
                            found_iter = True
                            for statement in CmsflowTracker.plot_statements:
                                CmsflowTracker.cursor.execute(statement)
                            CmsflowTracker.plot_statements = []
                        elif echo_line.strip().startswith('WSE') and CmsflowTracker.current_time > 0.0 and \
                                echo_line.find('*******') == -1:
                            # EXAMPLE
                            #        16     8.2127E-08     1.4443E-09     1.4195E-09
                            #    WSE(  1546)=  -0.2469, U(   545)=  0.049, V(   660)=  0.049
                            flow_vals = prev_echo_line.split()
                            for data in data_names:
                                data_str = str(data)
                                tbls = CmsflowTracker.prog.get_plot_data_db_table("plot1", data_str)
                                val_idx = -1
                                if data_str == 'Pressure':
                                    val_idx = 1
                                elif data_str == 'U':
                                    val_idx = 2
                                elif data_str == 'V':
                                    val_idx = 3
                                if val_idx >= 0:
                                    CmsflowTracker.plot_statements.append(
                                        f"INSERT INTO {str(tbls[0])} VALUES ("
                                        f"{str(CmsflowTracker.current_time)}, "
                                        f"{str(flow_vals[val_idx])})"
                                    )
                        elif echo_line.strip().startswith('Time Step Reduced.'):
                            # Clear out the plot statements before executing them.
                            CmsflowTracker.plot_statements = []
                    prev_echo_line = echo_line
                    echo_line = f.readline()
            CmsflowTracker.conn.commit()
        except Exception:
            pass  # File might not exist yet

        if found_iter:
            percent_done = CmsflowTracker.calculate_progress(CmsflowTracker.current_time)
            CmsflowTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def get_time_in_secs(time_value, units):
        """Gets a time value in seconds.

        Args:
            time_value (float): The time value to convert.
            units (CmsflowTracker): The units of the time value.

        Returns:
            A floating point value of the time in seconds.
        """
        if units == CmsflowTracker.UNITS_MINUTES:
            return time_value * 60
        elif units == CmsflowTracker.UNITS_HOURS:
            return time_value * 60 * 60
        elif units == CmsflowTracker.UNITS_DAYS:
            return time_value * 60 * 60 * 24
        elif units == CmsflowTracker.UNITS_WEEKS:
            return time_value * 60 * 60 * 24 * 7
        else:  # Assuming CmsflowTracker.UNITS_SECONDS
            return time_value

    @staticmethod
    def start_tracking():
        """Entry point for the CMS-Flow progress script."""
        CmsflowTracker.query = Query(progress_script=True)
        CmsflowTracker.prog = CmsflowTracker.query.xms_agent.session.progress_loop

        CmsflowTracker.conn = sqlite3.connect(CmsflowTracker.prog.plot_db_file)
        CmsflowTracker.cursor = CmsflowTracker.conn.cursor()

        # Get the simulation hidden component
        sim_uuid = CmsflowTracker.query.current_item_uuid()  # Get the simulation uuid
        sim_comp = CmsflowTracker.query.item_with_uuid(
            sim_uuid, model_name='CMS-Flow', unique_name='Simulation_Component'
        )
        data = SimulationData(sim_comp.main_file)
        duration = data.general.attrs['SIM_DURATION_VALUE']
        duration_units = data.general.attrs['SIM_DURATION_UNITS']

        duration_in_secs = CmsflowTracker.get_time_in_secs(duration, duration_units)
        CmsflowTracker.duration = duration_in_secs

        CmsflowTracker.prog.set_progress_function(CmsflowTracker.progress_function)
        CmsflowTracker.prog.start_loop()
