"""Module for the CulvertStructureWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['write_tide_gate_structures']

# 1. Standard Python modules
import logging
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UGrid2d
from xms.data_objects.parameters import Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData

# 4. Local modules
from xms.cmsflow.data.model import parameter_to_file
from xms.cmsflow.file_io.card_writer import SEQUENCE
from xms.cmsflow.file_io.row_or_column_structure_writer import RowOrColumnStructureWriter


def write_tide_gate_structures(
    coverage: Coverage, data: CoverageBaseData, ugrid: UGrid2d, logger: logging.Logger, wrote_header: bool,
    cards: TextIO
) -> bool:
    """
    Write all the data needed by tide gates in a structure coverage.

    Args:
        coverage: Coverage containing geometry to write.
        data: Data manager for the coverage. Should have its component_id_map initialized.
        ugrid: The QuadTree to snap the coverage to.
        logger: Where to log any warnings or errors.
        wrote_header: Whether the structures header has already been written.
        cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.

    Returns:
        Whether the `!Structures` header was written (either before or by calling this function).
    """
    writer = TideGateStructureWriter(coverage, data, ugrid, logger, wrote_header, cards)
    return writer.write()


class TideGateStructureWriter(RowOrColumnStructureWriter):
    """Class for writing all the data needed by tide gates in a structure coverage."""
    def __init__(
        self, coverage: Coverage, data: CoverageBaseData, ugrid: UGrid2d, logger: logging.Logger, wrote_header: bool,
        cards: TextIO
    ):
        """
        Initialize the writer.

        Args:
            coverage: Coverage containing geometry to write.
            data: Data manager for the coverage. Should have its component_id_map initialized.
            ugrid: The QuadTree to snap the coverage to.
            logger: Where to log any warnings or errors.
            cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.
            wrote_header: Whether the structures header has already been written.
        """
        super().__init__(coverage, data, ugrid, logger, wrote_header, cards)
        self._structure_display_name: str = 'tide gate'
        self._structure_group_name: str = 'tide_gate'

    def _write_specific_structure(self, cells: list[int]):
        """Write the information needed for a tide gate."""
        group = self._section.group('tide_gate')

        self._cards.write('TIDEGATE_STRUCT_BEGIN', indent=0)

        zero_based_cells = [cell + 1 for cell in cells]
        self._cards.write('CELL_IDS', len(cells), SEQUENCE, zero_based_cells)

        distribution_coefficient = group.parameter('distribution_coefficient').value
        self._cards.write('DISTRIBUTION_COEFFICIENT', distribution_coefficient)

        parameter = group.parameter('orientation_sea')
        value = parameter_to_file(parameter)
        self._cards.write('ORIENTATION_SEA', value)

        flow_coeff_bay = group.parameter('flow_coeff_from_bay').value
        self._cards.write('FLOW_COEFF_FROM_BAY', flow_coeff_bay)

        flow_coeff_sea = group.parameter('flow_coeff_from_sea').value
        self._cards.write('FLOW_COEFF_FROM_SEA', flow_coeff_sea)

        opening_height = group.parameter('opening_height').value
        self._cards.write('OPENING_HEIGHT', opening_height)

        bottom_elevation = group.parameter('bottom_elevation').value
        self._cards.write('BOTTOM_ELEVATION', bottom_elevation)

        parameter = group.parameter('method')
        value = parameter_to_file(parameter)
        self._cards.write('METHOD', value)

        parameter = group.parameter('schedule_type')
        schedule_type = parameter_to_file(parameter)
        self._cards.write('SCHED_OPERATION_TYPE', schedule_type)

        if schedule_type == 'REGULAR':
            start_time = group.parameter('start_time').value
            self._cards.write('REG_START_TIME', start_time)

            frequency = group.parameter('open_frequency').value
            self._cards.write('REG_OPEN_FREQUENCY', frequency)

            duration = group.parameter('open_duration').value
            self._cards.write('REG_OPEN_DURATION', duration)

        elif schedule_type == 'DESIGNATED':
            schedule = group.parameter('designated_times').value

            self._cards.write('NUM_DESIGNATED_TIMES', len(schedule))

            start_times = [row[0] for row in schedule]
            self._cards.write('DES_START_TIME', SEQUENCE, start_times)

            durations = [row[1] for row in schedule]
            self._cards.write('DES_OPEN_DURATION', SEQUENCE, durations)

        self._cards.write('TIDEGATE_STRUCT_END', indent=0)
