"""Module for the make_main_file method."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['make_main_file']

# 1. Standard Python modules
from importlib import import_module
import os
from pathlib import Path
from typing import TypeVar
import uuid
import xml.etree.ElementTree as Tree

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.components.bases.component_with_menus_base import ComponentWithMenusBase

ComponentType = TypeVar('ComponentType', bound=ComponentWithMenusBase)


def create_component(component_type: type[ComponentType]) -> ComponentType:
    """
    Create a component.

    This method provides the component with a unique component directory, and assigns its main-file a name that matches
    the `default_name` attribute on its `<main_file>` tag in its XML.

    Args:
        component_type: The type of the component to create. This should be the class itself, not an instance.

    Returns:
        An instance of the passed type.
    """
    main_file = make_main_file(component_type)
    return component_type(main_file)


def make_main_file(component) -> Path:
    """
    Make a main-file for a component.

    Args:
        component: The type of component to make a main-file for. This should be something derived from
            VisibleCoverageComponentBase. This need not be an instance of the component; it can just be the type.
    """
    default_name = _find_default_name(component)
    main_file = _new_component_dir() / default_name
    return main_file


def _name_in_file(file: Path, module_name: str, class_name: str) -> str:
    """
    Look for a component's default main-file name in a given .xml file.

    Args:
        file: Path to the file to search.
        module_name: The component's module name.
        class_name: The component's class name.

    Returns:
        Suggested default name if found, or '' otherwise.
    """
    # If this file has what we're looking for, it should be in a structure like
    #
    # <dynamic_model filetype="dynamic model">
    #   <component module_name="the module_name parameter" class_name="the class_name parameter">
    #     <main_file default_name="the value we should return">

    root = Tree.parse(file).getroot()
    if root.tag != 'dynamic_model':
        return ''

    if 'filetype' not in root.attrib or root.attrib['filetype'] != 'dynamic model':
        return ''

    for child in root:
        if child.tag != 'component':
            continue
        if 'module_name' not in child.attrib or child.attrib['module_name'] != module_name:
            continue
        if 'class_name' not in child.attrib or child.attrib['class_name'] != class_name:
            continue

        for grandchild in child:
            if grandchild.tag == 'main_file' and 'default_name' in grandchild.attrib:
                return grandchild.attrib['default_name']

    return ''


def _find_default_name(component) -> str:
    """
    Find the default main-file name for a component.

    This works by tracking down the .xml file for the component's package and parsing it to find the default name there.
    Doing so allows the default name to be specified in a single location, preventing inconsistency.

    Args:
        component: The component to find the default name for.

    Returns:
        Suggested default main-file name for the component.
    """
    module_name = component.static_module_name()
    class_name = component.static_class_name()

    # We can't trust our __file__ for this since we might not share a parent with the component. This happens when we're
    # dev installed or this is called from a derived component whose package is. The .xml file we want should always be
    # relative to the derived component, so we'll import it and use its __file__.
    module = import_module(module_name)
    package_dir = Path(module.__file__)
    while package_dir.parent != package_dir and package_dir.parent.name != 'xms':
        package_dir = package_dir.parent
    if package_dir.parent.name != 'xms':
        raise AssertionError('Cannot find default component name for components outside xms package')

    for file in package_dir.glob('*.xml'):
        if file.is_file() and (name_in_file := _name_in_file(file, module_name, class_name)):
            return name_in_file

    raise AssertionError(f'Cannot find default component name for {module_name}.{class_name} in {package_dir}')


def _new_component_dir() -> Path:
    """
    Get a new component directory.

    The returned directory will already exist.
    """
    comp_uuid = str(uuid.uuid4())
    comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(comp_dir)
    return Path(comp_dir)
