"""Methods for reading and writing component display files sent between components and XMS."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from array import array
import csv
import os
from pathlib import Path
import struct
from typing import Sequence

# 2. Third party modules
import orjson
import pandas as pd

# 3. Aquaveo modules
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules


def write_display_options_to_json(filename: Path | str, opt_list: CategoryDisplayOptionList):
    """Writes a display option category list to JSON file.

    Args:
         filename: File path to the output file. Will be replaced if exists.
         opt_list: The category display option list.
    """
    json_dict = {
        'target_type': int(opt_list.target_type),
        'is_ids': int(opt_list.is_ids),
        'uuid': opt_list.uuid,
        'comp_uuid': opt_list.comp_uuid,
        'show_targets': int(opt_list.show_targets),
        'logarithmic_scale': int(opt_list.logarithmic_scale),
        'target_scale': opt_list.target_scale,
        'obs_reftime': opt_list.obs_reftime.strftime(ISO_DATETIME_FORMAT),
        'obs_dset_uuid': opt_list.obs_dset_uuid,
        'enable_polygon_labels': int(opt_list.enable_polygon_labels),
        'categories': list(opt_list)
    }
    # Only write projection info specified. Not used for id-based draws.
    if opt_list.projection:
        json_dict['projection'] = opt_list.projection
    with open(filename, 'wb') as file:
        data = orjson.dumps(json_dict)
        file.write(data)


def read_display_options_from_json(filename: Path | str):
    """Reads a display option category list from a JSON file.

    Args:
         filename: Fileystem path to the input JSON file.

    Returns:
         opts (:obj:`dict`): List of display option category dictionaries
    """
    if os.path.isfile(filename):
        with open(filename, 'rb') as file:
            return orjson.loads(file.read())
    return {}


def write_display_option_ids(
    filename: Path | str, ids: Sequence[int], label_texts: Sequence[str] | None = None
) -> None:
    """Write an array of integer ids to a binary file.

    Args:
        filename: Filesystem path of the output file
        ids: The ids to write
        label_texts: Display label text for individual component ids. If provided, should be parallel with 'ids'.
    """
    id_array = array('L', ids)
    with open(filename, 'wb') as file:
        id_array.tofile(file)
    if label_texts is not None:  # Write the label text file if it has been specified
        _write_labels_file(filename, ids, label_texts)


def write_display_option_point_locations(
    filename: Path | str, locs: Sequence[float], label_texts: Sequence[str] | None = None
) -> None:
    """Write an array of float point locations (x,y,z) to a binary file.

    Args:
        filename: Filesystem path of the output file
        locs: The point locations to write. List should be formatted as
            [x1, y1, z1, x2, y2, z2, ..., xN, yN, zN]
        label_texts: Display label text for individual component ids. If provided, should be parallel with locs/3.
    """
    loc_array = array('d', locs)
    with open(filename, 'wb') as file:
        loc_array.tofile(file)
    if label_texts is not None:  # Write the label text file if it has been specified
        ids = list(range(int(len(locs) / 3)))  # IDs will be 0 to n-1
        _write_labels_file(filename, ids, label_texts)


def write_display_option_line_locations(
    filename: Path | str, locs: Sequence[Sequence[float]], label_texts: Sequence[str] | None = None
) -> None:
    r"""Write an array of float line point locations (x,y,z) to a binary file.

    locs list should be formatted as:

    ::

            [
                [x11, y11, z11, x12, y12, z12, ..., x1N, y1N, z1N],  # line 1
                 ...,
                 [xN1, yN1, zN1, xN2, yN2, zN2, ..., xNN, yNN, zNN]  # line N
            ]

    Args:
        filename: Filesystem path of the output file
        locs: The point locations to write. List should be formatted as shown above.
        label_texts: Display label text for individual component ids. If provided, should be parallel with locs/3.
    """
    with open(filename, 'wb') as file:
        file.write((len(locs)).to_bytes(4, byteorder='little', signed=False))
        for line in locs:
            file.write((len(line)).to_bytes(4, byteorder='little', signed=False))
        for line in locs:
            loc_array = array('d', line)
            loc_array.tofile(file)
    if label_texts is not None:  # Write the label text file if it has been specified
        ids = list(range(len(locs)))  # IDs will be 0 to n-1
        _write_labels_file(filename, ids, label_texts)


def write_display_option_line_ids(filename: str, ids: list[list[int]]):
    r"""Write an array of int ids for drawing lines to a binary file.

    Args:
        filename: Filesystem path of the output file
        ids: The point locations to write. List should be formatted as:

    ::

            [
                [id11, id12, ..., id1N],  # line 1
                 ...,
                 [idN1, idN2, ..., idNN]  # line N
            ]
    """
    with open(filename, 'wb') as file:
        file.write((len(ids)).to_bytes(4, byteorder='little', signed=False))
        for line in ids:
            file.write((len(line)).to_bytes(4, byteorder='little', signed=False))
        for line in ids:
            loc_array = array('L', line)
            loc_array.tofile(file)


def write_display_option_polygon_locations(filename: str, locs: list[dict]) -> None:
    r"""Write an array of float polygon point locations (x,y,z) to a binary file.

    Args:
        filename (str): Filesystem path of the output file
        locs: The point locations to
            write. Outermost list is a list of polygons. The dictionary has two keys:
            'outer' and 'voids'. The 'outer' key will have a list of floats defining the
            exterior boundary locations. The 'voids' key will have a list of lists of floats
            defining the interior void locations. Locations will be in counter-clockwise order.
            Innermost list should be formatted as:

    ::

            [
                [x11, y11, z11, x12, y12, z12, ..., x1N, y1N, z1N],  # polygon loop 1
                 ...,
                 [xN1, yN1, zN1, xN2, yN2, zN2, ..., xNN, yNN, zNN]  # polygon loop N
            ]
    """
    with open(filename, 'wb') as file:
        file.write((len(locs)).to_bytes(4, byteorder='little', signed=False))
        for poly in locs:
            if 'voids' in poly:
                file.write((len(poly['voids']) + 1).to_bytes(4, byteorder='little', signed=False))
            else:
                file.write((1).to_bytes(4, byteorder='little', signed=False))
        for poly in locs:
            file.write((len(poly['outer'])).to_bytes(4, byteorder='little', signed=False))
            if 'voids' in poly:
                for void in poly['voids']:
                    file.write((len(void)).to_bytes(4, byteorder='little', signed=False))
        for poly in locs:
            loc_array = array('d', poly['outer'])
            loc_array.tofile(file)
            if 'voids' in poly:
                for void in poly['voids']:
                    void_array = array('d', void)
                    void_array.tofile(file)


def read_display_option_ids(filename):
    """Read an array of integer ids from a binary file.

    Args:
        filename (str): Filesystem path of the input binary file

    Returns:
        (:obj:`list` of :obj:`int`): The read ids
    """
    if os.path.isfile(filename):
        id_array = array('L', [])
        with open(filename, 'rb') as file:
            id_array.fromfile(file, int(os.path.getsize(file.name) / 4))
        return id_array
    return []


def read_display_option_point_locations(filename):
    """Read an array of float point locations (x,y,z) from a binary file.

    Args:
        filename (str): Filesystem path of the input binary file

    Returns:
        (:obj:`list` of :obj:`float`): The read point locations
    """
    if os.path.isfile(filename):
        loc_array = array('d', [])
        with open(filename, 'rb') as file:
            loc_array.fromfile(file, int(os.path.getsize(file.name) / 8))
        return loc_array
    return []


def read_display_option_line_locations(filename):
    """Read an array of float line point locations (x,y,z) from a binary file.

    Args:
        filename (str): Filesystem path of the input binary file

    Returns:
        (:obj:`list` of :obj:`list` of :obj:`float`): The read line point locations
    """
    if os.path.isfile(filename):
        line_array = array('L', [])
        lines = []
        with open(filename, 'rb') as file:
            chunk = file.read(4)  # Read first 4 bytes
            num_lines = struct.unpack('<I', chunk)[0]
            line_array.fromfile(file, num_lines)
            for line in line_array:
                loc_array = array('d', [])
                loc_array.fromfile(file, line)
                lines.append(loc_array)
        return lines
    return []


def read_display_option_line_ids(filename):
    """Read an array of int ids for drawing lines from a binary file.

    Args:
        filename (str): Filesystem path of the input binary file

    Returns:
        (:obj:`list` of :obj:`list` of :obj:`int`): The read line point locations
    """
    if os.path.isfile(filename):
        line_array = array('L', [])
        lines = []
        with open(filename, 'rb') as file:
            chunk = file.read(4)  # Read first 4 bytes
            num_lines = struct.unpack('<I', chunk)[0]
            line_array.fromfile(file, num_lines)
            for line in line_array:
                loc_array = array('L', [])
                loc_array.fromfile(file, line)
                lines.append(loc_array)
        return lines
    return []


def read_display_option_polygon_locations(filename):
    """Read an array of float polygon point locations (x,y,z) from a binary file.

    Args:
        filename (str): Filesystem path of the input binary file

    Returns:
        (:obj:`list` of :obj:`dict` of :obj:`list` of :obj:`float`): The read polygon point
            locations. Outermost list is a list of polygons. The dictionary has two keys:
            'outer' and 'voids'. The 'outer' key will have a list of floats defining the
            exterior boundary locaions. The 'voids' key will have a list of lists of floats
            defining the interior void locations. Locations will be in counter-clockwise order.
    """
    if os.path.isfile(filename):
        poly_loop_array = array('L', [])
        poly_loop_size_array = array('L', [])
        poly_locs = array('d', [])
        polys = []
        with open(filename, 'rb') as file:
            chunk = file.read(4)  # Read first 4 bytes
            num_polys = struct.unpack('<I', chunk)[0]
            poly_loop_array.fromfile(file, num_polys)
            total_loops = sum(poly_loop_array)
            poly_loop_size_array.fromfile(file, total_loops)
            total_floats = sum(poly_loop_size_array)
            poly_locs.fromfile(file, total_floats)
            loc_idx = 0
            loop_size_idx = 0
            for poly in range(num_polys):
                num_loops = poly_loop_array[poly]
                polys.append({'outer': [], 'voids': []})
                for loop in range(num_loops):
                    loop_size = poly_loop_size_array[loop_size_idx]
                    loop_locs_size = loop_size
                    if loop == 0:
                        polys[-1]['outer'] = poly_locs[loc_idx:loc_idx + loop_locs_size]
                    else:
                        polys[-1]['voids'].append(poly_locs[loc_idx:loc_idx + loop_locs_size])
                    loc_idx += loop_locs_size
                    loop_size_idx += 1
        return polys
    return []


def _write_labels_file(filename: Path | str, ids: Sequence[int], label_texts: Sequence[str] | None) -> None:
    """Writes the labels to a file.

    Args:
        filename: Filesystem path of the output file
        ids: The ids to write
        label_texts: Display label text for individual component ids. If provided, should be parallel with 'ids'.
    """
    if len(ids) != len(label_texts):
        raise ValueError('The "ids" and "label_texts" arrays must be parallel.')
    filename = str(filename) + '_labels'
    df = pd.DataFrame({'ids': ids, 'labels': label_texts})
    df.to_csv(filename, quoting=csv.QUOTE_NONNUMERIC, header=False, index=False)
