from abc import ABC

from xms.constraint.rectilinear_geometry import Numbering
from xms.constraint.rectilinear_geometry import Orientation


class RectilinearData(ABC):
    """Abstract class with setters and getters for rectilinear data."""

    def __init__(self, instance):
        """Only called by subclasses.

        Args:
            instance: The C++ wrapped instance.
        """
        self._instance = instance

    @property
    def is_2d_grid(self):
        """Determine if this is a 2D constraint.

        Returns:
            True if a 2D grid?"""
        return self._instance.Is2dGrid()

    @is_2d_grid.setter
    def is_2d_grid(self, is_2d_grid):
        """Is this a 2D grid constraint?"""
        self._instance.SetIs2dGrid(is_2d_grid)

    @property
    def is_3d_grid(self):
        """Determine if this is a 3D constraint.

        Returns:
            True if a 3D grid?"""
        return self._instance.Is3dGrid()

    @is_3d_grid.setter
    def is_3d_grid(self, is_3d_grid):
        """Is this a 3D grid constraint?"""
        self._instance.SetIs3dGrid(is_3d_grid)

    @property
    def origin(self):
        """The location of the bottom-left corner of the grid."""
        return self._instance.GetOrigin()

    @origin.setter
    def origin(self, origin):
        """The location of the plan view bottom-left corner of the grid."""
        self._instance.SetOrigin(origin)

    @property
    def angle(self):
        """The angle of rotation about the bottom-left corner of the grid in
        degrees."""
        return self._instance.GetAngle()

    @angle.setter
    def angle(self, angle):
        """The angle of rotation about the bottom-left corner of the grid in degrees."""
        self._instance.SetAngle(angle)

    @property
    def numbering(self):
        """The numbering of the cells. Possible values: Numbering.kij (J changes first followed by I and then K) or
        Numbering.kji (I changes first)."""
        return Numbering(self._instance.GetNumbering())

    @numbering.setter
    def numbering(self, numbering):
        """The numbering of the cells. Possible values: Numbering.kij (J
        changes first followed by I and then K) or Numbering.kji (I changes
        first)."""
        self._instance.SetNumbering(numbering)

    @property
    def orientation(self):
        """The xyz direction for the I, J and K axes.

        Tuple of Orientation enum values.

        The K value only needs to be specified for a 3D grid."""
        orientation = self._instance.GetOrientation()
        return tuple(Orientation(o) for o in orientation)

    @orientation.setter
    def orientation(self, orientation):
        """The xyz direction for the I, J and K axes. Tuple of Orientation enum
        values. The K value only needs to be specified for a 3D grid."""
        if self.is_2d_grid and len(orientation) < 2:
            raise ValueError('Must specify at X and Y orientation for a 2D '
                             'grid.')
        if self.is_3d_grid and len(orientation) < 3:
            raise ValueError(
                'Must specify at X, Y, and Z orientation for a 2D grid.')
        has_x = Orientation.is_x(orientation[0]) or \
            Orientation.is_x(orientation[1])
        has_y = Orientation.is_y(orientation[0]) or \
            Orientation.is_y(orientation[1])
        if not has_x and not has_y:
            raise ValueError('First two values must specify X and Y '
                             'orientation.')
        if self.is_3d_grid and not Orientation.is_z(orientation[2]):
            raise ValueError('Last item must specify Z orientation.')
        self._instance.SetOrientation(tuple(int(o) for o in orientation))

    @property
    def locations_x(self):
        """The X coordinate of the base grid edges. Given as a distance from
        the origin."""
        return self._instance.GetLocationsX()

    @locations_x.setter
    def locations_x(self, locations):
        """The X coordinate of the base grid edges. Given as a distance from
        the origin."""
        if len(locations) < 2:
            raise ValueError('Must specify at least two edge locations.')
        self._instance.SetLocationsX(locations)

    @property
    def locations_y(self):
        """The Y coordinate of the base grid edges. Given as a distance from
        the origin."""
        return self._instance.GetLocationsY()

    @locations_y.setter
    def locations_y(self, locations):
        """The Y coordinate of the grid edges. Given as a distance from the origin."""
        if len(locations) < 2:
            raise ValueError('Must specify at least two edge locations.')
        self._instance.SetLocationsY(locations)

    @property
    def locations_z(self):
        """The Z coordinate of the edges. Given as a distance from the origin."""
        return self._instance.GetLocationsZ()

    @locations_z.setter
    def locations_z(self, locations):
        """The Z coordinate of the edges. Given as a distance from the origin."""
        if len(locations) < 2:
            raise ValueError('Must specify at least two edge locations.')
        self._instance.SetLocationsZ(locations)

    @property
    def has_square_xy_locations(self):
        """Are the grid cells square in plan-view?"""
        return self._instance.GetHasSquareXyLocations()

    @property
    def square_xy_edge_length(self):
        """The length of one side of square cells."""
        if not self.has_square_xy_locations:
            raise ValueError('Grid does not have square cells.')
        return self._instance.GetSquareXyEdgeLength()

    def set_square_xy_locations(self, num_x_locations, num_y_locations,
                                edge_length):
        """Set the grid edge locations for square cells.

        Args:
            num_x_locations: The number of point edges along x axis.
            num_y_locations: The number of point edges along y axis.
            edge_length: The length of one side of square cells.
        """
        if num_x_locations <= 0 or num_y_locations <= 0:
            raise ValueError('Number of edge locations must be greater than '
                             'zero.')
        if edge_length <= 0:
            raise ValueError('Edge length must be greater than zero.')
        self._instance.SetSquareXyLocations(num_x_locations, num_y_locations,
                                            edge_length)
