"""Code to store interface with the SMS activity coverage."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import uuid

# 2. Third party modules
import h5py
import numpy

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DataDumpIOBase
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules
from xms.coverage import coverage_util


class ActivityCoverage(DataDumpIOBase):
    """Class to interface with the SMS activity coverage."""
    def __init__(self, file_name=''):
        """Construct the activity coverage.

        Args:
            file_name(str): Path to the H5 dump file
        """
        super().__init__()
        super().SetSelf(self)
        self.__m_cov = None
        self.__m_activity = {}
        self.m_hasRead = False
        self.m_fileName = file_name
        self.__node_based = True  # Activity basis of take geometry activity
        self.__geom_activity = []  # Activity mapped to a take geometry - 1=active, 0=inactive, -1=unassigned

    @property
    def m_cov(self):
        """Get the activity coverage geometry.

        Returns:
            xms.data_objects.parameters.Coverage: The activity coverage geometry
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_cov

    @m_cov.setter
    def m_cov(self, val):
        """Set the activity coverage geometry.

        Args:
            val (xms.data_objects.parameters.Coverage): The activity coverage geometry
        """
        self.__m_cov = val

    @property
    def m_activity(self):
        """Get the activity mapping of the coverage.

        Returns:
            dict: Key is coverage polygon id and value is True if active
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__m_activity

    @m_activity.setter
    def m_activity(self, val):
        """Set the activity coverage geometry.

        Args:
            val (dict): Key is coverage polygon id and value is True if active
        """
        self.__m_activity = val

    @property
    def node_based(self):
        """Return True if geometry activity is node based."""
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__node_based

    @node_based.setter
    def node_based(self, val):
        """Set the geometry's activity basis.

        Args:
            val (node_based): True if geometry activity is node based
        """
        self.__node_based = val

    @property
    def geom_activity(self):
        """Get the geometry's activity array.

        Returns:
            list of int: The geometry's activity array
        """
        if self.m_hasRead is False and self.m_fileName:
            self.ReadDump(self.m_fileName)
        return self.__geom_activity

    @geom_activity.setter
    def geom_activity(self, val):
        """Set the geometry's activity array.

        Args:
            val (list of int): The geometry's activity array
        """
        self.__geom_activity = val

    def WriteDump(self, file_name):  # noqa: N802
        """Write the activity coverage geometry and attributes to an H5 file XMS can read.

        Args:
            file_name (str): Path to the output H5 file
        """
        cov_uuid = self.m_cov.uuid
        if not cov_uuid or cov_uuid.lower() == 'cccccccc-cccc-cccc-cccc-cccccccccccc':
            self.m_cov.uuid = str(uuid.uuid4())
        # Get the polygons before writing the coverage geometry to disk.
        polys = self.m_cov.polygons
        self.m_cov.write_h5(file_name)

        f = h5py.File(file_name, 'a')
        cov_group = f['Map Data/Coverage1']
        atts = cov_group.create_group("Attributes")
        act_dataset = atts.create_dataset("Activity", (len(self.m_activity), ), 'i')
        id_dataset = atts.create_dataset("Id", (len(self.m_activity), ), 'i')

        # Write coverage polygon activity.
        for i in range(0, len(polys)):
            act_dataset[i] = self.m_activity[polys[i].id]
            id_dataset[i] = polys[i].id

        # Write geometry activity arrays.
        if self.geom_activity:
            if self.node_based:
                atts.create_dataset('Node Activity', data=numpy.array(self.geom_activity))
            else:
                atts.create_dataset('Cell Activity', data=numpy.array(self.geom_activity))
        f.close()

    def ReadDump(self, file_name):  # noqa: N802
        """Read activity coverage geometry and attributes from an H5 file written by XMS.

        Args:
            file_name (str): Path to the file to read
        """
        self.m_hasRead = True
        f = h5py.File(file_name, 'r')
        the_cov_name = coverage_util.read_coverage_name(f)
        if the_cov_name == '':
            f.close()
            return

        activity_group = f['Map Data/' + the_cov_name + '/Attributes']
        poly_dset = activity_group['Id']
        activity_dset = activity_group['Activity']
        if activity_group.__contains__('Node Activity'):
            self.node_based = True
            self.geom_activity = activity_group['Node Activity'][:]
        elif activity_group.__contains__('Cell Activity'):
            self.node_based = False
            self.geom_activity = activity_group['Cell Activity'][:]
        for poly, act in zip(poly_dset, activity_dset):
            self.m_activity[poly] = act != 0
        f.close()

        # read geometry
        self.m_cov = Coverage(file_name, "/Map Data/" + the_cov_name)
        self.m_cov.get_points(FilterLocation.LOC_NONE)  # force geometry to load from H5

    def Copy(self):  # noqa: N802
        """Return a reference to this object."""
        return self

    def GetDumpType(self):  # noqa: N802
        """Get the XMS coverage dump type."""
        return "xms.coverage.activity"


def ReadDumpWithObject(file_name):  # noqa: N802
    """Read an activity coverage dump file.

    Args:
        file_name (str): Filepath to the dumped coverage to read

    Returns:
        ActivityCoverage: The loaded activity coverage
    """
    act_dump = ActivityCoverage(file_name)
    return act_dump
