"""The Assign EWN Feature polygon attribute dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as consts
from xms.ewn.gui.assign_sediment_volume_dialog_ui import Ui_AssignSedimentVolumeDialog


class AssignSedimentVolumeDialog(XmsDlg):
    """A dialog for assigning materials to polygons."""
    def __init__(self, parent, poly_data, multi_select_warning, num_available):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            poly_data (:obj:`xarray.Dataset`): EWN coverage polygon dataset
            multi_select_warning (:obj:`bool`): If True, displays a multiple polygons selected warning.
            num_available (:obj:`int`): The number of existing available cut/fill polygons
        """
        super().__init__(parent, 'xms.ewn.gui.assign_sediment_volume_dialog')
        self._dbl_validator = None
        self._percent_validator = None
        self._data = poly_data
        self._num_available = num_available
        self._ui = Ui_AssignSedimentVolumeDialog()
        self._ui.setupUi(self)
        self._setup_ui(multi_select_warning)

    def _setup_ui(self, multi_select):
        """Setup the dialog and its widgets.

        Args:
            multi_select (:obj:`bool`): True if changes will apply to multiple polygons.
        """
        if multi_select:
            self._ui.lbl_multi_select_warning.setStyleSheet('font-weight: bold; color: red')
        else:
            self._ui.lbl_multi_select_warning.hide()
        self._setup_comboboxes()
        self._setup_validators()
        self._setup_connections()
        self._load_data()

        # Initialize widget dependency states
        self.on_sediment_type_changed(self._ui.cbx_sediment_type.currentIndex())

    def _setup_comboboxes(self):
        """Fill combobox widget options."""
        # Sediment volume management type
        for idx, text in consts.EWN_POLYGON_SEDIMENT_TYPES.items():
            self._ui.cbx_sediment_type.addItem(text, idx)
        # Sediment volume cut/fill option
        for idx, text in consts.EWN_CUT_FILL_TYPE.items():
            self._ui.cbx_cut_fill_option.addItem(text, idx)
        # Priority grouping (1 - number of available cut/fill polygons)
        num_available = self._num_available
        if self._num_available == 0 or self._data.sediment_type.item() < consts.SEDIMENT_TYPE_AVAILABLE_CUT:
            # Increment the number of possible priorities if there are none or the sediment type is not available
            # cut/fill (they may change it while the dialog is up).
            num_available += 1
        for group_id in range(num_available):
            self._ui.cbx_priority.addItem(f'{group_id + 1}', group_id)

    def _setup_validators(self):
        """Add numeric validators to edit field widgets."""
        # Create validators
        self._dbl_validator = QxDoubleValidator(parent=self)
        self._percent_validator = QxDoubleValidator(parent=self)
        self._percent_validator.setRange(0.0, 100.0)

        # Connect validators to edit field widgets
        self._ui.edt_sediment_value.setValidator(self._dbl_validator)
        self._ui.edt_priority_value.setValidator(self._percent_validator)

    def _setup_connections(self):
        """Setup slot/signal connections."""
        self._ui.cbx_sediment_type.currentIndexChanged.connect(self.on_sediment_type_changed)

    def _load_data(self):
        """Load widget values from persistent polygon dataset."""
        self._ui.edt_poly_name.setText(self._data.polygon_name.item())
        idx = max(0, self._ui.cbx_sediment_type.findData(self._data.sediment_type.item()))
        self._ui.cbx_sediment_type.setCurrentIndex(idx)
        idx = max(0, self._ui.cbx_cut_fill_option.findData(self._data.cut_fill_type.item()))
        self._ui.cbx_cut_fill_option.setCurrentIndex(idx)
        self._ui.edt_sediment_value.setText(str(self._data.volume_value.item()))
        idx = max(0, self._ui.cbx_priority.findData(self._data.priority.item()))
        self._ui.cbx_priority.setCurrentIndex(idx)
        self._ui.edt_priority_value.setText(str(self._data.priority_percent.item()))

    def _save_data(self):
        """Save widget values to persistent polygon dataset."""
        self._data['polygon_name'] = self._ui.edt_poly_name.text()
        self._data['sediment_type'] = self._ui.cbx_sediment_type.itemData(self._ui.cbx_sediment_type.currentIndex())
        self._data['cut_fill_type'] = self._ui.cbx_cut_fill_option.itemData(self._ui.cbx_cut_fill_option.currentIndex())
        self._data['volume_value'] = float(self._ui.edt_sediment_value.text())
        if self._data['sediment_type'] < consts.SEDIMENT_TYPE_AVAILABLE_CUT:
            self._data['priority'] = np.nan  # If not using an available type, ensure no valid priority assigned.
        else:
            self._data['priority'] = self._ui.cbx_priority.itemData(self._ui.cbx_priority.currentIndex())
        self._data['priority_percent'] = float(self._ui.edt_priority_value.text())

    def on_sediment_type_changed(self, index):
        """Called when the sediment type combobox option is changed.

        Args:
            index (:obj:`int`): the new index
        """
        # Disable all widgets if no sediment fill type.
        enable = True
        if index == consts.SEDIMENT_TYPE_NONE:
            enable = False
        self._ui.cbx_cut_fill_option.setEnabled(enable)
        self._ui.edt_sediment_value.setEnabled(enable)

        # Enable/disable the priority widgets.
        enable = False
        if index >= consts.SEDIMENT_TYPE_AVAILABLE_CUT:
            enable = True
        self._ui.cbx_priority.setEnabled(enable)
        self._ui.edt_priority_value.setEnabled(enable)

    def accept(self):
        """Save data to persistent storage on OK."""
        self._save_data()
        super().accept()
