"""Code to handle XMS DMI component events for EwnCoverageComponent."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.ewn.tools.ewn_process import EwnProcess
from xms.ewn.tools.runners import runner_util


class InsertEwnFeatures:
    """Class to insert EWN Feature into a 2D ugrid."""
    def __init__(
        self, polygon_data, ugrid, is_geographic, is_levee=False, bias=0.3, lock_dataset=None, is_cartesian=False
    ):
        """Initializes the helper class.

        Args:
            polygon_data (:obj:`list`): Holds the outside polygon points and polygon attributes
            ugrid (:obj:`xms.grid.ugrid.UGrid`): 2d Unstructured grid
            is_geographic (:obj:`bool`): true if coordinates are geographic
            is_levee (:obj:`bool`): true if the coverage is a levee coverage
            bias (:obj:`float`): bias used in xms.mesher for speed of element size transition (0.0-1.0)
            lock_dataset (:obj:`list`): a dataset used for locking nodes.
            is_cartesian (:obj:`bool`): true if the grid is cartesian
        """
        self._polygon_data = polygon_data
        self._logger = logging.getLogger('xms.ewn')

        self._ewn_process = EwnProcess(ugrid, is_geographic, is_levee, bias, lock_dataset, is_cartesian)
        self._is_geographic = is_geographic
        self._is_levee = is_levee
        self._bias = bias
        self._ugrid = ugrid
        self._transect_polys = []
        self._lock_dataset = lock_dataset
        self._is_cartesian = is_cartesian
        self.cgrid_elevations = None

        self.is_levee = False
        # variables used when inserting an EWN feature
        self.out_ugrid = None
        self.non_intersecting_transects = []
        self.error_cov_name = 'EWN Non-intersecting Transects'

    def insert_features(self):
        """Inserts EWN features into a 2d ugrid."""
        cov_uuid = ''
        for idx, poly in enumerate(self._polygon_data):
            if poly['cov_uuid'] != cov_uuid:
                cov_uuid = poly['cov_uuid']
                self._transect_polys = []
            if idx > 0:
                self._ewn_process = EwnProcess(
                    self._ugrid, self._is_geographic, self._is_levee, self._bias, self._lock_dataset, self._is_cartesian
                )
            self._logger.info(f'Inserting EWN feature polygon ({idx + 1} of {len(self._polygon_data)})...')
            poly['transect_polys'] = self._transect_polys
            self._ewn_process.set_polygon_data(poly)
            self._ewn_process.insert_feature()
            if self._is_cartesian:
                if self.cgrid_elevations is None:
                    self.cgrid_elevations = []
                self.cgrid_elevations.extend(self._ewn_process.cgrid_elevations)
                self.out_ugrid = self._ugrid
            elif 'arc_pts' in poly and poly['arc_pts'] is not None:  # We need to post process arcs with 0.0 top width
                # Snap rip back to create line
                processed_ug = runner_util.post_process_ugrid_for_line(poly, self._ewn_process.stitched_ugrid)
                self.out_ugrid = self._ugrid = processed_ug
            else:
                self.out_ugrid = self._ugrid = self._ewn_process.stitched_ugrid
            self._transect_polys.append(self._ewn_process.transect_polygon)
        self.non_intersecting_transects = self._ewn_process.non_intersecting_transects
        if self.non_intersecting_transects:
            msg = f'Error intersecting feature with mesh. The {self.error_cov_name!r} coverage contains all segments ' \
                  f'that do not intersect the mesh.'
            self._logger.warning(msg)
