"""This module is for the grid component."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as xfs
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.gencade.components.gencade_component import GenCadeComponent
from xms.gencade.components.id_files import GRID_INITIAL_ATT_ID_FILE, GRID_INITIAL_COMP_ID_FILE
from xms.gencade.data.grid_data import GridData
from xms.gencade.gui.grid_options_dlg import GridOptionsDlg
from xms.gencade.gui.grid_point_atts_dlg import GridPointAttsDlg


class GridComponent(GenCadeComponent):
    """A hidden Dynamic Model Interface (DMI) component for the Gencade grid coverage."""

    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.class_name = 'GridComponent'
        self.module_name = 'xms.gencade.components.grid_component'
        self.uuid = os.path.basename(os.path.dirname(self.main_file))
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Grid Definition Options...', 'open_grid_options'),
            ('Display Options...', 'open_display_options')
        ]
        self.point_commands = [('Assign Point Type...', 'open_point_attributes')]
        self.data = GridData(self.main_file)
        self.comp_id_files = {
            'Begin': 'begin_point.display_ids',
            'End': 'end_point.display_ids',
            'Refine': 'refine_point.display_ids'
        }
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), 'grid_display_options.json')
        self._initialize()

    def _initialize(self):
        """Create default data files upon creation."""
        if os.path.exists(self.main_file):
            self.cov_uuid = self.data.info.attrs['cov_uuid']
            if not os.path.exists(self.disp_opts_file):
                # Read the default arc display options, and save ourselves a copy with a randomized UUID.
                point_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                point_default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                                  'grid_display_options.json')
                point_json_dict = read_display_options_from_json(point_default_file)
                point_categories.from_dict(point_json_dict)
                point_categories.comp_uuid = self.uuid
                write_display_options_to_json(self.disp_opts_file, point_categories)
                # Save our display list UUID to the main file
                self.data.info.attrs['point_display_uuid'] = point_categories.uuid
                self.data.commit()

    @staticmethod
    def _update_display_uuids_for_duplicate(new_comp_uuid, new_display_options_file):
        """Updates the display options file for a new component created from a duplicate event.

        Args:
            new_comp_uuid (:obj:`str`): The UUID of the new component.
            new_display_options_file (:obj:`str`): The display options file to update.

        Returns:
            The new display uuid as a string.
        """
        json_dict = read_display_options_from_json(new_display_options_file)
        categories = CategoryDisplayOptionList()
        new_uuid = categories.uuid
        categories.from_dict(json_dict)
        categories.uuid = new_uuid
        categories.comp_uuid = new_comp_uuid
        write_display_options_to_json(new_display_options_file, categories)
        return new_uuid

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed, and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            # Change the display UUID to be something different.
            new_comp = GridComponent(new_main_file)
            new_uuid = self._update_display_uuids_for_duplicate(new_comp.uuid, new_comp.disp_opts_file)
            new_comp.data.info.attrs['point_display_uuid'] = new_uuid
            new_comp.data.info.attrs['cov_uuid'] = ''
            new_comp.data.commit()
            # action_requests.append(self.get_display_options_action())
        return new_main_file, messages, action_requests

    def update_id_files(self):
        """Writes the display id files."""
        for display_name, filename in self.comp_id_files.items():
            vals = self.data.points.where(self.data.points.point_type == display_name, drop=True)
            if len(vals.comp_id) > 0:
                id_file = os.path.join(os.path.dirname(self.main_file), f'{filename}')
                write_display_option_ids(id_file, vals.comp_id.data)

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        return [], [self.get_display_options_action()]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        if query is not None:
            self.cov_uuid = query.parent_item_uuid()
            if not self.cov_uuid:
                return [('ERROR', 'Could not get GenCade grid coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.main_file), GRID_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(str(initial_att_file))
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), GRID_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(str(initial_comp_file))
            if query is not None:
                # Wrapped in if statement fot testing
                xfs.removefile(str(initial_att_file))
                xfs.removefile(str(initial_comp_file))
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.point, att_id, comp_id)
            self.update_id_files()

        self.data.info.attrs['cov_uuid'] = self.cov_uuid
        self.data.commit()
        # # Send the default display list to XMS.
        self.display_option_list.append(
            XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid,
                              draw_type=DrawType.draw_at_ids)
        )
        return [], []

    def open_point_attributes(self, query, params, win_cont):
        """Opens the save point dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        xms_pt_ids = params[0].get('selection', [])
        id_files = params[0].get('id_files', [])
        dlg_data = {'point_type': 'Unassigned', 'refine_size_const': 1.0}
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'POINT': (id_files[0], id_files[1])})
            comp_id = self.get_comp_id(TargetType.point, xms_pt_ids[0])
            comp_data = self.data.points.loc[{'comp_id': [comp_id]}]
            dlg_data['point_type'] = comp_data['point_type'].item()
            dlg_data['refine_size_const'] = comp_data['refine_size_const'].item()

        # TODO: handle multi select if all are unassigned or refine, if not abort dialog

        prj_horiz_units = query.display_projection.horizontal_units.lower()
        dlg_data['units'] = 'ft' if 'feet' in prj_horiz_units else 'm'
        dlg = GridPointAttsDlg(parent=win_cont, dlg_data=dlg_data)
        if dlg.exec() == QDialog.Accepted:
            # Load all currently used component ids
            query.load_component_ids(self, points=True)
            # Save attributes from dialog input
            comp_data = dlg.get_comp_data()
            new_comp_id = self.data.add_grid_point_atts()
            self.data.update_grid_point(new_comp_id, comp_data)
            self.data.commit()
            for pt_id in xms_pt_ids:
                self.update_component_id(TargetType.point, pt_id, new_comp_id)
            self.update_id_files()
            self.display_option_list.append(
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid,
                                  draw_type=DrawType.draw_at_ids)
            )
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for boundary condition arcs.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Read the default arc display options, and save ourselves a copy with a randomized UUID.
        point_categories = CategoryDisplayOptionList()
        point_json_dict = read_display_options_from_json(self.disp_opts_file)
        point_categories.from_dict(point_json_dict)

        dlg = CategoryDisplayOptionsDialog([point_categories], win_cont)
        if dlg.exec_():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    def open_grid_options(self, query, params, win_cont):
        """Opens the grid options coverage dialog.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        dlg_data = {'base_cell_size': self.data.info.attrs['base_cell_size']}
        prj_horiz_units = query.display_projection.horizontal_units.lower()
        dlg_data['units'] = '(ft):' if 'feet' in prj_horiz_units else '(m):'
        dlg = GridOptionsDlg(parent=win_cont, dlg_data=dlg_data)
        if dlg.exec():
            dlg_data = dlg.get_comp_data()
            self.data.info.attrs['base_cell_size'] = dlg_data['base_cell_size']
            self.data.commit()
        return [], []
