"""The MaterialGroupSetDialog dialog."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from itertools import count
from typing import Callable, Optional, Sequence

# 2. Third party modules
import numpy as np
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QAbstractItemView, QMessageBox, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.gmi.data.generic_model import GroupSet
from xms.gmi.gui.group_set_dialog import GroupSetDialog, QxListWidgetItem


class MaterialSectionDialog(GroupSetDialog):
    """A specialization of `SectionDialog` for materials."""
    def __init__(
        self,
        parent: QWidget | None,
        section: GroupSet,
        get_curve: Callable[[int, bool], tuple[Sequence[float | np.datetime64], Sequence[float]]],
        add_curve: Callable[[Sequence[float | np.datetime64], Sequence[float], bool], int],
        is_interior: bool,
        dlg_name: str,
        window_title: str = '',
        multi_select_message: str = '',
        show_groups: bool = True,
        enable_unchecked_groups: bool = False,
        hide_checkboxes: bool = False,
    ):
        """
        Initialize the dialog and set up the ui.

        Args:
            parent: Parent window.
            section: Section in the model to display and edit values for.
            get_curve: A callable that takes a curve ID and whether that curve uses dates, and returns the curve's
                X and Y values. The X values will be of type `np.datetime64` if using dates, or `float` otherwise.
            add_curve: A callable that takes a list of X values, a list of Y values, and whether to use dates, and
                returns a curve ID for a curve with the input values. X values will be of type `np.datetime64` if
                using dates, or `float` otherwise. This function should arrange so that calling `get_curve` later with
                the same curve ID and `use_dates` will return the same X and Y values passed to this function.
            is_interior: Whether the feature is on the interior of a coverage. If True, only groups that are legal on
                interior features will be shown.
            dlg_name: Unique name for this dialog. site-packages import path would make sense.
                Used as registry key for loading and saving the dialog geometry.
            window_title: Title of window.
            multi_select_message: A warning message to show the user. If nonempty, will be displayed prominently.
                Used by some models to display a warning when the user selects multiple features and tries to edit
                them all.
            show_groups: Whether to show the group panel. If there is always exactly one group, this panel is just
                visual noise and can be disabled with this parameter.
            enable_unchecked_groups: Whether to allow setting parameters in groups that are inactive.
            hide_checkboxes: Whether checkboxes are hidden from the left side groups, regardless of other settings.
        """
        super().__init__(
            parent=parent,
            section=section,
            get_curve=get_curve,
            add_curve=add_curve,
            is_interior=is_interior,
            dlg_name=dlg_name,
            window_title=window_title,
            multi_select_message=multi_select_message,
            show_groups=show_groups,
            enable_unchecked_groups=enable_unchecked_groups,
            hide_checkboxes=hide_checkboxes,
        )

        # There should always be an active material.
        self._last_checked_item = None
        self._ensure_at_least_one_item_checked()

        # The unassigned material's name shouldn't be editable. N.B. this changes the current item.
        unassigned = self.group_list.item(0)
        unassigned.setFlags(unassigned.flags() & (~Qt.ItemIsEditable))

        # Ensure the currently checked group is active so the values panel will show it.
        for row in range(self.group_list.count()):
            item = self.group_list.item(row)
            if item.checkState() == Qt.Checked:
                self.group_list.setCurrentItem(item)

        self.group_list.setEditTriggers(
            QAbstractItemView.DoubleClicked | QAbstractItemView.SelectedClicked | QAbstractItemView.EditKeyPressed
        )
        self.group_buttons_container.show()
        self.add_button.clicked.connect(self.add)
        self.remove_button.clicked.connect(self.remove)
        self.edit_button.clicked.connect(self.edit)

    def edit(self):
        """
        Edit the currently selected item's name.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        current_item = self.group_list.currentItem()
        if self.group_list.currentRow() == 0:
            QMessageBox.warning(self, 'XMS', 'Cannot edit default.', QMessageBox.Ok)
            return
        self.group_list.editItem(current_item)

    def add(self):
        """
        Add a new item to the group list and generic model.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        old_name = self.group_list.currentItem().group_id
        # Coverage doesn't like how this never exhausts the generator, but count() never ends, so that's impossible.
        new_name = next(str(i) for i in count(1) if str(i) not in self.section.group_names)  # pragma: no branch
        new_group = self.section.duplicate_group(old_name, new_name, new_label='<Unnamed>')

        group_item = QxListWidgetItem(new_name, new_group.label, self.group_list)
        if not self.hide_checkboxes:
            group_item.setCheckState(Qt.Unchecked)
        self._list_widget_items[new_name] = group_item
        if new_group.description:
            group_item.setToolTip(new_group.description)
        self.group_list.setCurrentItem(group_item)
        self.edit()

    def remove(self):
        """
        Remove the currently selected item from the group list and section.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        current_item = self.group_list.currentItem()
        if self.group_list.currentRow() == 0:
            QMessageBox.warning(self, 'XMS', 'Cannot delete default.', QMessageBox.Ok)
            return
        group_id = current_item.group_id
        del self._group_pages[group_id]
        del self._list_widget_items[group_id]
        self.section.remove_group(current_item.group_id)

        row = self.group_list.currentRow()
        self.group_list.takeItem(row)

    def _item_changed(self, item: QxListWidgetItem):
        """
        Handle when some change happens to a list item.

        Args:
            item: The newly selected item.
        """
        # Note that QListWidget.itemChanged is raised for basically anything anyone might conceivably care about,
        # so we have to be careful about behaving well on false alarms.
        if item.group_id not in self._list_widget_items:
            # this case happens when editing the text of item and removing it before finishing the edit
            return
        super()._item_changed(item)
        if item.text() == '':
            item.setText(self.section.group(item.group_id).label)
        else:
            self.section.group(item.group_id).label = item.text()
        self._ensure_at_least_one_item_checked(item)

    def _ensure_at_least_one_item_checked(self, item: Optional[QxListWidgetItem] = None):
        """
        Ensure at least one item is checked.

        Args:
            item: The most recently changed item.
        """
        if self.hide_checkboxes:
            return

        # Is the recently changed item checked?
        if item is not None and item.checkState() == Qt.Checked:
            self._last_checked_item = item
            return

        # Is any other item checked?
        for row in range(self.group_list.count()):
            item = self.group_list.item(row)
            if item.checkState() == Qt.Checked:
                return

        # Can we reset to the most recently checked item?
        if self._last_checked_item is not None:
            self._last_checked_item.setCheckState(Qt.Checked)
            return

        # Then it's Unassigned.
        self.group_list.item(0).setCheckState(Qt.Checked)
