"""LinkNumbersComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.components.bases.component_base import ComponentBase
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.gmi.components.gmi_component import duplicate_display_opts

# 4. Local modules


class LinkNumbersComponent(ComponentBase):
    """A Dynamic Model Interface (DMI) component to display stream arcs and their link numbers."""
    def __init__(self, main_file):
        """Initializer.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.disp_opts_files = [self.main_file]

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:

                new_main_file (str): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (list[tuple(str)]): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))

        if save_type == 'DUPLICATE':
            action = self._get_display_action()
            for i, disp_file in enumerate(self.disp_opts_files):
                new_disp_file = os.path.join(new_path, os.path.basename(disp_file))
                json_dict = duplicate_display_opts(new_path, new_disp_file)
                if i == 0:
                    action.main_file = new_disp_file
                    action.component_uuid = json_dict['comp_uuid']
                    action_requests.append(action)

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:

                messages (list[tuple(str)]): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        return [], [self._get_display_action()]

    def get_initial_display_options(self, query: Query, params: list[dict]):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        for disp_file in self.disp_opts_files:
            self.display_option_list.append(XmsDisplayMessage(file=disp_file, draw_type=DrawType.draw_at_locations))
        return [], []

    def _get_display_action(self):
        """Gets an action request to update the display.

        Returns:
            (ActionRequest): the ActionRequest
        """
        return ActionRequest(
            modality='NO_DIALOG',
            main_file=self.main_file,
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options'
        )
