r"""Imports GSSHA model native files, with the help of some non-native files (e.g. .map).

WMS uses the "FLINE" card in the .gssha file to identify a WMS .map file which defines coverages. If the .map file is
found, it is read and a coverage is created from it. The stream network could also be read from the .cif (GSSHA_CHAN)
file, but we aren't reading it at the moment.
"""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import os
from pathlib import Path
import uuid

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.constraint import Grid, Orientation, RectilinearGridBuilder
from xms.coverage.arcs import arc_util
from xms.coverage.file_io import map_file_reader
from xms.coverage.xy.xy_series import XySeries
from xms.data_objects.parameters import (
    Arc, Component, Coverage, FilterLocation, Projection, Simulation, UGrid as DoUGrid
)
from xms.datasets.dataset_writer import DatasetWriter
from xms.gmi.data.generic_model import GenericModel, Group, Parameter, Section, Type
from xms.grid.ugrid import UGrid
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs import dialog_util, process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.gssha.components import dmi_util, gmi_util
from xms.gssha.components.bc_coverage_component import BcCoverageComponent
from xms.gssha.components.gmi_util import DisplayIds
from xms.gssha.components.sim_component import SimComponent
from xms.gssha.data import bc_generic_model, sim_generic_model
from xms.gssha.data.bc_generic_model import ChannelType
from xms.gssha.data.bc_util import NodeArcs
from xms.gssha.data.sim_generic_model import InfilType, OutputUnits, PrecipitationType, RichardsCOption, Routing
from xms.gssha.file_io import (cmt_file_reader, gag_file_reader, ihl_file_reader, io_util, xys_file_reader)

# Type aliases
Streams = list[tuple[Arc, int]]


def read() -> 'ProjectReader':
    """Imports GSSHA model native .gssha (GSSHAPROJECT) files and related files.

    This function would be called 'import' but that's a reserved word in Python.
    """
    # For testing
    # from xms.guipy import debug_pause
    # debug_pause()
    # _set_test_environment()

    dialog_util.ensure_qapplication_exists()
    query = dmi_util.create_query()
    thread = ProjectReader(query)
    process_feedback_dlg.run_feedback_dialog(thread)
    return thread


class ProjectReader(FeedbackThread):
    """Imports a GSSHA simulation."""
    def __init__(self, query: Query):
        """Initializes the class."""
        super().__init__(query)
        self._query = query

        self._gssha_file_path: Path = Path(query.read_file)  # .gssha file path
        self._gssha_dict: dict[str, str] = {}  # .gssha file in a dict. card -> value
        self._do_sim_uuid: str = ''  # Uuid of data_objects Simulation
        self._sim_comp_uuid: str = ''  # Uuid of SimComponent
        self._sim_comp: 'SimComponent | None' = None
        self._co_grid: 'Grid | None' = None
        self._ugrid: 'UGrid | None' = None
        self._gm: GenericModel = sim_generic_model.create(default_values=False)
        self._datasets: list[DatasetWriter] = []
        self._bc_cov: 'Coverage | None' = None
        self._bc_comp: BcCoverageComponent | None = None
        self._time_series: dict[int, XySeries] = {}  # Time series from the TIME_SERIES_FILE
        self._special_tables_dict: dict[str, tuple[str, pd.DataFrame]] = {}  # TIME_SERIES_INDEX and OVERLAND_BOUNDARY
        self._start_date_time: datetime = datetime(2000, 1, 1)
        self._output_hydrograph_links: list[int] = []
        self._point_source_xy_series: dict[int, XySeries] = {}  # link -> XySeries
        self._project_path: str = ''

        # self._flowbctype_to_olf is FLOWBCTYPE -> olf type param name (see olf_type_params)
        olf_types = list(bc_generic_model.olf_type_params.items())
        self._flowbctype_to_olf = {1: olf_types[0], 2: olf_types[1], 3: olf_types[2], 4: olf_types[3], 5: olf_types[4]}

    def _run(self):
        """Reads the GSSHA model."""
        try:
            self._log.info('Importing GSSHA model...')

            self._check_file_exists()
            self._read_gssha_file_to_dict()
            self._read_project_path()
            self._check_for_required_cards()
            self._create_sim_component()
            self._read_grid()
            self._read_model_control()
            self._read_time_series_file()
            self._read_cmt_file()
            self._read_ihl_file()
            self._read_ihw_file()
            self._read_map_file()
            self._finalize_sim_component()

            # Add stuff to XMS
            self._add_sim()
            self._add_and_link_grid()
            self._add_and_link_coverage()
            self._add_datasets()
            self._log.info('Simulation import complete.\n')
        except Exception as exc:
            self._log.error(f'{exc}')

    def _check_file_exists(self) -> None:
        """Raises an exception if the file doesn't exist."""
        if not self._gssha_file_path.is_file() or self._gssha_file_path.stat().st_size == 0:
            raise FileNotFoundError(f'Error: file "{str(self._gssha_file_path)}" does not exist or is empty.')

    def _read_gssha_file_to_dict(self) -> None:
        """Reads the .gssha file into a dict."""
        self._log.info(f'Reading "{str(self._gssha_file_path)}"...')
        self._gssha_dict = io_util.read_gssha_file_to_dict(self._gssha_file_path)

    def _read_project_path(self) -> None:
        """Reads and stores the PROJECT_PATH card so we only do it once."""
        self._project_path = self._gssha_dict.get('PROJECT_PATH', '').strip('"')
        if self._project_path and not Path(self._project_path).is_dir():
            self._project_path = ''  # If the path doesn't exist, set self._project_path to ''

    def _check_for_required_cards(self) -> None:
        """Checks that the essential cards exist in the .gssha file and raises an exception if not.

        See https://www.gsshawiki.com/Project_File:Required_Inputs
        """
        essential_cards = [
            'GRIDSIZE', 'ROWS', 'COLS', 'TOT_TIME', 'TIMESTEP', 'OUTSLOPE', 'OUTROW', 'OUTCOL', 'ELEVATION',
            'WATERSHED_MASK', 'HYD_FREQ', 'SUMMARY', 'OUTLET_HYDRO'
        ]
        missing_cards = []
        for card in essential_cards:
            if card not in self._gssha_dict:
                missing_cards.append(card)
        if missing_cards:
            raise ValueError(f'Error: missing card(s): {", ".join(missing_cards)}.')

    def _read_grid(self) -> None:
        """Reads the grid."""
        self._log.info('Creating UGrid...')

        # Use the 'WATERSHED_MASK' file to create the grid
        mask_file_path = self._get_full_path('WATERSHED_MASK')
        grass_data = io_util.read_grass_file(mask_file_path)

        # Get average cell size based on grid extents (don't use 'GRIDSIZE' in .gssha file)
        nrows = int(self._gssha_dict['ROWS'])
        ncolumns = int(self._gssha_dict['COLS'])
        cell_size_x = (grass_data.east - grass_data.west) / ncolumns
        cell_size_y = (grass_data.north - grass_data.south) / nrows
        cell_size = (cell_size_x + cell_size_y) / 2.0

        # Build grid
        builder = RectilinearGridBuilder()
        builder.is_2d_grid = True
        builder.origin = (grass_data.west, grass_data.south)
        builder.orientation = (Orientation.y_decrease, Orientation.x_increase)
        builder.set_square_xy_locations(ncolumns + 1, nrows + 1, cell_size)
        self._co_grid = builder.build_grid()
        self._co_grid.uuid = str(uuid.uuid4())
        self._co_grid.model_on_off_cells = grass_data.values.astype(int)
        self._ugrid = self._co_grid.ugrid

    def _read_model_control(self) -> None:
        """Reads various things into what we call model control."""
        section = self._gm.global_parameters

        # Create a dict letting us quickly find the parameter given a card
        parameter_data: dict[str, Parameter] = {}
        for group_name in section.group_names:
            group = section.group(group_name)
            for parameter_name in section.group(group_name).parameter_names:
                parameter_data[parameter_name] = group.parameter(parameter_name)

        # Fill the model values. This takes care of a lot of parameters, but not all.
        for card, value in self._gssha_dict.items():
            if card in parameter_data:
                parameter = parameter_data[card]
                reader = self._reader_from_parameter(parameter)
                parameter.value = reader(value, parameter)

        # Parameters requiring special handling

        # overland_flow
        self._read_option2('overland_flow', 'OVERTYPE', ['ADE', 'ADE-PC'], section)
        # WMS writes "RETEN_DEPTH". We write "RETENTION" to match mapping table name
        if 'RETEN_DEPTH' in self._gssha_dict:
            section.group('overland_flow').parameter('RETENTION').value = True

        # infiltration
        options = {
            'INF_REDIST': InfilType.INF_REDIST,
            'INF_LAYERED_SOIL': InfilType.INF_LAYERED_SOIL,
            'INF_RICHARDS': InfilType.INF_RICHARDS
        }
        c_options = [RichardsCOption.BROOKS, RichardsCOption.HAVERCAMP]
        self._read_option1('infiltration', 'infil_type', options, section)
        self._read_option2('infiltration', 'RICHARDS_C_OPTION', c_options, section)
        self._read_option2('infiltration', 'RICHARDS_K_OPTION', ['GEOMETRIC', 'ARITHMETIC'], section)
        self._read_option2('infiltration', 'RICHARDS_UPPER_OPTION', ['GREEN_AMPT', 'AVERAGE'], section)

        # channel_routing
        if 'DIFFUSIVE_WAVE' in self._gssha_dict:
            section.group('channel_routing').parameter('routing').value = Routing.DIFFUSIVE_WAVE
        self._read_bool_plus('channel_routing', 'write_chan_hotstart_chk', 'WRITE_CHAN_HOTSTART', section)
        self._read_bool_plus('channel_routing', 'read_chan_hotstart_chk', 'READ_CHAN_HOTSTART', section)

        # output_general
        options = {'METRIC': OutputUnits.METRIC, 'QOUT_CFS': OutputUnits.QOUT_CFS}
        self._read_option1('output_general', 'output_units', options, section)

        # precipitation
        options = {'PRECIP_UNIF': PrecipitationType.PRECIP_UNIF, 'PRECIP_FILE': PrecipitationType.HYETOGRAPH}
        self._read_option1('precipitation', 'precipitation_type', options, section)
        self._read_start_date_and_time(section)
        self._read_hyetograph()

    def _read_hyetograph(self) -> None:
        """Reads the hyetograph stuff from PRECIP_FILE (.gag) file."""
        if 'PRECIP_FILE' not in self._gssha_dict:
            return

        full_path = self._get_full_path('PRECIP_FILE')
        x, y, avg_depth, self._start_date_time = gag_file_reader.read(full_path)

        # Add the curve to the sim data and set the parameters
        group = self._gm.global_parameters.group('precipitation')
        group.parameter('hyetograph_xy').value = (x, y)
        group.parameter('hyetograph_avg_depth').value = avg_depth

    def _read_option1(self, group_name: str, parameter_name: str, options: dict[str, str], section: Section) -> None:
        """Helper method to set an option parameter based on what card is present among a set.

        Example:
            'INF_REDIST': Infiltration.INF_REDIST,
            'INF_LAYERED_SOIL': Infiltration.INF_LAYERED_SOIL,
            'INF_RICHARDS': Infiltration.INF_RICHARDS

        If multiple options are found, the last one wins.

        Args:
            group_name: Name of the GMI group.
            parameter_name: Name of the GMI parameter.
            options: Dict of card -> parameter value.
            section: GMI Section.
        """
        for file_card, option in options.items():
            if file_card in self._gssha_dict:
                section.group(group_name).parameter(parameter_name).value = option

    def _read_option2(self, group_name: str, card: str, file_options: list[str], section: Section) -> None:
        """Helper method to set an option parameter based on a card's value.

        Example:
            RICHARDS_C_OPTION has two possible values: "brooks", or "havercamp"

        Args:
            group_name: Name of the GMI group.
            card: The GSSHA CARD, which is also the generic model parameter name.
            file_options: List of strings that go with the parameter options.
            section: GMS Section.
        """
        if card in self._gssha_dict:
            index = file_options.index(self._gssha_dict[card])
            options = section.group(group_name).parameter(card).options
            if index >= 0:
                section.group(group_name).parameter(card).value = options[index]

    def _read_bool_plus(self, group_name: str, bool_name: str, card: str, section: Section) -> None:
        """Helper method to read a checkbox + card parameter.

        Args:
            group_name: Name of the GMI group.
            bool_name: Name of the GMI bool parameter.
            card: The card from the file, and the parameter name.
            section: GMI Section.
        """
        if card in self._gssha_dict:
            section.group(group_name).parameter(bool_name).value = True
            section.group(group_name).parameter(card).value = self._gssha_dict[card]

    def _read_start_date_and_time(self, section: Section) -> None:
        """Reads START_DATE and/or START_TIME.

        Args:
            section: The GMI Section.
        """
        dt = io_util.datetime_from_cards(self._gssha_dict.get('START_DATE', ''), self._gssha_dict.get('START_TIME', ''))
        if dt:
            date_str = dt.isoformat()
            section.group('precipitation').parameter('start_date_time').value = date_str
            self._start_date_time = dt

    def _read_time_series_file(self) -> None:
        """Reads the time series file (TIME_SERIES_FILE)."""
        if 'TIME_SERIES_FILE' not in self._gssha_dict:
            return

        full_path = self._get_full_path('TIME_SERIES_FILE')
        self._time_series = xys_file_reader.read(full_path, self._start_date_time)

    def _read_cmt_file(self) -> None:
        """Reads the MAPPING_TABLE (.cmt) file."""
        section = self._gm.model_parameters
        group = section.group('mapping_tables')
        if 'MAPPING_TABLE' in self._gssha_dict:
            full_path = self._get_full_path('MAPPING_TABLE')
            self._special_tables_dict = cmt_file_reader.read(full_path, self._co_grid, group, self._datasets)

    def _read_map_file(self) -> None:
        """Reads the XMS .map file containing the GSSHA BC coverage, if there is one."""
        if 'FLINE' not in self._gssha_dict:
            return

        # Read coverage
        map_file_path = self._get_full_path('FLINE')
        try:
            coverages, extras, xy_dict = map_file_reader.read(map_file_path)
        except FileNotFoundError:
            return
        self._bc_cov = _find_gssha_coverage(coverages, extras)
        if not self._bc_cov:
            return

        self._set_coverage_projection(extras)
        self._bc_comp = _create_bc_coverage_component(self._bc_cov.uuid)
        display_ids = self._read_attributes(extras, xy_dict)
        self._bc_comp.data.commit()
        gmi_util.initialize_display(self._bc_comp, display_ids)

    def _read_ihl_file(self) -> None:
        """Reads the IN_HYD_LOCATION (.ihl) file containing hydrograph output locations."""
        if 'IN_HYD_LOCATION' not in self._gssha_dict:
            return

        full_path = self._get_full_path('IN_HYD_LOCATION')
        ihl_file_reader.read(full_path, self._output_hydrograph_links)

    def _read_ihw_file(self) -> None:
        """Reads the (.ihw) file, a non-GSSHA, WMS file containing point source hydrographs.

        These hydrographs are written to the CHAN_POINT_INPUT (.ihg) file but that file format makes it
        impossible to get the hydrographs back out again, hence the need for the .ihw file.
        """
        if '#CHAN_POINT_INPUT_WMS' not in self._gssha_dict:
            return

        full_path = self._get_full_path('#CHAN_POINT_INPUT_WMS')
        with open(full_path, 'r') as file:
            for line in file:
                words = line.rstrip('\n').split()
                link, _, series_id = int(words[0]), int(words[1]), int(words[2])
                next(file)  # Throw away 'GSSHA_TS' line
                xy_series = xys_file_reader.read_time_series(file, self._start_date_time, series_id)
                # _ refers to the GSSHA "node", which is the point number on the polyline, not the arc node. It only
                # makes sense if it's the 1st node, or the starting node, and I think that's the only thing that _ can
                # be in WMS. Thus, we ignore _ and assume it's the starting arc node.
                self._point_source_xy_series[link] = xy_series

    def _set_coverage_projection(self, extras: dict) -> None:
        """Sets the coverage projection."""
        # Look for coverage specific projection. WMS writes it in the .map file
        wkt = self._get_projection_wkt()
        for line in extras[self._bc_cov.uuid]['coverage']:
            if line.startswith('WKT'):
                wkt = line.split(' ', 1)[1]
                break
        if wkt:
            self._bc_cov.projection = Projection(wkt=wkt)

    def _create_sim_component(self):
        """Creates the simulation component."""
        self._log.info('Creating simulation...')

        # Set up folder
        self._sim_comp_uuid = str(uuid.uuid4())
        sim_comp_dir = Path(XmEnv.xms_environ_temp_directory()) / 'Components' / self._sim_comp_uuid
        os.makedirs(sim_comp_dir)
        sim_main_file = sim_comp_dir / 'sim_comp.nc'

        # Make sim component
        self._sim_comp = SimComponent(str(sim_main_file))

    def _finalize_sim_component(self) -> None:
        """Writes the sim to disk."""
        self._sim_comp.data.global_values = self._gm.global_parameters.extract_values()
        self._sim_comp.data.model_values = self._gm.model_parameters.extract_values()
        self._sim_comp.data.commit()

    def _add_sim(self) -> None:
        """Adds the data_objects Simulation component."""
        self._log.info('Adding simulation to XMS...')

        # Make data_objects component
        name = self._gssha_file_path.stem
        self._do_sim_uuid = str(uuid.uuid4())
        do_sim = Simulation(name=name, model='GSSHA', sim_uuid=self._do_sim_uuid)
        do_comp = Component(
            name=name,
            comp_uuid=self._sim_comp_uuid,
            main_file=str(self._sim_comp.main_file),
            model_name='GSSHA',
            unique_name='Sim_Manager',
            locked=False
        )
        self._query.add_simulation(do_sim, [do_comp])

    def _add_and_link_grid(self) -> None:
        """Adds the grid to the query and links it to the simulation."""
        if not self._co_grid:
            return

        self._log.info('Adding UGrid...')
        xmc_file = Path(XmEnv.xms_environ_process_temp_directory()) / f'{self._co_grid.uuid}.xmc'
        self._co_grid.write_to_file(str(xmc_file), binary_arrays=True)
        wkt = self._get_projection_wkt()
        projection = Projection(wkt=wkt)
        name = self._gssha_file_path.stem
        do_ugrid = DoUGrid(str(xmc_file), name=name, uuid=self._co_grid.uuid, projection=projection)
        do_ugrid.force_ugrid = True
        self._query.add_ugrid(do_ugrid)
        self._query.link_item(self._do_sim_uuid, do_ugrid.uuid)

    def _add_and_link_coverage(self) -> None:
        """Adds the coverage to the query and links it to the simulation."""
        if not self._bc_cov:
            return

        self._log.info('Adding coverage...')
        do_comp = dmi_util.do_comp_from_bc_comp(self._bc_comp)
        self._query.add_coverage(
            self._bc_cov, model_name='GSSHA', coverage_type='Boundary Conditions', components=[do_comp]
        )
        self._query.link_item(self._do_sim_uuid, self._bc_cov.uuid)

    def _add_datasets(self) -> None:
        """Adds the datasets to the query."""
        for dataset in self._datasets:
            self._query.add_dataset(dataset)

    def _get_projection_wkt(self) -> str:
        """Reads and returns the projection wkt if it exists in the file."""
        wkt = ''
        if '#PROJECTION_FILE' in self._gssha_dict:
            gssha_file_path = self._get_full_path('#PROJECTION_FILE')
            with gssha_file_path.open('r') as file:
                wkt = file.read().rstrip('\n')
        return wkt

    def _get_full_path(self, card_or_path: str | Path) -> Path:
        """Returns the path, resolving the relative path with 'PROJECT_PATH' if necessary.

        Args:
            card_or_path: Can be the card, in which case the path is obtained from what was read from the gssha file, or
             a path.

        Returns:
            See description.
        """
        if isinstance(card_or_path, str) and card_or_path in self._gssha_dict:
            file_path = self._gssha_dict[card_or_path].strip('"')
        else:
            file_path = card_or_path

        return io_util.get_full_path(self._gssha_file_path, self._project_path, file_path)

    def _read_dataset(self, file_path: str, parameter: 'Parameter | None' = None, name: str | None = None) -> str:
        """Reads the file into a dataset and returns the dataset uuid.

        Args:
            file_path: Path to the grass file containing the dataset values.
            parameter: The generic model parameter.

        Returns:
            See description.
        """
        file_path = self._get_full_path(file_path.strip('"'))
        if parameter:
            name = parameter.parameter_name.lower()
        dataset = io_util.read_grass_file_to_dataset(file_path, name, self._co_grid)
        self._datasets.append(dataset)
        return dataset.uuid

    def _reader_from_parameter(self, parameter: Parameter):
        """Returns a type to use to cast a string to."""
        # Standalone cards with no value (bools). If they exist, they should be True.
        match parameter.parameter_type:
            case Type.BOOLEAN:
                # return lambda value, parameter: bool(value)
                # Assume bools are cards with no value. If present, value is True
                return lambda value, parameter_: True
            case Type.DATASET:
                return self._read_dataset
            case Type.FLOAT:
                return lambda value, parameter_: float(value)
            case Type.INTEGER:
                return lambda value, parameter_: int(value)
            case Type.OPTION:
                return lambda value, parameter_: str(value)
            case Type.TEXT:
                return lambda value, parameter_: str(value)
            case _:
                raise ValueError(f'No reader for {parameter.parameter_name}')

    def _read_attributes(self, extras: dict, xy_dict: dict[int, XySeries]) -> DisplayIds:
        """Sets the bc coverage attribute data and returns a dict with the info needed to initialized the display.

        Stream arcs directions get reversed because WMS has them pointing uphill, and we have them go downhill.

        Args:
            extras: Extra lines from the .map file containing the attribute info
            xy_dict: Dict of xy series id -> XySeries, from the .map file.

        Returns:
            (DisplayIds): Info needed to initialize the display.
        """
        display_ids: DisplayIds = {}

        # Read arc attributes
        values, group_names, ids = self._read_arc_atts(extras, xy_dict)
        component_ids = self._bc_comp.data.add_features(TargetType.arc, values, group_names)
        display_ids[TargetType.arc] = ids, component_ids

        # Read point attributes
        values, group_names, ids = self._read_point_atts(extras)
        component_ids = self._bc_comp.data.add_features(TargetType.point, values, group_names)
        display_ids[TargetType.point] = ids, component_ids

        return display_ids

    def _read_arc_atts(self, extras: dict, xy_dict: dict[int, XySeries]) -> tuple[list[str], list[str], list[int]]:
        """Reads the arc attributes and returns the values, group names, and arc ids.

        Args:
            extras: Extra lines from the .map file containing the attribute info
            xy_dict: Dict of xy series id -> XySeries, from the .map file.

        Returns:
            See description.
        """
        values, group_names, ids, group, olf_param_name = self._init_att_vars()
        section: Section = bc_generic_model.create().arc_parameters
        arc_extras: dict[int, list[str]] = extras[self._bc_cov.uuid]['arcs']
        streams: Streams = []  # indexes into values where streams are
        node_arcs: NodeArcs = {}

        for arc in self._bc_cov.arcs:
            group = None
            section.clear_values()
            for line in arc_extras[arc.id]:

                # An arc can be a channel, or overland flow, but not both for us (but it can be for WMS)

                # Channels
                if line.startswith('TRAP'):  # trapezoidal channel
                    group = self._on_trap(line, arc, section)
                elif line.startswith('BRKPUP'):  # cross section channel
                    group = self._on_brkpup(line, xy_dict, arc, section)
                elif line.startswith('LINK'):  # link number
                    self._on_link(line, arc, group)

                # Overland flow
                elif line.startswith('FLOWBCTYPE'):  # overland flow
                    section.clear_values()
                    group, olf_param_name = self._on_flowbctype(line, section)
                elif line.startswith('CONSTBC'):  # overland flow - constant
                    self._on_constbc(line, group, olf_param_name)
                elif line.startswith('VARBC'):  # overland flow - variable xy series
                    self._on_varbc(line, group, olf_param_name)

            # Save the data
            if group:
                group_names.append(group.group_name)
                values.append(section.extract_values())
                ids.append(arc.id)
                if group.group_name == 'channel':  # Save streams
                    streams.append((arc, len(values) - 1))
                    node_arcs.setdefault(arc.start_node.id, []).append(arc)
                    node_arcs.setdefault(arc.end_node.id, []).append(arc)

        if streams:
            _set_most_downstream_arc(streams, node_arcs, section, values)
        return values, group_names, ids

    def _read_point_atts(self, extras: dict) -> tuple[list[str], list[str], list[int]]:
        """Reads the point attributes and returns the values, group names, and arc ids.

        Args:
            extras: Extra lines from the .map file containing the attribute info

        Returns:
            See description.
        """
        values, group_names, ids, group, olf_param_name = self._init_att_vars()
        section: Section = bc_generic_model.create().point_parameters
        point_extras: dict[int, list[str]] = extras[self._bc_cov.uuid]['points']

        for point in self._bc_cov.get_points(FilterLocation.PT_LOC_DISJOINT):
            group = None
            section.clear_values()
            for line in point_extras[point.id]:
                if line.startswith('FLOWBCTYPE'):  # overland flow
                    section.clear_values()
                    group, olf_param_name = self._on_flowbctype(line, section)
                elif line.startswith('CONSTBC'):  # overland flow - constant
                    self._on_constbc(line, group, olf_param_name)
                elif line.startswith('VARBC'):  # overland flow - variable xy series
                    self._on_varbc(line, group, olf_param_name)

            if group:
                group_names.append(group.group_name)
                values.append(section.extract_values())
                ids.append(point.id)
        return values, group_names, ids

    def _init_att_vars(self) -> tuple[list[str], list[str], list[int], Group, str]:
        """Helper function to declare some variables and eliminate duplicate code."""
        values: list[str] = []
        group_names: list[str] = []
        ids: list[int] = []
        group: 'Group | None' = None
        olf_param_name: str = ''  # see olf_type_params
        return values, group_names, ids, group, olf_param_name

    def _on_trap(self, line: str, arc: Arc, section: Section) -> Group:
        """Reads the TRAP card, indicating a trapezoidal channel.

        Args:
            line: A line from the file.
            arc: The arc.
            section: The GMI Section.
        """
        group = section.group('channel')
        group.is_active = True
        group.parameter('channel_type').value = ChannelType.TRAPEZOIDAL
        words = line.split()[1:]
        group.parameter('mannings_n').value = float(words[0])
        group.parameter('bottom_width').value = float(words[1])
        group.parameter('bankfull_depth').value = float(words[2])
        group.parameter('side_slope').value = float(words[3])
        arc_util.reverse_arc_direction(arc)  # WMS has arcs pointing uphill. We point 'em downhill.
        return group

    def _on_brkpup(self, line: str, xy_dict: dict[int, XySeries], arc: Arc, section: Section) -> Group:
        """Reads the BRKPUP card, indicating a cross-section channel.

        Args:
            line: A line from the file.
            xy_dict: xy series dict.
            arc: The arc.
            section: The GMI Section.
        """
        group = section.group('channel')
        group.is_active = True
        group.parameter('channel_type').value = ChannelType.CROSS_SECTION
        words = line.split()[1:]
        group.parameter('mannings_n').value = float(words[0])
        xy_series = xy_dict[int(words[2])]
        curve_id = self._bc_comp.data.add_curve(xy_series.x, xy_series.y, use_dates=False)
        group.parameter('cross_section').value = curve_id
        arc_util.reverse_arc_direction(arc)  # WMS has arcs pointing uphill. We point 'em downhill.
        return group

    def _on_link(self, line: str, arc: Arc, group: Group) -> None:
        """Handles reading the LINK card.

        Args:
            line: A line from the file.
            arc: The arc.
            group: The GMI group.
        """
        link = int(line.split(' ')[1])
        if link in self._output_hydrograph_links:
            group.parameter('hydrograph_down').value = True

        # Check for point source input
        if link in self._point_source_xy_series:
            xy_series = self._point_source_xy_series[link]
            group.parameter('specify_point_source').value = True
            curve_id = self._bc_comp.data.add_curve(xy_series.x, xy_series.y, use_dates=False)
            group.parameter('point_source_xy').value = curve_id

    def _on_flowbctype(self, line: str, section: Section) -> tuple[Group, str]:
        """Handles reading the FLOWBCTYPE card, used with overland flow BCs.

        Args:
            line: A line from the file.
            section: The GMI Section.

        Returns:
            (tuple): group and olf parameter name
        """
        flow_bc_type = int(_word_1(line))
        olf_tuple = self._flowbctype_to_olf.get(flow_bc_type)
        group = None
        if olf_tuple:
            section.clear_values()
            group = section.group('overland_flow')
            group.is_active = True
            group.parameter('overland_flow_type').value = olf_tuple[0]
        return group, olf_tuple[1]

    def _on_constbc(self, line: str, group: Group, param_name: str) -> None:
        """Handles reading of the CONSTBC card, used with overland flow BCs.

        Args:
            line: A line from the file.
            group: GMI group
            param_name: Parameter name.
        """
        if group and group.group_name == 'overland_flow':
            group.parameter(param_name).value = float(_word_1(line))

    def _on_varbc(self, line: str, group: Group, param_name: str) -> None:
        """Handles reading of the CONSTBC card, used with overland flow BCs.

        Args:
            line: A line from the file.
            group: GMI group
            param_name: Parameter name.
        """
        if group and group.group_name == 'overland_flow':
            xy_id = int(_word_1(line))
            xy_series = self._time_series.get(xy_id, XySeries())
            curve_id = self._bc_comp.data.add_curve(xy_series.x, xy_series.y, use_dates=False)
            group.parameter(param_name).value = curve_id


def _find_gssha_coverage(coverages: list[Coverage], extras: dict) -> 'Coverage | None':
    """Finds and returns the coverage with the 'ACTGSSHACOV' attribute."""
    for coverage in coverages:
        for line in extras[coverage.uuid]['coverage']:
            if line.startswith('ACTGSSHACOV'):
                return coverage
    return None


def _set_test_environment():  # pragma no cover - This is only called when manually debugging
    os.environ[XmEnv.ENVIRON_RUNNING_TESTS] = 'TRUE'


def _create_bc_coverage_component(bc_cov_uuid: str) -> BcCoverageComponent:
    """Creates the BcCoverageComponent."""
    # Create coverage component
    comp_uuid = str(uuid.uuid4())
    bc_comp_dir = Path(_create_component_folder(comp_uuid))
    main_file = bc_comp_dir / 'bc_comp.nc'
    bc_comp = BcCoverageComponent(str(main_file))
    bc_comp.data.coverage_uuid = bc_cov_uuid
    bc_comp.cov_uuid = bc_cov_uuid
    return bc_comp


def _create_component_folder(comp_uuid: str) -> str:
    """Create a folder for a component in the XMS temp components folder.

    Args:
        comp_uuid (str): UUID of the new component

    Returns:
        str: Path to the new component folder
    """
    temp_comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(temp_comp_dir)
    return temp_comp_dir


def _word_1(line: str) -> str:
    """Returns the word at index 1 (0-based) on the line."""
    return line.split()[1:][0]


def _set_most_downstream_arc(streams: Streams, node_arcs: NodeArcs, section: Section, values: list[str]) -> None:
    """Finds and sets the most downstream arc.

    Args:
        streams: List of tuples of stream arcs and the index into the values list.
        node_arcs: Dict of node ID -> list of arc IDs (arcs attached to each node) for arc connectivity.
        section: GMS Section
        values: List of group values
    """
    for arc, index in streams:
        if len(node_arcs[arc.end_node.id]) == 1:
            # No other stream arcs connected to the end of this one, so it must be the most downstream
            section.clear_values()
            section.restore_values(values[index])
            section.group('channel').parameter('most_downstream_arc').value = True
            values[index] = section.extract_values()
            break
