"""Store display options for a display list category."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from typing import Any

# 2. Third party modules
from PySide2.QtGui import QFont

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType


@dataclass
class CategoryDisplayOption:
    """A class that holds display options for a single category."""
    on: bool = True
    label_on: bool = True
    options: Any | None = None  # PointOptions, LineOptions, or PolygonOptions
    label_options: QFont = field(default_factory=lambda: QFont('Arial', 8))  # For distinct QFont objects
    description: str = ''
    file: str = ''
    id: int = 0  # Is not read by XMS, here as a convenience for components
    is_unassigned_category: bool = False

    # If the next flag is False and no label is defined for a specific component id, the label will be empty when
    # rendered in XMS. If True and no label is defined for a specific component id, the label will be the
    # description text string. If a label has been defined for a specific component id, it will always take
    # precedence.
    use_description_for_label: bool = True

    def from_dict(self, opt_dict, target_type):  # noqa: C901
        """Populate from a dict.

        Args:
            opt_dict (dict): dict containing the display list category attributes
            target_type (TargetType): Target type enum of the display list
        """
        if 'on' in opt_dict:
            self.on = bool(opt_dict['on'])
        if 'label_on' in opt_dict:
            self.label_on = bool(opt_dict['label_on'])
        if 'options' in opt_dict:
            if target_type in [TargetType.point, TargetType.ugrid_point, TargetType.ugrid_cell_center]:
                self.options = PointOptions()
            elif target_type in [TargetType.arc, TargetType.ugrid_pointstring, TargetType.ugrid_cellstring]:
                self.options = LineOptions()
            elif target_type in [
                TargetType.polygon, TargetType.ugrid_cellface_fill, TargetType.ugrid_cellface_fill_interior
            ]:
                self.options = PolygonOptions()
            else:  # TargetType.arc_group - options unused but create some to make code work.
                self.options = LineOptions()

            if self.options:
                self.options.from_dict(opt_dict['options'])
        if 'label_options' in opt_dict:
            self.label_options.fromString(opt_dict['label_options'])
        if 'description' in opt_dict:
            self.description = opt_dict['description']
        if 'file' in opt_dict:
            self.file = opt_dict['file']
        if 'id' in opt_dict:
            self.id = int(opt_dict['id'])
        if 'is_unassigned_category' in opt_dict:
            self.is_unassigned_category = bool(opt_dict['is_unassigned_category'])
        if 'use_description_for_label' in opt_dict:
            self.use_description_for_label = bool(opt_dict['use_description_for_label'])

    def __iter__(self):
        """Used for conversion to dict."""
        yield 'on', int(self.on)  # Convert to int for sending to C++
        yield 'label_on', int(self.label_on)  # Convert to int for sending to C++
        yield 'options', dict(self.options)  # Point, line, or polygon specific options
        yield 'label_options', self.label_options.toString()  # Stringify QFont
        yield 'description', self.description
        yield 'file', self.file
        yield 'id', int(self.id)
        yield 'is_unassigned_category', int(self.is_unassigned_category)
        yield 'use_description_for_label', int(self.use_description_for_label)
