"""Plots and table base."""

# 1. Standard Python modules
import os

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules


def np_arrays_from_file(file_name):
    """Read data in the files into numpy arrays.

    Args:
        file_name (str): file name

    Returns:
        (tuple(list[array_names], numpy 2d array)): output
    """
    with open(file_name, 'r') as f:
        lines = f.readlines()

    # remove lines that start with '//'
    while lines[0].startswith('//') and not lines[0].startswith('//    Time(hr)'):
        lines.pop(0)
    if lines[0].startswith('//'):
        lines[0] = lines[0][2:]
    headings = lines[0].split()
    # remove these columns from monitor point files because this data doesn't make sense
    if len(headings) > 0 and headings[-1] == 'CONC_T_ppm':
        headings.pop()
    if len(headings) > 0 and 'ERO_DEP_' in headings[-1]:
        headings.pop()

    arrays = [np.zeros(len(lines) - 1) for _ in range(len(headings))]
    for i in range(1, len(lines)):
        vals = lines[i].split()
        nvals = min(len(vals), len(arrays))
        for j in range(nvals):
            arrays[j][i - 1] = float(vals[j])

    # point sediment file
    fname = file_name[:-4] + '_SED.dat'
    if os.path.isfile(fname):
        sed_headings, sed_arrays = np_arrays_from_file(fname)
        headings = headings + sed_headings[1:]
        arrays = arrays + sed_arrays[1:]

    return headings, arrays


class PlotsAndTableDataBase:
    """A base class for plots and table."""
    def __init__(self, pe_tree, feature_id, feature_type, cov_uuid, model_name, feature_ids=None):
        """Initializes default plot and table options.

        Args:
            pe_tree (xms.guipy.tree.tree_node.TreeNode): The SMS project explorer tree
            feature_id (int): id of point or arc
            feature_type (str): 'Point' or 'Arc'
            cov_uuid (str): uuid of the monitor coverage
            model_name (string): Name of the model we are displaying information from
            feature_ids (list): list of feature ids
        """
        # get the models directory
        project_file = os.getenv('XMS_PYTHON_APP_PROJECT_PATH')
        proj_dir = os.path.dirname(project_file)
        proj_name = os.path.basename(os.path.splitext(project_file)[0])
        self._model_dir = os.path.join(proj_dir, f'{proj_name}_models', model_name)  # self.get_plot_data_files
        if not os.path.isdir(self._model_dir):
            self._model_dir = os.path.join(proj_dir, f'{proj_name}', model_name)

        self.checked_plots = []  # on_list_state_change | update_plots
        self.default_plot_on = 'Default'
        self.err_files = []  # on_list_state_change | update_plots
        self.feature_ids = feature_ids  # Widget uses
        self.feature_ids_plot_data_files = {}  # update_plots
        self.feature_type = feature_type  # setup_feature_ids_list
        self.fids = [feature_id] if feature_ids is None else feature_ids  # Widget uses
        self.file_data = {}
        self.help_url = ''
        self.model_name = model_name  # Widget reference
        self.plot_data_files = {}  # update_plots
        self.plot_list = []  # update_plots | setup_plots_list | setup_main_warning
        self.simulations = []  # Widget tracks changes
        self.window_title = f'{model_name}'  # set window title for dialog not widget

        self._cov_uuid = cov_uuid
        self._feature_id = feature_id
        self._pe_tree = pe_tree
        self._project_simulations = []

    def get_plot_data_for_feature(self, f_id, checked_sims, checked_plots):
        """Retrieve and prepare data for plotting for a given feature ID."""
        return {}, []

    def get_plot_data_and_dataframe(self, sim, min_x, max_x):
        """Retrieves data and dataframe from simulation.

        Args:
            sim: retrieved from self.simulations
            min_x (int): minimum x value.
            max_x (int): maximum x value.
        """
        return [], pd.DataFrame(), False

    def get_plot_max_x(self):
        """Returns the max x value."""
        return 0

    def get_plot_min_max_y(self):
        """Returns the min y, max y, min x, max x."""
        return 0, 0, 1, 1

    def get_f_label(self, f_id):
        """Returns f label."""
        return ''

    def get_x_label(self, x_heading):
        """Returns x label."""
        return ''

    def get_main_warning(self):
        """Returns main error warning at top of widget."""
        return ''

    def get_bottom_warning(self):
        """Returns error message at the bottom of the widget."""
        return ''

    def get_error(self):
        """Returns error message for reading file."""
        return ''
