"""Delegate for a table view that makes a button appear to show node IDs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Callable, Optional

# 2. Third party modules
from PySide2.QtCore import QAbstractItemModel, QEvent, QModelIndex, Qt
from PySide2.QtGui import QBrush, QPainter, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate, QStyleOptionViewItem, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs.view_node_ids_dlg import ViewNodeIdsDlg


class NodeIdButtonDelegate(QStyledItemDelegate):
    """Delegate for the view node ids button column."""
    def __init__(self, data_source: Callable[[int], list[int]], parent: Optional[QWidget] = None):
        """Initializes the class.

        Args:
            data_source: Callback to retrieve data from. Will be called with the index of the row to retrieve nodes for,
                and should return a list of node IDs.
            parent: The parent object.
        """
        super().__init__(parent)
        self.data_source = data_source

    def updateEditorGeometry(self, editor: QWidget, option: QStyleOptionViewItem, index: QModelIndex):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor: The editor widget.
            option: The style options.
            index: The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter: QPainter, option: QStyleOptionViewItem, index: QModelIndex):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter: The painter.
            option: The style options.
            index: The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Node IDs...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(  # noqa: N802
        self, event: QEvent, model: QAbstractItemModel, option: QStyleOptionViewItem, index: QModelIndex
    ):
        """Override of QStyledItemDelegate method of same name.

        Args:
            event: The editor event that was triggered.
            model: The data model.
            option: The style options.
            index: The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                nodes = self.data_source(index.row())
                dlg = ViewNodeIdsDlg(nodes, self.parent())
                dlg.exec()
                return True
        return super().editorEvent(event, model, option, index)
