"""A tree item selector dialog."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs.treeitem_selector_ui import Ui_dlg_treeitem_selector
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg


class TreeItemSelectorDlg(XmsDlg):
    """A dialog that appears when selecting a tree item."""
    def __init__(
        self,
        title,
        target_type,
        pe_tree,
        previous_selection='',
        override_icon=None,
        show_root=False,
        parent=None,
        ui_class=None,
        selectable_xms_types=None,
        allow_multi_select=False,
        allow_change_selection=True,
        selectable_callback=None,
        always_show_check_boxes=True
    ):
        """Initializes the class, sets up the ui.

        Args:
            title (str): The dialog title
            target_type (type): The type of tree item to select
            pe_tree (TreeNode): Root of the project explorer tree to display
            previous_selection (str or iterable): UUID of the previous selection. If provided, that item will be
                selected when the dialog appears. Can be an iterable container of previously selected UUIDs (implies
                allow_multi_select=True).
            override_icon (callable): Callable method to provide icons that override the default for a tree
                item type. Method should take a TreeNode and return an icon path if it is overriden, else empty string.
                Icon path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            parent (Something derived from QWidget): The parent window.
            ui_class: User interface class from a Qt .ui file. Use when deriving a class. If None, the default ui class
                is used.
            selectable_xms_types (list): XMS tree item types of selectable items that are not directly supported
                by xmsapi.
            allow_multi_select (bool): If True, multiple tree items may be selected.
            allow_change_selection (bool): If False, the selection cannot be changed (used for display purposes).
            selectable_callback: A callable function that takes a TreeNode and returns True if it is selectable.
            always_show_check_boxes (bool): If False, only shows checkboxes if allow_multi_select is True.
        """
        super().__init__(parent, 'xmsguipy.dialogs.treeitem_selector')

        # Setup the ui
        if ui_class:
            self.ui = ui_class
        else:
            self.ui = Ui_dlg_treeitem_selector()
        self.ui.setupUi(self)
        self.setWindowTitle(title)

        # Setup the QxProjectExplorerSelectorWidget
        selectable_xms_types = selectable_xms_types if selectable_xms_types else []
        self.ui.tree_pe_selector.initialize(
            root_node=pe_tree,
            selectable_item_type=target_type,
            previous_selection=previous_selection,
            override_icon=override_icon,
            show_root=show_root,
            selectable_xms_types=selectable_xms_types,
            allow_multi_select=allow_multi_select,
            allow_change_selection=allow_change_selection,
            selectable_callback=selectable_callback,
            always_show_check_boxes=always_show_check_boxes
        )

    def get_selected_item_uuid(self):
        """Returns the selected item's uuid or None if nothing selected.

        If multi-select is enabled, return value is a list of the selected tree items' UUIDs. Empty list if no items
        selected.

        Returns:
            See description.
        """
        return self.ui.tree_pe_selector.get_selected_item_uuid()

    def has_selectable_items(self):
        """Return True if trimmed tree has items."""
        return self.ui.tree_pe_selector.has_selectable_items()
