"""Base class for XMS Python dialogs."""

# 1. Standard Python modules
import os
import sys
from urllib.error import URLError
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt, QTimer
from PySide2.QtGui import QIcon, QWindow
from PySide2.QtWidgets import QDialog, QSplitter, QWidget
import win32com.client

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv, XmsEnvironment

# 4. Local modules
from xms.guipy.dialogs import dialog_util, message_box
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.settings import SettingsManager


def parse_parent_window_command_args():
    """Parse the window ids of the parent XMS dialog and main XMS window. Also parses full path to window icon.

    Returns:
        (tuple(int,int,str)): HWND of parent XMS dialog, HWND of the main XMS window, full path to XMS icon
    """
    parent_hwnd = -1
    main_hwnd = -1
    if len(sys.argv) > 1:  # First argument after script is parent XMS dialog's HWND
        parent_hwnd = int(sys.argv[1])
    if len(sys.argv) > 2:  # Second argument after script is main XMS window's HWND
        main_hwnd = int(sys.argv[2])
    icon_path = dialog_util.get_xms_icon()
    return parent_hwnd, main_hwnd, icon_path


def get_xms_icon():
    """Returns the full path to the XMS window icon of the XMS process that launched this script.

    Note that this method will return empty string when the script is run outside of the XMS environment.
    """
    # This used to be defined here but I moved it to dialog_util to avoid circular imports with xms_excepthook.
    # I'm not sure what all expects it here though so I didn't delete this.
    return dialog_util.get_xms_icon()


def get_parent_window_container(hwnd):
    """Get a parent window container for child XMS Python dialogs.

    Args:
        hwnd (int): HWND of the parent hidden dialog in XMS

    Returns:
        (QWidget): See description
    """
    try:
        win = QWindow.fromWinId(hwnd)
        win.setFlags(Qt.FramelessWindowHint)
        win.setModality(Qt.WindowModality.WindowModal)
        return QWidget.createWindowContainer(win)
    except Exception:
        return None


def ensure_qapplication_exists():
    """Ensures a QApplication singleton exists. We don't have to call .exec_().

    https://stackoverflow.com/questions/11145583

    Returns:
         The QApplication singleton
    """
    # This used to be defined here but I moved it to dialog_util to avoid circular imports with message_box.
    # I'm not sure what all expects it here though so I didn't delete this.
    return dialog_util.ensure_qapplication_exists()


def process_id_window_title(first_part: str) -> str:
    """Returns the window title to use when debugging which shows the process ID.

    Args:
        first_part (str):  First part of the title (the regular window title).

    Returns:
        (str): See description.
    """
    return dialog_util.process_id_window_title(first_part)


def can_add_process_id() -> bool:
    """Returns true if we can add the process ID to the window title: in dev (version 99.99) and file hack is found."""
    return dialog_util.can_add_process_id()


def add_process_id_to_window_title(dialog) -> None:
    """Adds the process ID into the window title if in dev (version 99.99) and file hack is found."""
    if can_add_process_id():
        dialog.setWindowTitle(process_id_window_title(dialog.windowTitle()))


class XmsDlg(QDialog):
    """Base class for saving and restoring window position."""
    def __init__(self, parent, dlg_name):
        """Construct the dialog.

        Args:
            parent (QObject): The dialog's Qt parent.
            dlg_name (str): Unique name for this dialog. site-packages import path would make sense.
        """
        super().__init__(parent)
        self._xms_timer = None  # Look-back poll to kill ourselves if XMS dies
        self._xms_pid = -1
        self._dlg_name = dlg_name
        self._help_getter: HelpGetter | None = None

        self._setup_window_icons()

    @property
    def help_getter(self) -> HelpGetter | None:
        """Return the help getter."""
        return self._help_getter

    @help_getter.setter
    def help_getter(self, help_getter: HelpGetter) -> None:
        """Set the help getter.

        Args:
            help_getter: The help getter.
        """
        self._help_getter = help_getter

    def _exception_hook(self, type, value, tback):
        """
        Called when an exception occurs on the UI thread.

        Qt suppresses exceptions that occur on the UI thread. This method allows dialogs to inspect and handle them
        instead. See https://stackoverflow.com/questions/1015047/logging-all-exceptions-in-a-pyqt4-app

        The default implementation logs all exceptions to the Python debug log, displays a generic error message, and
        forwards the exception to the original system exception hook. That hook just prints the stack trace to stdout,
        which in our case is normally just thrown away.

        Overrides can either compare `type` to the type of exception they want, or use
        `xms.guipy.dialogs.xms_excepthook.exceptions_equal` to compare their arguments too.

        Overrides should probably be structured something like "If I know how to handle this exception then handle it
        and return. Otherwise, call `super()._exception_hook(type, value, tback)` to let the base class deal with it".

        Args:
            type: Type of the exception that was raised, e.g. `AssertionError`.
            value: The exception that was raised, e.g. `AssertionError('Something went wrong')`.
            tback: Traceback for the exception that was raised. Making decisions based on which line an exception was
                thrown on is bound to be brittle, so most overrides should just pass this through to the base class.
        """
        message = 'Unexpected error. Please contact tech support.'
        app_name = XmsEnvironment.xms_environ_app_name()
        # Echo the traceback to 'python_debug.log' in the XMS temp directory.
        XmsEnvironment.report_error(value, XmsEnvironment.xms_environ_debug_file())
        # Report a pretty message to the user.
        message_box.message_with_ok(parent=self, message=message, app_name=app_name, details=str(value))
        sys.__excepthook__(type, value, tback)  # call the default handler

    def _check_xms_alive(self):
        """Kill the process if our parent XMS has died."""
        wmi = win32com.client.GetObject('winmgmts:')
        for p in wmi.InstancesOf('win32_process'):
            if int(p.Properties_('ProcessId')) == self._xms_pid:
                return  # Our parent XMS is still alive, continue polling.
        sys.exit(0)  # Our parent XMS is no longer running, commit suicide.

    def _setup_window_icons(self):
        """Set the window icon for appropriate XMS app and disable help menu button."""
        # Disable help icon in menu bar, it is dumb and we already have button slot to handle it.
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)
        icon_path = dialog_util.get_xms_icon()
        if os.path.isfile(icon_path):
            self.setWindowIcon(QIcon(icon_path))

    def _splitter_settings_key(self, splitter: QSplitter) -> str:
        """Returns the string used by SettingsManager as a key to find the object's settings.

        Args:
            splitter: A splitter object.

        Returns:
            See description.
        """
        return f'{self._dlg_name}.{splitter.objectName()}.sizes'

    def _restore_geometry(self):
        """Restore previous dialog size and position."""
        settings = SettingsManager()
        geometry = settings.get_setting('xmsguipy', f'{self._dlg_name}.geometry')
        if geometry:
            self.restoreGeometry(geometry)
        self._restore_splitter_sizes(settings)

    def _restore_splitter_sizes(self, settings: SettingsManager) -> None:
        """Restores the sizes of all splitters in the dialog.

        Args:
            settings: The settings manager.
        """
        splitters = self.findChildren(QSplitter)
        for splitter in splitters:
            key = self._splitter_settings_key(splitter)
            sizes = settings.get_setting('xmsguipy', key)
            if sizes:
                splitter_sizes = [int(size) for size in sizes]
                splitter.setSizes(splitter_sizes)

    def _save_geometry(self):
        """Saves current dialog size and position."""
        settings = SettingsManager()
        settings.save_setting('xmsguipy', f'{self._dlg_name}.geometry', self.saveGeometry())
        self._save_splitter_sizes(settings)

    def _save_splitter_sizes(self, settings: SettingsManager) -> None:
        """Saves the sizes of all splitters in the dialog.

        Args:
            settings: The settings manager.
        """
        splitters = self.findChildren(QSplitter)
        for splitter in splitters:
            key = self._splitter_settings_key(splitter)
            settings.save_setting('xmsguipy', key, splitter.sizes())

    def showEvent(self, event):  # noqa: N802
        """Restore window position and size."""
        add_process_id_to_window_title(self)
        self._restore_geometry()
        super().showEvent(event)

    def help_requested(self):
        """
        Slot for the Help button.

        The default implementation looks for the dialog's import path on the wiki, similar to how XMS does. In most
        cases, you shouldn't need to override this. Just have the necessary information added to the wiki. It should go
        in pages like `xmswiki.com/wiki/GMS:GMS_99.99_Dialog_Help`, with "GMS" and "99.99" changed to the appropriate
        product and version.

        Note that the page has multiple formats of links on it. This handler looks for lines that end with the import
        path to your dialog and grabs the link from that line. So for example, if you can import your dialog as
        `from xms.model.dialogs.my_dialog import MyDialog`, then there should be a line on the wiki page that looks
        something like `Page for My Dialog | xms.model.dialogs.my_dialog.MyDialog`, where "Page for My Dialog" is a link
        to where the documentation for your dialog is (the documentation team will typically decide the name and link,
        they mostly just need the part after the | and information they should put on the page).
        """
        if self._help_getter is None:
            # Either provide a self._help_getter, or override this method so it doesn't use it.
            message_box.message_with_ok(
                self, 'An internal error occurred. Please contact Aquaveo tech support.', XmEnv.xms_environ_app_name()
            )
            return

        try:
            webbrowser.open(self._help_getter.url())
        except AssertionError:
            message_box.message_with_ok(
                self, 'An internal error occurred. Please contact Aquaveo tech support.', XmEnv.xms_environ_app_name()
            )
        except URLError:
            message_box.message_with_ok(
                self, 'Help page was not found. Internet access to xmswiki.com may be unavailable.',
                XmEnv.xms_environ_app_name()
            )

    def _default_help_key(self) -> str:
        """Return a key to use when looking up the help URL."""
        type_ = type(self)
        key = f'{type_.__module__}.{type_.__name__}'
        return key

    def exec(self):
        """Overload to attach a look-back poll checking if the parent XMS process is still running.

        You can use 'MANUAL' (os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = 'MANUAL') to avoid time out when
        testing manually instead of 'TRUE' which some dialogs use as a flag to return immediately.
        """
        running_tests = XmEnv.xms_environ_running_tests()
        if running_tests not in {'TRUE', 'ACCEPT', 'REJECT', 'CANCEL', 'MANUAL'}:  # Only poll if not running tests
            self._xms_pid = int(os.environ.get(XmEnv.ENVIRON_XMS_APP_PID, -1))
            self._xms_timer = QTimer(self)
            self._xms_timer.setInterval(10000)  # Check every 10 seconds
            self._xms_timer.timeout.connect(self._check_xms_alive)
            self._xms_timer.start()

        if running_tests in {'ACCEPT', 'TRUE'}:  # Accept immediately
            self.accept()
            return QDialog.Accepted
        elif running_tests in {'REJECT', 'CANCEL'}:  # Reject immediately
            self.reject()
            return QDialog.Rejected
        else:
            # Our exception hook is only for when we're in charge of the UI thread. When we're done we'll need to put
            # the old one back.
            old_hook = sys.excepthook
            # Qt's event loop swallows most exceptions. Before we run its loop, install our hook so we can peek at them.
            sys.excepthook = self._exception_hook

            result = super().exec_()

            # If we have a parent, this will give it control of exceptions again. If we *don't* have a parent, this will
            # give control back to the standard handler so we don't eat exceptions from outside code.
            sys.excepthook = old_hook

            return result

    def accept(self):
        """Save window position and size."""
        self._save_geometry()
        super().accept()

    def reject(self):
        """Save window position and size."""
        self._save_geometry()
        super().reject()
