"""Miscellaneous stuff we use in tests."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
import os
from pathlib import Path
from random import Random
from typing import Iterable
import uuid
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment
from xms.api.tree import TreeNode
from xms.testing import file_comparison

# 4. Local modules
from xms.guipy import file_io_util

repeatable_random: Random | None = None  # Can be set to Random() and used to create repeatable random numbers and uuids


def new_uuid() -> str:
    """Returns a uuid as a string.

    See RepeatableRandomness for how to get repeated uuids when testing.
    """
    msg = 'This function is deprecated. Use xms.testing.tools.new_uuid() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)

    global repeatable_random  # noqa: F824 `global repeatable_random` is unused: name is never assigned in scope
    if repeatable_random:
        return str(uuid.UUID(int=repeatable_random.getrandbits(128), version=4))
    else:
        return str(uuid.uuid4())


class RepeatableRandomness:
    """Context manager to make random numbers repeatable."""
    def __init__(self) -> None:
        """Initializes the class."""
        msg = 'This class is deprecated. Use xms.testing.tools.repeat_uuids() instead.'
        warnings.warn(msg, DeprecationWarning, stacklevel=2)

    def __enter__(self):
        """Called on enter."""
        global repeatable_random
        repeatable_random = Random()
        repeatable_random.seed(123)
        return self

    def __exit__(self, _type, value, traceback) -> None:
        """Called on exit."""
        global repeatable_random
        repeatable_random = None


def is_binary_file(file_path: str | Path) -> bool:
    """Returns True if the file is binary.

    See https://stackoverflow.com/questions/898669/how-can-i-detect-if-a-file-is-binary-non-text-in-python

    Args:
        file_path: The file path.

    Returns:
        See description.
    """
    warnings.warn(_deprecation_msg('is_binary_file'), DeprecationWarning, stacklevel=2)
    return file_comparison.is_binary_file(file_path)


def compare_ascii_files(baseline: str | Path, output: str | Path, comments: Iterable | None = None) -> bool:
    """Compare to ascii files, optionally ignoring certain tokens.

    Args:
        baseline: Filename of the baseline
        output: Filename of the file to compare to the baseline
        comments: If comment is found at the beginning of the line, those lines are not compared

    Returns:
        (bool): True if the files are equal
    """
    warnings.warn(_deprecation_msg('ascii_files_equal'), DeprecationWarning, stacklevel=2)
    return file_comparison.ascii_files_equal(baseline, output, comments)


def compare_binary_files(base: str | Path, out: str | Path, bytes_to_skip: int = 0) -> bool:
    """Compares two binary files.

    Args:
        base (str | Path): Filepath of first file.
        out (str | Path): Filepath of second file.
        bytes_to_skip (int): Number of bytes to skip at the beginning.

    Returns:
        (bool): True if equal, else False.
    """
    warnings.warn(_deprecation_msg('binary_files_equal'), DeprecationWarning, stacklevel=2)
    return file_comparison.binary_files_equal(base, out, bytes_to_skip)


def compare_image_files(base: str | Path, out: str | Path, tolerance: float, method: str = 'rms') -> bool:
    """Compares two image files.

    Uses Pillow library. Currently only the 'rms' (root mean squared) method is available.

    Args:
        base (str | Path): Filepath of first file.
        out (str | Path): Filepath of second file.
        tolerance (float): Allowable difference between two image files (to be tested)
        method (str): Method to use for comparison. Options are 'rms' (root mean squared).

    Returns:
        (bool): True if the image difference is less than or equal to tolerance.
    """
    warnings.warn(_deprecation_msg('image_files_equal'), DeprecationWarning, stacklevel=2)
    return file_comparison.image_files_equal(base, out, tolerance, method)


def are_dir_trees_equal(
    dir1: str | Path,
    dir2: str | Path,
    skip_extensions: Iterable[str] | None = None,
    comments: Iterable[str] | None = None
) -> bool:
    """Compare two directories recursively.

    See stackoverflow.com/questions/4187564
    Files in each directory are assumed to be equal if their names and contents are equal.

    Args:
        dir1: First directory path
        dir2: Second directory path
        skip_extensions: List of extensions of files to skip when comparing
        comments: List of strings that can appear at the beginning of lines to ignore

    Return:
        True if the directory trees are the same and there were no errors while accessing the directories or files;
        False otherwise.
    """
    warnings.warn(_deprecation_msg('are_dir_trees_equal'), DeprecationWarning, stacklevel=2)
    return file_comparison.are_dir_trees_equal(dir1, dir2, skip_extensions, comments)


def write_tree_to_file(tree_node: TreeNode, filename: Path) -> None:
    """Writes the project tree to a file.

    Args:
        tree_node (TreeNode): The starting node of the project explorer tree.
        filename (Path): File path to write project tree to
    """
    the_dict = dict(tree_node)
    file_io_util.write_json_file(the_dict, filename)


def read_tree_from_file(filename: Path | str) -> TreeNode | None:
    """Reads the project tree from a file, and returns the top node, or None if trouble reading the file.

    Args:
        filename: Path to file containing the project tree.

    Returns:
        See description.
    """
    if not Path(filename).is_file():
        return None
    the_dict = file_io_util.read_json_file(filename)
    project_tree = TreeNode()
    project_tree.from_dict(the_dict)
    return project_tree


@contextlib.contextmanager
def test_environment(running_tests_str: str):
    """Context manager to change and restore the environment variable XmsEnvironment.ENVIRON_RUNNING_TESTS.

    Useful to run a test manually as well as have it run with tox.

    Args:
        running_tests_str (str): 'TRUE', 'FALSE', 'MANUAL', or '' for now. '' means don't change anything
    """
    msg = 'This function is deprecated. Use xms.testing.tools.env_running_tests() instead.'
    # stacklevel=2 is in contextlib, which is an unhelpful place to point at. Use 3 to identify the actual caller.
    warnings.warn(msg, DeprecationWarning, stacklevel=3)

    orig = XmsEnvironment.xms_environ_running_tests()  # Save the original value
    if running_tests_str != '':
        os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = running_tests_str  # Change it
    yield
    os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = orig  # Set it back to the original


@contextlib.contextmanager
def test_env(running_tests_str: str = '', temp_dir: Path | str = None) -> None:
    """Context manager to change and restore the environment variable XmsEnvironment.ENVIRON_RUNNING_TESTS.

    Useful to run a test manually as well as have it run with tox.

    Args:
        running_tests_str (str): 'TRUE', 'FALSE', 'MANUAL', or '' for now. '' means don't change anything
        temp_dir: If provided, patches 'xms.api.dmi.XmsEnvironment.xms_environ_temp_directory' to return temp_dir.
    """
    msg = 'This function is deprecated. Use xms.testing.tools.env_running_tests() and env_temp_dir() instead.'
    # stacklevel=2 is in contextlib, which is an unhelpful place to point at. Use 3 to identify the actual caller.
    warnings.warn(msg, DeprecationWarning, stacklevel=3)

    with test_env_str(running_tests_str), test_env_temp_dir(temp_dir):
        try:
            yield
        finally:
            pass


@contextlib.contextmanager
def test_env_str(running_tests_str: str) -> None:
    """Context manager to change and restore the environment variable XmsEnvironment.ENVIRON_RUNNING_TESTS.

    Args:
        running_tests_str: 'TRUE', 'FALSE', 'MANUAL', or '' for now. '' means don't change anything
    """
    msg = 'This function is deprecated. Use xms.testing.tools.env_running_tests() instead.'
    # stacklevel=2 is in contextlib, which is an unhelpful place to point at. Use 3 to identify the actual caller.
    warnings.warn(msg, DeprecationWarning, stacklevel=3)

    orig = XmsEnvironment.xms_environ_running_tests()  # Save the original value
    try:
        if running_tests_str != '':
            os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = running_tests_str  # Change it
        yield None
    finally:
        os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = orig  # Set it back to the original


@contextlib.contextmanager
def test_env_temp_dir(temp_dir: Path | str) -> None:
    """Context manager to change and restore the environment variable XmsEnvironment.ENVIRON_XMS_TEMP_FOLDER.

    Args:
        temp_dir: If provided, patches 'xms.api.dmi.XmsEnvironment.xms_environ_temp_directory' to return temp_dir.
    """
    msg = 'This function is deprecated. Use xms.testing.tools.env_temp_dir() instead.'
    # stacklevel=2 is in contextlib, which is an unhelpful place to point at. Use 3 to identify the actual caller.
    warnings.warn(msg, DeprecationWarning, stacklevel=3)

    orig = XmsEnvironment.xms_environ_temp_directory()  # Save the original value
    try:
        if temp_dir is not None:
            os.environ[XmsEnvironment.ENVIRON_XMS_TEMP_FOLDER] = str(temp_dir)
        yield
    finally:
        os.environ[XmsEnvironment.ENVIRON_XMS_TEMP_FOLDER] = orig  # Set it back to the original


def _remove_files_skipped_by_extension(files: list[str], skip_extensions: Iterable[str]) -> list[str]:
    """Removes files from the list if their extension is in skip_extensions.

    Args:
        files: List of files.
        skip_extensions: Extensions (e.g. '.txt') to skip (ignore).

    Returns:
        See description.
    """
    new_files = []
    for file in files:
        path = Path(file)
        if path.suffix not in skip_extensions and path.name not in skip_extensions:
            new_files.append(file)
    return new_files


def _deprecation_msg(new_func_name: str) -> str:
    """Return the deprecation warning message.

    Args:
        new_func_name: Name of the new function to call in xms.testing.file_comparison.

    Returns:
        See description.
    """
    return f'This function is deprecated. Use xms.testing.file_comparison.{new_func_name}() instead.'


def add_parent_paths(parent_path: str, node_dict: dict) -> None:
    """Recursive function to add parent paths to the dicts (can be useful when debugging).

    Args:
        parent_path: Path of the parent.
        node_dict: A dict representing the current node.
    """
    node_dict['parent_path'] = parent_path
    if 'children' in node_dict:
        parent_path = f'{parent_path}/{node_dict["name"]}'
        for child in node_dict['children']:
            add_parent_paths(parent_path, child)
