"""Class to build a GMI definition."""

# 1. Standard Python modules
from logging import Logger
from typing import cast, Optional, Sequence

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.components.display.display_options_helper import MULTIPLE_TYPES, UNASSIGNED_TYPE
from xms.constraint import UGrid2d
from xms.coverage.grid.polygon_coverage_builder import PolygonCoverageBuilder
from xms.coverage.xy.xy_series import XySeries
from xms.data_objects.parameters import Polygon, Projection
from xms.gmi.data.generic_model import GenericModel, Parameter, Section, Type, UNASSIGNED_MATERIAL_ID, \
    UNASSIGNED_MATERIAL_NAME
from xms.grid.ugrid import UGrid
from xms.guipy.dialogs.log_timer import Timer

# 4. Local modules
from xms.hydroas.file_io.gmi_parser import (
    BC_VAL_CARD_FIRST_VARIABLE_FIELD, CARD_INDEX, DEF_CARD_FIRST_VARIABLE_FIELD, DEP_CARD_FIRST_OPTION_INDEX,
    FIRST_FIELD, GP_VAL_CARD_FIRST_VARIABLE_FIELD, MAT_VAL_CARD_FIRST_VARIABLE_FIELD, OPTS_CARD_FIRST_OPTION_INDEX,
    OPTS_CARD_GROUP_INDEX, OPTS_CARD_PARAM_INDEX, SECOND_FIELD, SI_CARD_MAX_LENGTH
)


class GmiBuilder:
    """
    Class for building a generic model from cards.

    Assumes that the cards:
    - Were produced by GmiParser
    - Were validated by GmiValidator
    - Were fixed by GmiFixer.
    Violating these assumptions may give surprising results.
    """
    def __init__(
        self, cards: list, curves: dict[int, XySeries], default_name: str = 'Mesh', log: Optional[Logger] = None
    ):
        """
        Initialize the builder.

        Args:
            cards: Cards to build a generic model from.
            curves: Mapping from curve_id->curve. Contains all the curves in the .2dm file.
            default_name: Name to assign the mesh if it doesn't specify its own.
            log: Where to log progress to.
        """
        self._log = log or Logger('xms.gmi')
        self._timer = Timer()

        self.mesh_name = default_name  # Name of the generated mesh
        self.ugrid: Optional[UGrid2d] = None  # UGrid representing the mesh in the file.
        self.projection: Optional[Projection] = None

        self.points: list[int] = []  # Indexes of points in the mesh that have values assigned.
        self.point_ids: list[int] = []  # Feature IDs of points. Parallel to self.points.
        self.point_values: list[str] = []  # Parameter values for a point. Parallel to self.points.
        self.point_types: list[str] = []  # Active group for a point. Parallel to self.points.

        # If an arc's node is in self.points, it will have the same feature ID.
        self.arcs: list[list[int]] = []  # Arcs in the mesh that have values assigned.
        self.arc_ids: list[int] = []  # Feature IDs of arcs. Parallel to self.arcs.
        self.arc_values: list[str] = []  # Parameter values for an arc. Parallel to self.arcs.
        self.arc_types: list[str] = []  # Active group for an arc. Parallel to self.arcs.
        self.arc_names: list[str] = []  # Arc names. Parallel to self.arcs. Unnamed arcs have '' for a name.

        self.polygons: list[Polygon] = []  # Polygons in the mesh that have values assigned. *Not* renumbered.
        self.polygon_values: list[str] = []  # Parameter values for a polygon. Parallel to self.polygons.
        self.polygon_types: list[str] = []  # Active group for a polygon. Parallel to self.polygons.

        self.material_cells: Sequence[int] = []  # Material ID for each cell. Parallel to cell list in UGrid.
        self.material_types: list[str] = []  # Active group for each material.
        self.material_names = {UNASSIGNED_MATERIAL_ID: UNASSIGNED_MATERIAL_NAME}

        self.model = GenericModel(exclusive_material_conditions=True)  # Parameter definitions
        # Values for single-instance sections of the model. Can be restored with e.g.
        # self.model.global_parameters.restore_values(self.model.values)
        self.model_instantiation = None
        self.global_instantiation = None
        self.material_instantiation = None

        self._node_values = {}  # Values for node parameters
        self._arc_values = {}  # Values for arc parameters
        self._polygon_values = {}  # Values for polygon parameters

        self._curves = curves
        self._curves[-1] = XySeries([0.0], [0.0])
        self._cards = cards
        self._card = []
        self._card_index = None
        self._builders = {
            'bc': self._build_bc,
            'bc_def': self._build_bc_def,
            'bc_dep': self._scan_dep,
            'bc_disp_opts': self._null_builder,
            'bc_opts': self._build_bc_opts,
            'bc_val': self._build_bc_val,
            'bcpgc': self._build_bcpgc,
            'befont': self._build_befont,
            'beg2dmbc': self._build_beg2dmbc,
            'begparamdef': self._build_begparamdef,
            'disp_opts': self._null_builder,
            'dy': self._build_dy,
            'e3t': self._build_element,
            'e4q': self._build_element,
            'end2dmbc': self._build_end2dmbc,
            'endparamdef': self._build_endparamdef,
            'gm': self._build_gm,
            'gp': self._build_gp,
            'gp_def': self._build_gp_def,
            'gp_dep': self._scan_dep,
            'gp_opts': self._build_gp_opts,
            'gp_val': self._build_gp_val,
            'key': self._build_key,
            'mat': self._build_mat,
            'mat_def': self._build_mat_def,
            'mat_dep': self._scan_dep,
            'mat_multi': self._null_builder,
            'mat_opts': self._build_mat_opts,
            'mat_val': self._build_mat_val,
            'mesh2d': self._null_builder,
            'meshname': self._build_meshname,
            'nd': self._build_nd,
            'num_materials_per_elem': self._null_builder,
            'nume': self._null_builder,
            'ns': self._build_ns,
            'si': self._build_si,
            'td': self._build_td,
            'tu': self._build_tu,
        }

        self._global_parents = {}
        self._global_group_name_to_id = {}
        self._last_section = None
        self._bc_parents = {}
        self._material_parents = {}

        self._section_map = {
            'points': self.model.point_parameters,
            'arcs': self.model.arc_parameters,
            'polygons': self.model.polygon_parameters
        }

        self._dep_cards = []  # (index, last_section_name)

        # Mesh building stuff
        self._node_cards = []
        self._cell_cards = []
        self._nodestring_cards = []

        self._point_feature_id_to_idx = {}  # feature ID to index in self._point_locations.
        self._point_locations = []  # x, y, z locations. parallel to self._node_cards.

        self._points = {}  # feature ID to constructed Point

        self._materials_on_grid = []  # cell index -> material ID

    def build(self):
        """Build the generic model."""
        self._log.info('Processing cards...')

        self._stringify_ids()

        for index, card in enumerate(self._cards):
            if not card:
                continue
            self._card = card
            self._card_index = index
            card_name = card[CARD_INDEX]
            if self._builders[card_name]():
                break
            if self._timer.report_due:
                self._log.info(f'Processed {index} cards...')

        # *_DEP cards refer to two parents, which might appear in any order, so we postpone *_DEP cards to the end
        # after we know everything is defined.
        self._builders['bc_dep'] = self._build_bc_dep
        self._builders['gp_dep'] = self._build_gp_dep
        self._builders['mat_dep'] = self._build_mat_dep
        mapping = {
            self.model.global_parameters.name: self.model.global_parameters,
            self.model.point_parameters.name: self.model.point_parameters,
            self.model.arc_parameters.name: self.model.arc_parameters,
            self.model.polygon_parameters.name: self.model.polygon_parameters,
            self.model.material_parameters.name: self.model.material_parameters
        }
        for index, section_name in self._dep_cards:
            self._card = self._cards[index]
            self._card_index = index
            self._last_section = mapping[section_name]
            card_name = self._card[CARD_INDEX]
            self._builders[card_name]()

        self._log.info('Building domain geometry...')
        self._build_mesh()
        self._log.info('Building point coverage...')
        self._build_points()
        self._log.info('Building arc coverage...')
        self._build_arcs()
        self._log.info('Building polygon coverage...')
        self._build_polygons()
        self._log.info('Building material coverage...')
        self._build_materials()

        self._log.info('Done building geometry')
        if self.model_instantiation is not None:
            self.model_instantiation = cast(Section, self.model_instantiation)
            self.model_instantiation = self.model_instantiation.extract_values(include_default=False)
        else:
            self.model_instantiation = ''
        if self.global_instantiation is not None:
            self.global_instantiation = self.global_instantiation.extract_values(include_default=False)
        self.model = self.model.to_template()

    def _stringify_ids(self):
        """Convert group and parameter IDs to strings."""
        first_parameter_cards = {
            'gp',
            'gp_def',
            'gp_opts',
            'gp_dep',
            'bc_def',
            'bc_opts',
            'bc_dep',
            'mat',
            'mat_def',
            'mat_opts',
            'mat_dep',
            'gp_val',
            'mat_val',
        }
        second_parameter_cards = {
            'gp',
            'gp_def',
            'gp_opts',
            'gp_dep',
            'bc_def',
            'bc_opts',
            'bc_dep',
            'mat_def',
            'mat_opts',
            'mat_dep',
            'gp_val',
            'mat_val',
        }
        third_parameter_cards = {
            'bc',
            'bc_val',
            'mat_val',
        }
        fourth_parameter_cards = {'bc_val'}
        sixth_parameter_cards = {'bc'}
        for card in self._cards:
            if not card:
                continue
            if card[CARD_INDEX] in first_parameter_cards:
                card[FIRST_FIELD] = str(card[FIRST_FIELD])
            if card[CARD_INDEX] in second_parameter_cards:
                card[FIRST_FIELD + 1] = str(card[FIRST_FIELD + 1])
            if card[CARD_INDEX] in third_parameter_cards:
                card[FIRST_FIELD + 2] = str(card[FIRST_FIELD + 2])
            if card[CARD_INDEX] in fourth_parameter_cards:
                card[FIRST_FIELD + 3] = str(card[FIRST_FIELD + 3])
            if card[CARD_INDEX] in sixth_parameter_cards:
                card[FIRST_FIELD + 5] = str(card[FIRST_FIELD + 5])

    def _build_mesh(self):
        """Build the data_objects UGrid."""
        if not self._node_cards:
            return  # No nodes means no mesh. Nothing to build.

        # Parse grid geometry from previously read lines
        self._build_mesh_points()
        cellstream = [stream_val for i in range(len(self._cell_cards)) for stream_val in self._build_cellstream_vals(i)]
        # Write the CoGrid file
        ugrid = UGrid(self._point_locations, cellstream)
        self.ugrid = UGrid2d(ugrid=ugrid)
        # Create the data_objects UGrid to send back to XMS
        self.projection = Projection()

    def _build_mesh_points(self):
        """Build the grid locations list and create mapping between id in file and the point indices."""
        # Need the locations later for converting nodestrings to feature arcs, so store them off before constructing
        # the grid.
        self._point_locations = np.full((len(self._node_cards), 3), 0.0)
        for point_idx, point_line in enumerate(self._node_cards):
            self._point_feature_id_to_idx[int(point_line[1])] = point_idx
            self._point_locations[point_idx][0] = float(point_line[2])
            self._point_locations[point_idx][1] = float(point_line[3])
            self._point_locations[point_idx][2] = float(point_line[4])

    def _build_cellstream_vals(self, idx):
        """Get the XmGrid cellstream definition for the current cell line.

        Returns:
            list: [UGrid.cell_type_enum, node1_id, ..., noden_id]
        """
        cell_line = self._cell_cards[idx]
        self._materials_on_grid.append(int(cell_line[-1]))
        if cell_line[0].upper() == 'E4Q':  # Quad
            return [
                UGrid.cell_type_enum.QUAD,
                4,
                self._point_feature_id_to_idx[int(cell_line[2])],
                self._point_feature_id_to_idx[int(cell_line[3])],
                self._point_feature_id_to_idx[int(cell_line[4])],
                self._point_feature_id_to_idx[int(cell_line[5])],
            ]
        else:  # Tri
            return [
                UGrid.cell_type_enum.TRIANGLE,
                3,
                self._point_feature_id_to_idx[int(cell_line[2])],
                self._point_feature_id_to_idx[int(cell_line[3])],
                self._point_feature_id_to_idx[int(cell_line[4])],
            ]

    def _build_points(self):
        """Build the points."""
        for node_card in self._node_cards:
            feature_id = node_card[FIRST_FIELD]
            if feature_id in self._node_values:
                node_index = self._point_feature_id_to_idx[feature_id]

                self.points.append(node_index)
                self.point_ids.append(feature_id)
                section = self._node_values[feature_id]
                self.point_values.append(section.extract_values(include_default=False))
                self.point_types.append(section.active_group_name(UNASSIGNED_TYPE, MULTIPLE_TYPES))

    def _build_arcs(self):
        """Build the arcs."""
        for nodestring_line in self._nodestring_cards:
            ns_name = nodestring_line.pop()  # This is always here thanks to the fixer
            feature_id = nodestring_line.pop()

            nodes = [self._point_feature_id_to_idx[feature_id] for feature_id in nodestring_line[FIRST_FIELD:]]

            self.arcs.append(nodes)
            self.arc_ids.append(feature_id)
            if feature_id in self._arc_values:
                section = self._arc_values[feature_id]
            else:
                section = self.model.arc_parameters.copy()
                section.clear_values()
            self.arc_values.append(section.extract_values(include_default=False))
            self.arc_types.append(section.active_group_name(UNASSIGNED_TYPE, MULTIPLE_TYPES))
            self.arc_names.append(ns_name)

    def _build_polygons(self):
        """Build the polygons."""
        multipolys = {}

        for cell_card in self._cell_cards:
            cell_id = cell_card[FIRST_FIELD]
            if cell_id not in self._polygon_values:
                continue

            node_ids = cell_card[SECOND_FIELD:-1]
            node_indices = [self._point_feature_id_to_idx[node_id] for node_id in node_ids]
            node_indices.append(node_indices[0])
            multipolys[len(multipolys)] = [[node_indices]]

            self.polygon_values.append(self._polygon_values[cell_id].extract_values(include_default=False))
            # Ideally we'd have a "Multiple" type, but we'll just pick the first one for now.
            self.polygon_types.append(self._polygon_values[cell_id].active_group_names[0])

        builder = PolygonCoverageBuilder(self._point_locations, None, '')
        coverage = builder.build_coverage(multipolys)
        self.polygons = coverage.polygons

    def _build_materials(self):
        """Build the data_objects Materials coverage."""
        values = self.model.material_parameters.copy()
        values.deactivate_groups()
        self.material_instantiation = values.extract_values(include_default=False)

        if not self.ugrid or len(self.material_names) == 1:  # length==1 implies only 'Unassigned' is present
            return

        self.material_cells = self._materials_on_grid

        all_dataset_values = set(self.material_cells)
        all_dataset_values.discard(0)  # Don't make a type for unassigned polygons.
        all_names = [str(value) for value in all_dataset_values]
        self.material_types = all_names

    def _apply_parameter_value(self, parameter: Parameter, value):
        """Apply a value to a parameter, handling curves specially as needed."""
        if parameter.parameter_type == Type.FLOAT_CURVE:
            new_value = list(parameter.value)  # Will be default if not set, or previous if set
            if value[0] == 'FLOAT':
                new_value[0] = 'FLOAT'
                new_value[1] = value[1]
            elif value[0] == 'CURVE':
                curve = self._curves[value[1]]
                new_value[0] = 'CURVE'
                new_value[2] = curve.x
                new_value[3] = curve.y
            else:
                raise AssertionError('Unknown curve type')  # pragma: nocover  New type added without adding support
            parameter.value = tuple(new_value)
        elif parameter.parameter_type == Type.CURVE:
            curve = self._curves[value]
            parameter.value = (curve.x, curve.y)
        else:
            parameter.value = value

    def _null_builder(self):
        """
        Build a card that doesn't actually need to be built.

        These are mainly deprecated cards, or things with redundant information we ignore.
        """
        pass

    def _build_bc(self):
        """Build a BC card."""
        _card, entity_name, name, bc_id, _filler, legal_on_interior, gp_group = self._card
        section = self._section_map[entity_name]
        self._last_section = section
        correlation = self._global_group_name_to_id[gp_group] if gp_group and gp_group != '(none)' else None
        section.add_group(bc_id, name, is_active=None, allow_interior=legal_on_interior, correlation=correlation)

    def _build_bc_def(self):
        """Build a BC_DEF card."""
        bc_id, param_id, param_name, param_type = self._card[FIRST_FIELD:DEF_CARD_FIRST_VARIABLE_FIELD]
        group = self._last_section.group(bc_id)

        if param_type == 0:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_boolean(param_id, param_name, default)
        elif param_type == 1:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_integer(param_id, param_name, default, low, high)
        elif param_type == 2:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float(param_id, param_name, default, low, high)
        elif param_type == 3:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_text(param_id, param_name, default)
        elif param_type == 4:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_option(param_id, param_name, default)
            self._bc_parents.setdefault(self._last_section.name, {})
            self._bc_parents[self._last_section.name].setdefault(bc_id, {})
            self._bc_parents[self._last_section.name][bc_id][param_name] = param_id
        elif param_type == 5:
            axes = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_curve(param_id, param_name, axes)
        elif param_type == 6:
            default_value, low, high, default_mode, x_axis, y_axis = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float_curve(param_id, param_name, [x_axis, y_axis], default_value, low, high, default_mode)
        else:
            # This should have been rejected by the parser, or a new type was added without support here.
            raise AssertionError("Unknown parameter type")  # pragma: nocover

    def _scan_dep(self):
        """Collect the necessary information for an *_DEP card so it can be built later."""
        card = self._card[CARD_INDEX]
        if card == 'gp_dep':
            name = self.model.global_parameters.name
        elif card == 'mat_dep':
            name = self.model.material_parameters.name
        elif card == 'bc_dep':
            name = self._last_section.name
        else:
            raise AssertionError('Unknown card')  # pragma: nocover
        self._dep_cards.append((self._card_index, name))

    def _build_bc_dep(self):
        """Build a BC_DEP card."""
        group_id, param_id, _dependency_type, parent_name, _filler = self._card[FIRST_FIELD:DEP_CARD_FIRST_OPTION_INDEX]
        flags_list = self._card[DEP_CARD_FIRST_OPTION_INDEX:]

        parent_id = self._bc_parents[self._last_section.name][group_id][parent_name]
        parent = [self._last_section.name, group_id, parent_id]

        flags = {}
        flags_list = iter(flags_list)
        for value, enabled in zip(flags_list, flags_list):
            flags[value] = enabled

        parameter = self._last_section.group(group_id).parameter(param_id)
        parameter.add_dependency(parent, flags)

    def _build_bc_opts(self):
        """Build a BC_OPTS card."""
        bc_id, param_id = self._card[OPTS_CARD_GROUP_INDEX], self._card[OPTS_CARD_PARAM_INDEX]
        options = self._card[OPTS_CARD_FIRST_OPTION_INDEX:]
        parameter = self._last_section.group(bc_id).parameter(param_id)
        parameter.options = options

    def _build_bc_val(self):
        """Build a BC_VAL card."""
        _card, entity_name, item_id, bc_id, param_id = self._card[CARD_INDEX:BC_VAL_CARD_FIRST_VARIABLE_FIELD]
        section = self._section_map[entity_name]

        value = self._card[BC_VAL_CARD_FIRST_VARIABLE_FIELD:]
        if len(value) == 1:
            value = value[0]

        if entity_name == 'points':
            values = self._node_values
        elif entity_name == 'arcs':
            values = self._arc_values
        elif entity_name == 'polygons':
            values = self._polygon_values
        else:
            # This is just to help catch programming errors. It shouldn't ever be hit normally.
            raise AssertionError('Unknown section')  # pragma: no cover

        if item_id not in values:
            values[item_id] = section.copy()

        group = values[item_id].group(bc_id)
        group.is_active = True

        parameter = group.parameter(param_id)
        self._apply_parameter_value(parameter, value)

    def _build_bcpgc(self):
        """Build the BCPGC card."""
        self.model_instantiation.group('model').parameter('correlation').value = self._card[FIRST_FIELD]

    def _build_befont(self):
        """Build a BEFONT card."""
        pass
        # This is a best guess at how to build the card. It could use some clarity improvement.
        # It's also possible this information should be stored in something like CategoryDisplayOption instead.

        # entity = self._card[FIRST_FIELD]
        # fields = self._card[FIRST_FIELD + 1:]
        # size_field = 1
        # name_field = 0
        # label_field = 1
        # val_field = 2
        # # In C++ fonts break down to a LOGFONT structure, so these are correlated to the defaults for LOGFONT
        # default_attributes = [
        #     ['height', 'Height', 0],
        #     ['size', 'Size', 1],
        #     ['escapement', 'Escapement', 1],
        #     ['orientation', 'Orientation', 1],
        #     ['weight', 'Weight', 400],
        #     ['italic', 'Italic', 0],
        #     ['underline', 'Underline', 0],
        #     ['strikeout', 'Strikeout', 0],
        #     ['character_set', 'Character set', 0],
        #     ['precision', 'Precision', 0],
        #     ['clip_precision', 'Clip precision', 0],
        #     ['quality', 'Quality', 0],
        #     ['pitch_and_family', 'Pitch and family', 0],
        #     ['face_name', 'Face name', ''],
        # ]
        #
        # if len(self._card) == BEFONT_CARD_SHORT_LENGTH:
        #     default_attributes[size_field][val_field] = self._card[SECOND_FIELD]
        # else:
        #     for i, attribute in enumerate(default_attributes):
        #         attribute[val_field] = fields[i]
        #
        # group_name = f'font_{entity}'
        # group_label = f'Font attributes - {entity}'
        #
        # group = self.model.model_parameters.add_group(group_name, group_label, is_active=True)
        #
        # for attribute in default_attributes:
        #     name = attribute[name_field]
        #     label = attribute[label_field]
        #     val = attribute[val_field]
        #     if isinstance(val, int):
        #         if name == 'size':
        #             group.add_integer(name, label, val, low=1, high=2)
        #         else:
        #             group.add_integer(name, label, val)
        #     else:
        #         group.add_text(name, label, val)

    def _build_beg2dmbc(self):
        """Build the BEG2DMBC card."""
        self.global_instantiation = self.model.global_parameters.copy()

    def _build_begparamdef(self):
        """
        Add all the predefined model metadata parameters.
        """
        group = self.model.model_parameters.add_group('model', 'Model')

        group.add_boolean('correlation', 'Boundary condition/parameter group correlation', False)
        group.add_boolean('is_dynamic', 'Model is dynamic', False)

        group.add_text('model_name', 'Model name', '')
        group.add_text('key', 'Key', '')
        group.add_text('time_units', 'Time units', '')

        group.add_float('time_step', 'Time step length', 0.0, low=0.0)
        group.add_float('total_time', 'Total time', 0.0, low=0.0)

        standards = ['feet', 'meters', 'geographic (lat/lon)', 'feet (international)']
        group.add_option('distance_units', 'Distance units', 'meters', standards)

        self.model_instantiation = self.model.model_parameters.copy()

        self.model.material_parameters.add_group(UNASSIGNED_MATERIAL_ID, UNASSIGNED_MATERIAL_NAME)

    def _build_dy(self):
        """Build the DY card."""
        self.model_instantiation.group('model').parameter('is_dynamic').value = self._card[FIRST_FIELD]

    def _build_element(self):
        """Build an E3T or E4Q card."""
        self._cell_cards.append(self._card)

    @staticmethod
    def _build_end2dmbc():
        """Build the END2DMBC card."""
        return True

    def _build_endparamdef(self):
        """Build the ENDPARAMDEF card."""
        for material_id, material_label in self.material_names.items():
            if material_id == UNASSIGNED_MATERIAL_ID:
                self.model.material_parameters.group(UNASSIGNED_MATERIAL_ID).label = material_label
            else:
                # Fixer moved everything to group 0, so it's always there and we can just copy it.
                self.model.material_parameters.duplicate_group(UNASSIGNED_MATERIAL_ID, material_id, material_label)

    def _build_gp(self):
        """Build a GP card."""
        _card, group_id, group_name, active = self._card
        self._global_group_name_to_id[group_name] = group_id
        self.model.global_parameters.add_group(group_id, group_name, is_active=active)

    def _build_gp_def(self):
        """Build a GP_DEF card."""
        group_id, param_id, param_name, param_type = self._card[FIRST_FIELD:DEF_CARD_FIRST_VARIABLE_FIELD]
        group = self.model.global_parameters.group(group_id)

        if param_type == 0:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_boolean(param_id, param_name, default)
        elif param_type == 1:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_integer(param_id, param_name, default, low, high)
        elif param_type == 2:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float(param_id, param_name, default, low, high)
        elif param_type == 3:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_text(param_id, param_name, default)
        elif param_type == 4:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_option(param_id, param_name, default)
            self._global_parents.setdefault(group_id, {})
            self._global_parents[group_id][param_name] = param_id
        elif param_type == 5:
            axes = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_curve(param_id, param_name, axes)
        elif param_type == 6:
            default_value, low, high, default_mode, x_axis, y_axis = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float_curve(param_id, param_name, [x_axis, y_axis], default_value, low, high, default_mode)
        else:
            # This should have been rejected by the parser, or a new type was added without support here.
            raise AssertionError("Unknown parameter type")  # pragma: nocover

    def _build_gp_dep(self):
        """Build a GP_DEP card."""
        group_id, param_id, _dependency_type, parent_name, _filler = self._card[FIRST_FIELD:DEP_CARD_FIRST_OPTION_INDEX]
        flags_list = self._card[DEP_CARD_FIRST_OPTION_INDEX:]

        parent_id = self._global_parents[group_id][parent_name]
        parent = ['global', group_id, parent_id]

        flags = {}
        flags_list = iter(flags_list)
        for value, enabled in zip(flags_list, flags_list):
            flags[value] = enabled

        parameter = self.model.global_parameters.group(group_id).parameter(param_id)
        parameter.add_dependency(parent, flags)

    def _build_gp_opts(self):
        """Build a GP_OPTS card."""
        _card_name, group_id, param_id = self._card[CARD_INDEX:OPTS_CARD_FIRST_OPTION_INDEX]
        options = self._card[OPTS_CARD_FIRST_OPTION_INDEX:]
        parameter = self.model.global_parameters.group(group_id).parameter(param_id)
        parameter.options = options

    def _build_gp_val(self):
        """Build a GP_VAL card."""
        group_id, param_id = self._card[FIRST_FIELD:GP_VAL_CARD_FIRST_VARIABLE_FIELD]
        value = self._card[GP_VAL_CARD_FIRST_VARIABLE_FIELD:]
        if len(value) == 1:
            value = value[0]
        parameter = self.global_instantiation.group(group_id).parameter(param_id)
        self._apply_parameter_value(parameter, value)

    def _build_gm(self):
        """Build the GM card."""
        self.model_instantiation.group('model').parameter('model_name').value = self._card[FIRST_FIELD]

    def _build_key(self):
        """Build the KEY card."""
        self.model_instantiation.group('model').parameter('key').value = self._card[FIRST_FIELD]

    def _build_mat(self):
        """Build a MAT card."""
        mat_id, mat_name = self._card[FIRST_FIELD:]
        self.material_names[mat_id] = mat_name

    def _build_mat_def(self):
        """Build a MAT_DEF card."""
        group_id, param_id, param_name, param_type = self._card[FIRST_FIELD:DEF_CARD_FIRST_VARIABLE_FIELD]
        # We need all the material parameters in group 0 so we can duplicate it to make new materials,
        # but we want to preserve the original IDs for export.
        group = self.model.material_parameters.group('0')
        param_id = _material_param_id(group_id, param_id)

        if param_type == 0:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_boolean(param_id, param_name, default)
        elif param_type == 1:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_integer(param_id, param_name, default, low, high)
        elif param_type == 2:
            default, low, high = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float(param_id, param_name, default, low, high)
        elif param_type == 3:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_text(param_id, param_name, default)
        elif param_type == 4:
            default = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]
            group.add_option(param_id, param_name, default)
            self._material_parents.setdefault(group_id, {})
            self._material_parents[group_id][param_name] = param_id
        elif param_type == 5:
            axes = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_curve(param_id, param_name, axes)
        elif param_type == 6:
            default_value, low, high, default_mode, x_axis, y_axis = self._card[DEF_CARD_FIRST_VARIABLE_FIELD:]
            group.add_float_curve(param_id, param_name, [x_axis, y_axis], default_value, low, high, default_mode)
        else:
            # This should have been rejected by the parser, or a new type was added without support here.
            raise AssertionError("Unknown parameter type")  # pragma: nocover

    def _build_mat_dep(self):
        """Build a MAT_DEP card."""
        group_id, param_id, _dependency_type, parent_name, _filler = self._card[FIRST_FIELD:DEP_CARD_FIRST_OPTION_INDEX]
        flags_list = self._card[DEP_CARD_FIRST_OPTION_INDEX:]

        parent_id = self._material_parents[group_id][parent_name]
        parent = ['materials', '0', parent_id]

        flags = {}
        flags_list = iter(flags_list)
        for value, enabled in zip(flags_list, flags_list):
            flags[value] = enabled

        param_id = _material_param_id(group_id, param_id)
        for name in self.model.material_parameters.group_names:
            # MAT_DEP cards reference two other cards, and there's no guarantee that both of them already appeared
            # before the MAT_DEP card, so we postpone all the MAT_DEP processing to the end after we know everything is
            # already defined. Unfortunately, this means the materials have already all been duplicated by all the MAT
            # cards, so we have to patch them all up.
            parameter = self.model.material_parameters.group(name).parameter(param_id)
            parameter.add_dependency(parent, flags)

    def _build_mat_opts(self):
        """Build a MAT_OPTS card."""
        group_id, param_id = self._card[FIRST_FIELD:OPTS_CARD_FIRST_OPTION_INDEX]
        param_id = _material_param_id(group_id, param_id)
        options = self._card[OPTS_CARD_FIRST_OPTION_INDEX:]
        parameter = self.model.material_parameters.group('0').parameter(param_id)
        parameter.options = options

    def _build_mat_val(self):
        """Build a MAT_VAL card."""
        material_id, group_id, param_id = self._card[FIRST_FIELD:MAT_VAL_CARD_FIRST_VARIABLE_FIELD]

        value = self._card[MAT_VAL_CARD_FIRST_VARIABLE_FIELD:]
        if len(value) == 1:
            value = value[0]

        param_id = _material_param_id(group_id, param_id)
        parameter = self.model.material_parameters.group(material_id).parameter(param_id)
        self._apply_parameter_value(parameter, value)

    def _build_meshname(self):
        """Build the MESHNAME card."""
        self.mesh_name = self._card[FIRST_FIELD]

    def _build_nd(self):
        """Build an ND card."""
        self._node_cards.append(self._card)

    def _build_ns(self):
        """Build an NS card."""
        # We can safely assume there are no multi-line NS cards thanks to the fixer.
        self._card[-3] = -self._card[-3]  # Make the negative ID positive.
        self._nodestring_cards.append(self._card)

    def _build_si(self):
        """Build the SI card."""
        standard_index = self._card[FIRST_FIELD]
        if standard_index == 0 and len(self._card) == SI_CARD_MAX_LENGTH:
            standard_index = 3
        parameter = self.model_instantiation.group('model').parameter('distance_units')
        parameter.value = parameter.options[standard_index]

    def _build_td(self):
        """Build the TD card."""
        group = self.model_instantiation.group('model')
        group.parameter('time_step').value = self._card[FIRST_FIELD]
        group.parameter('total_time').value = self._card[SECOND_FIELD]

    def _build_tu(self):
        """Build the TU card."""
        self.model_instantiation.group('model').parameter('time_units').value = self._card[FIRST_FIELD]


def _material_param_id(group_id: int, param_id: int):
    """We want all the materials to be in group 0 so they're easy to duplicate, but preserve original IDs for export."""
    return f'{group_id} {param_id}'
