"""Functions to help with duplicating a simulation."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import re

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util


def _multireplace(string: str, replacements: dict[str, str], ignore_case: bool = False) -> str:
    """Given a string and a replacement map, it returns the replaced string.

    https://stackoverflow.com/questions/6116978/how-to-replace-multiple-substrings-of-a-string
    https://gist.github.com/bgusach/a967e0587d6e01e889fd1d776c5f3729

    Args:
        string: string to execute replacements on
        replacements: replacement dictionary {value to find: value to replace}
        ignore_case: whether the match should be case insensitive

    Returns:
        (str): The fixed string.
    """
    if not replacements:
        # Edge case that'd produce a funny regex and cause a KeyError
        return string

    # If case insensitive, we need to normalize the old string so that later a replacement
    # can be found. For instance with {"HEY": "lol"} we should match and find a replacement for "hey",
    # "HEY", "hEy", etc.
    if ignore_case:

        def normalize_old(s):
            return s.lower()

        re_mode = re.IGNORECASE

    else:

        def normalize_old(s):
            return s

        re_mode = 0

    replacements = {normalize_old(key): val for key, val in replacements.items()}

    # Place longer ones first to keep shorter substrings from matching where the longer ones should take place
    # For instance given the replacements {'ab': 'AB', 'abc': 'ABC'} against the string 'hey abc', it should produce
    # 'hey ABC' and not 'hey ABc'
    rep_sorted = sorted(replacements, key=len, reverse=True)
    rep_escaped = map(re.escape, rep_sorted)

    # Create a big OR regex that matches any of the substrings to replace
    pattern = re.compile("|".join(rep_escaped), re_mode)

    # For each match, look up the new string in the replacements, being the key the normalized old string
    return pattern.sub(lambda match: replacements[normalize_old(match.group(0))], string)


def _replace_uuids_in_file(filepath: str, old_to_new_uuids: dict[str, str]) -> None:
    """Replace old uuids in file with new ones.

    Args:
        filepath (str): Path to file.
        old_to_new_uuids (dict[str, str]): Dict of old uuids and their new uuids.
    """
    # Read the file as one string
    file_string = ''  # The entire file as one string
    with open(filepath, 'r') as file:
        file_string = file.read()

    # Overwrite the original file with the new lines
    new_string = _multireplace(file_string, old_to_new_uuids)
    if Path(filepath).is_file():  # Needed to fix weird test failures (
        with open(filepath, 'w') as file:
            file.write(new_string)


def _replace_all_uuids_in_folder(new_main_file: str, old_to_new_uuids: dict[str, str]) -> None:
    """Replace any old uuids in all files in the new component folder, recursively.

    Args:
        new_main_file (str): Path to the new main file.
        old_to_new_uuids (dict[str, str]): Dict of old uuids and their new uuids.
    """
    filepaths = Path(new_main_file).parent.glob('**/*')
    for filepath in filepaths:
        if filepath.is_file() and not io_util.file_is_binary(filepath) and filepath.suffix not in {'.grb'}:
            _replace_uuids_in_file(str(filepath), old_to_new_uuids)


def replace_old_uuids(files: dict) -> None:
    """Replace all old uuids with new ones in all duplicated folders.

    Args:
        files (dict): Dict from old main file to duplicated main file
    """
    # Get a dict of all the old uuids and their new uuids
    old_to_new_uuids = {io_util.uuid_from_path(o): io_util.uuid_from_path(n) for o, n in files.items()}

    for _old_main_file, new_main_file in files.items():
        _replace_all_uuids_in_folder(new_main_file, old_to_new_uuids)
