"""PackageComponentBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from typing import List, Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.api.tree import tree_util, TreeNode
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.guipy.dialogs import message_box, xms_parent_dlg
from xms.testing import tools
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.components import dialog_runner, dmi_util, duplication
from xms.mf6.components.component_creator import (
    add_and_link,
    create_add_component_action,
    create_data_objects_component,
)
from xms.mf6.components.default_package_creator import DefaultPackageCreator
from xms.mf6.components.mf6_component_base import Mf6ComponentBase
from xms.mf6.data import data_util
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.data.model_data_base import ModelDataBase
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui import gui_util
from xms.mf6.gui.package_dialog import PackageDialog
from xms.mf6.misc.misc_type_aliases import ActionRv


class PackageComponentBase(Mf6ComponentBase):
    """A Dynamic Model Interface (DMI) component for a GWF (Groundwater Flow) model."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.ftype = ''  # package ftype
        self.dialog = PackageDialog  # dialog class

    def ensure_disp_opts_file_exists(self) -> None:
        """Copy the default display options json file if needed."""
        json_file = f'{self.ftype}_display_options.json'
        dmi_util.ensure_disp_opts_file_exists(self.main_file, json_file)

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))
        actions = []
        if save_type == 'DUPLICATE':
            self.duplicate(self.main_file, new_main_file)
            if self.ftype in BaseFileData.displayable_ftypes():
                actions.append(self._make_duplicate_display_action(new_main_file))
        return new_main_file, [], actions

    def duplicate(self, old_main_file: Path | str, new_main_file: Path | str) -> None:
        """Called after duplicating the main file so paths can be updated.

        We fix paths in our main file from the old uuid to the new one.

        Args:
            old_main_file: Path to the old main file.
            new_main_file: Path to the new main file.
        """
        dmi_util.duplicate_display_opts(new_main_file)
        duplication.replace_old_uuids({old_main_file: str(new_main_file)})

    def _make_duplicate_display_action(self, new_main_file: str) -> ActionRequest:
        """Create an action request to update the display of a duplicated component.

        Arguments:
            new_main_file: Path to the duplicated component's main file

        Returns:
            (ActionRequest): See description.
        """
        return ActionRequest(
            main_file=new_main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='_duplicate_display_action',
            comp_uuid=io_util.uuid_from_path(new_main_file)
        )

    def _duplicate_display_action(self, query: Query, params: Optional[List[dict]]) -> tuple:
        """Initialize the display of all our displayable children after a tree duplicate.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        my_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        ugrid_uuid = dmi_util.ugrid_uuid_from_model_node(my_node.parent)
        if not ugrid_uuid:
            return [], []
        for disp_file in Path(self.main_file).parent.rglob('*_display_options.json'):
            self.display_option_list.append(XmsDisplayMessage(file=str(disp_file), edit_uuid=ugrid_uuid))
        return [], []

    def create_event(self, lock_state):
        """This will be called after the component (self) is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        with open(self.main_file, 'w') as _:
            pass

        new_component_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='new_package'
        )
        messages = []
        action_requests = [new_component_action]
        return messages, action_requests

    def delete_event(self, lock_state):
        """This will be called when the component is deleted.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change or delete the files if
                locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        # If our parent name file exists, create an ActionRequest so we can rewrite the name file to remove ourselves
        comp_uuid = io_util.uuid_from_path(self.main_file)
        comp_dir = Path(self.main_file).parent.parent
        if not _find_parent_name_file(comp_dir, comp_uuid):  # If not found, we're probably deleting the parent
            return messages, action_requests

        action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='on_delete'
        )
        action_requests.append(action)
        return messages, action_requests

    def delete_component(self, query: Query, params: Optional[List[dict]]) -> ActionRv:
        """Deletes some other component specified in params.

        See _create_delete_action().

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.

        Returns:
            ActionRv.
        """
        if params and 'component_uuid' in params[0]:
            comp_uuid = params[0]['component_uuid']  # Uuid of component to be deleted
            query.delete_item(comp_uuid)
        return [], []

    def new_package(self, query: Query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Adds a new package.

        The component (self) has already been created. We need to create the correct main file.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont (PySide2.QtWidgets.QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages: list[tuple[str]] = []
        actions: list[ActionRequest] = []

        tree_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        parent = tree_node.parent

        # If this component is array-based and the model uses DISU, delete this and create list-based equivalent
        if _array_based_and_disu(tree_node, parent):
            return self._replace_with_list_based_equivalent(tree_node, query)

        mfsim, model, _ = self.read_sim(query)
        dis_enum = self._get_dis_enum(model)
        creator = DefaultPackageCreator()
        mfsim_dir = os.path.dirname(mfsim.filename)
        unique_name = tree_node.unique_name
        creator.create_package(mfsim_dir, self.main_file, unique_name, parent.name, dis_enum, True, model)
        _rewrite_parent_name_file(mfsim, model, parent)
        _rename_node_if_necessary(tree_node, query)
        return messages, actions

    def _replace_with_list_based_equivalent(self, tree_node: TreeNode, query: Query) -> ActionRv:
        """Replace this array-based component with a list-based equivalent.

            This is called when the model uses DISU and the user selects New Package and picks an array-based package
            (uses READASARRAYS). Array-based packages cannot be used with DISU, so we create the list-based equivalent
            package and delete this component. Ideally we would not allow the user to create array-based packages in
            this scenario, but currently there is no way to change what is in the New Package list.

        Args:
            tree_node: This component's tree node.
            query: Object for communicating with GMS.

        Returns:
            See description.
        """
        # Call a function that is easier to test
        klass = self.__class__  # Get self's subclass
        new_name = _unique_name_no_spaces(tree_node, with_the_a=False)
        return _replace_with_list_based(
            self.main_file, tree_node.unique_name, tree_node.parent.uuid, klass, new_name, query
        )

    def on_delete(self, query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Called AFTER delete_event(), allowing us to rewrite parent name file.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont (PySide2.QtWidgets.QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages: list[tuple[str]] = []
        actions: list[ActionRequest] = []

        # It appears that we can't get the tree_node or the parent tree node from the query
        # tree_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        # parent = tree_node.parent
        # parent = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())

        # Get the uuid from the path, find the name file containing the uuid, remove that line from the name file
        comp_uuid = io_util.uuid_from_path(self.main_file)
        comp_dir = Path(self.main_file).parent.parent
        parent_name_file = _find_parent_name_file(comp_dir, comp_uuid)
        _remove_from_parent_name_file(comp_uuid, parent_name_file)

        return messages, actions

    def _get_dis_enum(self, model: ModelDataBase) -> DisEnum:
        """Returns the DisEnum of the DIS package used by the model.

        Returns DisEnum.DIS if model_node is None.

        Args:
            model: The model.

        Returns:
            See description
        """
        if model:
            return model.grid_info().dis_enum
        return DisEnum.DIS

    @override
    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer
        self._add_tree_menu_command('Open...', 'open_dialog', '', menu_list=menu_list, main_file=main_file_list[0][0])
        menu_list.append(None)  # None == spacer

        return menu_list

    def open_dialog(self, query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Opens the package dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages: list[tuple[str]] = []
        actions = self._open_dialog(query, win_cont)
        return messages, actions

    def _open_dialog(self, query: Query, win_cont: Optional[QWidget]) -> list[ActionRequest]:
        """Open the package dialog.

        Args:
            query: Object for communicating with GMS
            win_cont: The window container.

        Returns:
            list[ActionRequest]: List of action requests.
        """
        if not self.dialog:
            message_box.message_with_ok(
                parent=win_cont, message='The interface for this package has not yet been implemented.'
            )
            return []
        return dialog_runner.run_dialog(query=query, win_cont=win_cont, comp=self)

    def _get_sim_and_model_tree_nodes_and_mfsim_nam(self, package_node: TreeNode):
        """Returns the simulation tree node, the model tree node (if under a model), and the mfsim.nam filepath.

        Args:
            package_node: This components tree node.

        Returns:
            (tuple(TreeNode, TreeNode, str)): See description.
        """
        model_node = None
        if self.ftype in data_util.sim_child_ftypes():
            sim_node = package_node.parent
        elif self.ftype in data_util.model_ftypes():
            sim_node = package_node.parent
            model_node = package_node
        else:
            sim_node = package_node.parent.parent
            model_node = package_node.parent
        mfsim_nam = sim_node.main_file
        return sim_node, model_node, mfsim_nam

    def read_sim(self, query: Query) -> tuple[MfsimData, ModelDataBase, BaseFileData]:
        """Reads the mfsim.nam and returns the MfsimData, the model_node (TreeNode), and the package data object.

        Args:
            query: Object for communicating with GMS

        Returns:
            (tuple): tuple containing:
                - The mfsim.
                - The model.
                - The package data object.
        """
        # Get the simulation node, the mfsim.nam file, and the model_node
        current_item_uuid = query.current_item_uuid()
        package_node = tree_util.find_tree_node_by_uuid(query.project_tree, current_item_uuid)
        sim_node, model_node, mfsim_nam = self._get_sim_and_model_tree_nodes_and_mfsim_nam(package_node)

        # Read the sim
        reader = io_factory.reader_from_ftype('MFSIM6')
        mfsim = reader.read(mfsim_nam, sim_node, query)

        # Find the package
        model = reader.node_to_data.get(model_node.uuid) if model_node else None
        package = reader.node_to_data.get(package_node.uuid)

        return mfsim, model, package

    def update_displayed_cell_indices(self, model_mainfile, data):
        """Updates the cell ids used to display symbols.

        Args:
            model_mainfile: Path to the model main file.
            data: Package data class.
        """
        data.update_displayed_cell_indices()
        return dmi_util.update_display_action(data.model.ftype, model_mainfile, self.main_file)

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        open_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='open_dialog',
            parameters={"main_file": self.main_file}
        )
        actions.append(open_action)

        return messages, actions

    # def on_map_from_shapefile(self, query, params, win_cont):
    #     """Called by Arrays to Datasets menu command.
    #
    #     Args:
    #         query: Object for communicating with GMS
    #         params: ActionRequest parameters
    #         win_cont: The window container.
    #
    #     Returns:
    #         (tuple): tuple containing:
    #             - messages (list of tuple of str): List of tuples with the first element of the
    #               tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
    #               text.
    #             - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
    #     """
    #     del params  # Unused parameter
    #     messages = []
    #     actions = []
    #
    #     # Warn and above if locked
    #     locked = query.current_item().locked
    #     if locked:
    #         message_box.message_with_ok(parent=win_cont, message='Package is locked.')
    #         return messages, actions
    #
    #     previous_shapefile = Settings.get(self.main_file, 'MAP_SHAPEFILE', '')
    #     append_or_replace = Settings.get(self.main_file, 'MAP_SHAPEFILE_REPLACE', True)
    #
    #     # Get the shapefile from the user
    #     shapefilepath, append_or_replace = map_from_shapefile_dialog.run_dialog(
    #         previous_shapefile, append_or_replace, win_cont
    #     )
    #     if not shapefilepath:
    #         return messages, actions
    #     Settings.set(self.main_file, 'MAP_SHAPEFILE', shapefilepath)
    #     Settings.set(self.main_file, 'MAP_SHAPEFILE_REPLACE', append_or_replace)
    #
    #     model_node, package = map_runner.map_from_shapefile(query, self, shapefilepath, append_or_replace, win_cont)
    #
    #     # Update display
    #     if package.ftype in BaseFileData.displayable_ftypes():
    #         actions.append(self.update_displayed_cell_indices(model_node.main_file, package))
    #
    #     return messages, actions


def _rewrite_parent_name_file(mfsim: MfsimData, model: ModelDataBase, parent: TreeNode) -> None:
    """Rewrites the parent - either the model name file, or the mfsim.nam file.

    Args:
        mfsim: Sim data.
        model: Model data.
        parent: Parent tree node.
    """
    dmi_sim_dir = str(Path(mfsim.filename).parent.parent)
    mfsim_dir = str(Path(mfsim.filename).parent)
    writer_options = WriterOptions(mfsim_dir=mfsim_dir, dmi_sim_dir=dmi_sim_dir, just_name_file=True)
    if parent.unique_name in data_util.model_ftypes():
        model.write(writer_options)
    elif parent.item_typename == 'TI_DYN_SIM':
        mfsim.write(writer_options)


def _find_parent_name_file(comp_dir: Path, comp_uuid: str) -> Path | None:
    """Finds the name file (either a model name file or mfsim.nam) containing comp_uuid.

    Args:
        comp_dir: 'Components' directory.
        comp_uuid: The component uuid string.

    Returns:
        Filepath of name file, or None if not found.
    """
    name_file = None
    for filepath in comp_dir.rglob('*'):  # rglob to search recursively
        if filepath.is_file() and filepath.suffix == '.nam':
            try:
                with open(filepath, 'r') as file:
                    if comp_uuid in file.read():
                        name_file = filepath
                if name_file:
                    break
            except UnicodeDecodeError:
                pass
    return name_file


def _remove_from_parent_name_file(comp_uuid: str, name_file: Path | None) -> None:
    """Remove the line containing comp_uuid from name_file, which can be either a model name file or mfsim.nam.

    Args:
        comp_uuid: The component uuid string.
        name_file: Filepath to name file.
    """
    if not name_file:
        return

    with name_file.open('r') as fp:
        lines = fp.read().splitlines(True)

    line_to_remove = -1
    for i, line in enumerate(lines):
        if comp_uuid in line:
            line_to_remove = i
            break

    if line_to_remove >= 0:
        with name_file.open('w') as fp:
            fp.writelines(lines[0:line_to_remove])
            fp.writelines(lines[line_to_remove + 1:])


def _get_save_dialog_window_title() -> str:
    """Returns either 'Save' or 'Save' plus the process ID if debugging and file hack is found."""
    if xms_parent_dlg.can_add_process_id():
        window_title = xms_parent_dlg.process_id_window_title('Save')
    else:
        window_title = 'Save'
    return window_title


def _fix_based_name(name: str, with_the_a: bool) -> str:
    """Fix name containing 'list-based' or 'array-based', e.g. 'RCH list-based' -> 'RCH'.

    Packages that can have READASARRAYS will start out with names that include 'list-based' or 'array-based' but
    MODFLOW can't handle spaces in the name and we like to simplify the name to just RCH or RCHA etc.

    Args:
        name: The original name.
        with_the_a: True if you want it with the 'A', e.g. EVTA instead of EVT.

    Returns:
        The new name.
    """
    words = name.split()
    if len(words) < 2:
        return name
    new_name = words[0]
    if 'array-based' in name and with_the_a:
        new_name += 'A'
    return new_name


def _array_based_and_disu(tree_node: TreeNode, parent: TreeNode) -> bool:
    """Return True if tree_node is an array-based (READASARRAYS) file and model uses DISU.

    Args:
        tree_node: The components tree node.
        parent: The components parent tree node.

    Returns:
        See description.
    """
    if parent.unique_name in data_util.model_ftypes():
        dis_ftype = _find_dis_ftype(parent)
        if dis_ftype == 'DISU6' and tree_node.unique_name in data_util.readasarrays_ftypes(with_the_a=True):
            return True
    return False


def _find_dis_ftype(model_node: TreeNode) -> str:
    """Find and return the dis* node below the model node.

    Args:
        model_node: The model tree node.

    Returns:
        See description.
    """
    return tree_util.child_with_unique_name_in_list(model_node, ['DIS6', 'DISV6', 'DISU6']).unique_name


def _make_new_comp_dir(main_file: Pathlike) -> Path:
    """Creates a directory for a new component, in the same Components folder indicated by main_file.

    Args:
        main_file: An existing main file.

    Returns:
        Path to the new folder.
    """
    comp_dir = Path(main_file).parent.parent
    new_comp_uuid = tools.new_uuid()
    new_comp_dir = comp_dir / new_comp_uuid
    new_comp_dir.mkdir()
    return new_comp_dir


def _create_new_main_file(new_comp_dir: Path, main_file: Pathlike, ftype: str) -> Path:
    """Create an empty main file in new_comp_dir with stem the same as main_file but with suffix determined by ftype.

    Args:
        main_file: An existing name file, used to get the name for the new one.
        new_comp_dir: Directory where new main file will be created.
        ftype: ftype of the new main file.

    Returns:
        Path to the new main file.
    """
    extension = data_util.extension_from_ftype(ftype)
    name = f'{Path(main_file).stem}{extension}'
    new_main_file = new_comp_dir / name
    with open(new_main_file, 'w') as _:
        pass
    return new_main_file


def _create_delete_action(uuid_to_delete: str, comp: type[PackageComponentBase]) -> ActionRequest:
    """Return an ActionRequest to delete a component.

    Args:
        uuid_to_delete: Main file of the component that is to be deleted.
        comp: A component that will NOT be deleted but will receive the action request.

    Returns:
        ActionRequest.
    """
    delete_action = ActionRequest(
        modality='NO_DIALOG',
        main_file=comp.main_file,
        class_name=comp.class_name,
        module_name=comp.module_name,
        method_name='delete_component',
        parameters={'component_uuid': uuid_to_delete}
    )
    return delete_action


def _replace_with_list_based(
    old_main_file: Pathlike, old_unique_name: str, parent_uuid: str, klass: type[PackageComponentBase], new_name: str,
    query: Query
) -> ActionRv:
    """Replace this array-based component with a list-based equivalent.

    See notes in self._replace_with_list_based_equivalent().

    1. Create the new component folder and file
    2. Create the do_comp
    3. Create the action that will create the new component
    4. Create the action that will delete this component
    5. Use query to add and link the new component

    The delete action causes delete_component() to be called on the new component, which will delete this component.

    Args:
        old_main_file: Main file of component that is being replaced.
        old_unique_name: TreeNode.unique_name of the component that is being replaced.
        parent_uuid: Uuid of the parent tree node of the component that is being replaced.
        klass: Class of the new component to be created.
        new_name: Name of the new component / new tree node.
        query: Object for communicating with GMS.

    Returns:
        ActionRv
    """
    # Create the new list-based component directory next to the old component directory
    new_comp_dir = _make_new_comp_dir(old_main_file)

    # Create the new list-based main file in new dir with name stem same as the old main file
    list_based_ftype = data_util.fix_ftype(old_unique_name, with_the_a=False)
    new_main_file = _create_new_main_file(new_comp_dir, old_main_file, list_based_ftype)

    # Create the do_comp
    do_comp = create_data_objects_component(new_main_file, new_name, 'MODFLOW 6', list_based_ftype)

    # Create the action to add the component
    comp = klass(str(new_main_file))
    create_action = create_add_component_action(comp, 'get_initial_display_options')

    # Create an ActionRequest for the new component that will delete this component
    uuid_to_delete = io_util.uuid_from_path(old_main_file)
    delete_action = _create_delete_action(uuid_to_delete, comp)

    actions = [create_action, delete_action]
    add_and_link(query, do_comp, actions, parent_uuid)

    message = ('INFO', 'Array-based packages cannot be used with DISU. The list-based equivalent has been substituted.')
    return [message], []


def _unique_name_no_spaces(tree_node: TreeNode, with_the_a: bool) -> str:
    """Return a name for the tree_node that is unique among its siblings and which contains no spaces.

    If the name is like 'EVT list-based' or 'EVT array-based', the new name will be 'EVT' or 'EVTA'.

    Args:
        tree_node: The components tree node.
        with_the_a: True if you want it with the 'A', e.g. EVTA instead of EVT.
    """
    name = tree_node.name

    # First fix names that contain 'list-based' or 'array-based'
    if '-based' in tree_node.name:
        name = _fix_based_name(tree_node.name, with_the_a=with_the_a)

    # Make sure name is not the same as it's siblings and contains no spaces
    sibling_names = {child.name for child in tree_node.parent.children if child != tree_node}
    return gui_util.unique_name_no_spaces(sibling_names, name)


def _rename_node_if_necessary(tree_node: TreeNode, query: Query) -> None:
    """Rename tree_node if necessary to get rid of spaces and 'list-based' or 'array-based' text.

    Called from new_package().

    Args:
        tree_node: The components tree node.
        query: Object for communicating with GMS
    """
    new_name = _unique_name_no_spaces(tree_node, with_the_a=True)
    if new_name != tree_node.name:
        query.rename_item(tree_node.uuid, new_name)
