"""RchListData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class RchListData(ListPackageData):
    """Data class to hold the info from a RCH list-based package file."""
    def __init__(self, **kwargs):
        """Initializes the class."""
        super().__init__(**kwargs)
        self.ftype = 'RCH6'

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        rch_rate_units = units_util.string_from_units(self, units_util.UNITS_INFILTRATION)
        return {names.index('RECHARGE'): f'Recharge flux rate {rch_rate_units}'}

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        Args:
            block (str): Name of the block.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {'RECHARGE': (object, 0.0)}
        return gui_util.column_info_tuple_from_dict(columns)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Recharge (RCH) Package'

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {'Name': None, 'RECH': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        if feature_type != 'polygons':
            return {}
        info = {'RECH': None}
        self._add_aux_and_boundname_info(info)
        return info

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('FIXED_CELL', brief='Do not reassign to underlying cell if cell is inactive'),
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField('AUXMULTNAME', brief='Auxiliary variable used as multiplier of recharge', type_='str'),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )
