"""ImsReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.misc import log_util
from xms.mf6.misc.settings import Settings


def samg_setting(ims_main_file):
    """Returns True if SAMG is specified in the file.

    Args:
        ims_main_file (str): File path to IMS mainfile.

    Returns:
        See description.
    """
    reader = io_factory.reader_from_ftype('IMS6')
    ims_data = reader.read(ims_main_file)
    if ims_data:
        return ims_data.linear_solver_use and 'SAMG' in ims_data.linear_solver.upper()
    return False


class ImsReader(PackageReader):
    """Reads an IMS package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='IMS6')
        self._stop_after_block = 'LINEAR'
        self._log = log_util.get_logger()

    def _on_end_block(self, block_name):
        """Called when an END [block] line is found.

        Args:
            block_name (str): Name of the current block.
        """
        if block_name != 'OPTIONS':
            return

        if self._data.options_block.has('NO_PTC'):
            no_ptc = self._data.options_block.get('NO_PTC')
            if no_ptc is None or len(no_ptc) == 0:
                self._data.options_block.set('NO_PTC', True, 'ALL')  # Not specifying anything is the same as 'ALL'

    def _read_nonlinear(self, line):  # noqa: C901  (too complex)
        """Reads the nonlinear block.

        Args:
            line: A line from the file.
        """
        words = line.split()
        if not words or len(words) < 2:
            self.invalid_line_in_file(line)
            return

        if words[0].upper() in ['OUTER_HCLOSE', 'OUTER_DVCLOSE']:  # OUTER_HCLOSE is the old name
            self._data.outer_dvclose_use = True
            self._data.outer_dvclose = float(words[1])
        elif words[0].upper() == 'OUTER_MAXIMUM':
            self._data.outer_maximum_use = True
            self._data.outer_maximum = int(words[1])
        elif words[0].upper() == 'UNDER_RELAXATION':
            self._data.under_relaxation_use = True
            self._data.under_relaxation = words[1].upper()
        elif words[0].upper() == 'UNDER_RELAXATION_THETA':
            self._data.under_relaxation_theta_use = True
            self._data.under_relaxation_theta = float(words[1])
        elif words[0].upper() == 'UNDER_RELAXATION_KAPPA':
            self._data.under_relaxation_kappa_use = True
            self._data.under_relaxation_kappa = float(words[1])
        elif words[0].upper() == 'UNDER_RELAXATION_GAMMA':
            self._data.under_relaxation_gamma_use = True
            self._data.under_relaxation_gamma = float(words[1])
        elif words[0].upper() == 'UNDER_RELAXATION_MOMENTUM':
            self._data.under_relaxation_momentum_use = True
            self._data.under_relaxation_momentum = float(words[1])
        elif words[0].upper() == 'BACKTRACKING_NUMBER':
            self._data.backtracking_number_use = True
            self._data.backtracking_number = int(words[1])
        elif words[0].upper() == 'BACKTRACKING_TOLERANCE':
            self._data.backtracking_tolerance_use = True
            self._data.backtracking_tolerance = float(words[1])
        elif words[0].upper() == 'BACKTRACKING_REDUCTION_FACTOR':
            self._data.backtracking_reduction_factor_use = True
            self._data.backtracking_reduction_factor = float(words[1])
        elif words[0].upper() == 'BACKTRACKING_RESIDUAL_LIMIT':
            self._data.backtracking_residual_limit_use = True
            self._data.backtracking_residual_limit = float(words[1])
        elif words[0].upper() == 'LINEAR_SOLVER':  # SAMG extension to MODFLOW 6
            self._data.linear_solver_use = True
            self._data.linear_solver = words[1].upper()

    def _read_linear(self, line):  # noqa: C901  (too complex)
        """Reads the linear block.

        Args:
            line: A Line from the file.
        """
        words = line.split()
        if not words or len(words) < 2:
            self.invalid_line_in_file(line)
            return

        if words[0].upper() == 'INNER_MAXIMUM':
            self._data.inner_maximum_use = True
            self._data.inner_maximum = int(words[1])
        elif words[0].upper() in ['INNER_HCLOSE', 'INNER_DVCLOSE']:  # INNER_HCLOSE is the old name
            self._data.inner_dvclose_use = True
            self._data.inner_dvclose = float(words[1])
        elif words[0].upper() == 'INNER_RCLOSE':
            self._data.inner_rclose_use = True
            self._data.inner_rclose = float(words[1])
            if len(words) > 2:
                self._data.rclose_option_use = True
                self._data.rclose_option = words[2].upper()
        elif words[0].upper() == 'LINEAR_ACCELERATION':
            self._data.linear_acceleration_use = True
            self._data.linear_acceleration = words[1].upper()
        elif words[0].upper() == 'RELAXATION_FACTOR':
            self._data.relaxation_factor_use = True
            self._data.relaxation_factor = float(words[1])
        elif words[0].upper() == 'PRECONDITIONER_LEVELS':
            self._data.preconditioner_levels_use = True
            self._data.preconditioner_levels = int(words[1])
        elif words[0].upper() == 'PRECONDITIONER_DROP_TOLERANCE':
            self._data.preconditioner_drop_tolerance_use = True
            self._data.preconditioner_drop_tolerance = float(words[1])
        elif words[0].upper() == 'NUMBER_ORTHOGONALIZATIONS':
            self._data.number_orthogonalizations_use = True
            self._data.number_orthogonalizations = int(words[1])
        elif words[0].upper() == 'SCALING_METHOD':
            self._data.scaling_method_use = True
            self._data.scaling_method = words[1].upper()
        elif words[0].upper() == 'REORDERING_METHOD':
            self._data.reordering_method_use = True
            self._data.reordering_method = words[1].upper()

    @override
    def read_settings(self) -> None:
        """Reads the package settings file if it exists."""
        super().read_settings()
        if os.path.exists(Settings.settings_filename(self._data.filename)):
            settings = Settings.read_settings(self._data.filename)
            self._data.slnmnames = settings.get('slnmnames', [])
