"""Geometry functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import IntEnum
from pathlib import Path

# 2. Third party modules
import pandas as pd
import shapefile  # From pyshp

# 3. Aquaveo modules

# 4. Local modules


class ShapeTypeEnum(IntEnum):
    """Enumeration of DIS packages."""
    # Point = 1  # We should always be 3D
    # PolyLine = 3  # We should always be 3D
    # Polygon = 5  # We should always be 3D
    PointZ = 11
    PolyLineZ = 13
    PolygonZ = 15


def xyz_tuples_from_shape(shape, start=0, stop=None):
    """Returns a list of x,y,z coordinates as tuples.

    Args:
        shape (shapefile.Shape): A shape from the shapefile.
        start (int): Index of starting point.
        stop (int): Index of stopping point.

    Returns:
        (list(tuple)): See description.
    """
    points = []
    if not stop:
        stop = len(shape.points)
    for i in range(start, stop):
        points.append((shape.points[i][0], shape.points[i][1], shape.z[i]))
    return points


def geom_from_shape(shape):
    """Converts the shape to geometry text."""
    if shape.shapeType == ShapeTypeEnum.PointZ:
        return (shape.points[0][0], shape.points[0][1], shape.z[0])
    elif shape.shapeType == ShapeTypeEnum.PolyLineZ:
        return xyz_tuples_from_shape(shape)
    elif shape.shapeType == ShapeTypeEnum.PolygonZ:
        loops = []
        parts_with_end = [*shape.parts, len(shape.points)]
        for i in range(0, len(parts_with_end) - 1):
            loops.append(xyz_tuples_from_shape(shape, parts_with_end[i], parts_with_end[i + 1]))
        return loops
    else:
        raise RuntimeError(f'Shapefile shape type not supported: {shape.shapeTypeName}')  # pragma no cover


def shapefile_feature_type(shapefilepath: Path | str) -> str:
    """Opens shapefile and returns 'points', 'arcs', or 'polygons'.

    Args:
        shapefilepath: The shapefile filepath.

    Returns:
        See description.
    """
    try:
        with shapefile.Reader(shapefilepath) as reader:
            match reader.shapeType:
                case ShapeTypeEnum.PointZ:
                    return 'points'
                case ShapeTypeEnum.PolyLineZ:
                    return 'arcs'
                case ShapeTypeEnum.PolygonZ:
                    return 'polygons'
                case _:
                    raise ValueError(f'Shapefile shape type not supported: {reader.shapeType}')  # pragma no cover
    except Exception:
        raise ValueError(f'Could not read shapefile: {str(shapefilepath)}')


def shapefile_to_dataframe(shp_path: Path | str) -> pd.DataFrame:
    """Reads a shapefile and returns a pandas DataFrame.

    Args:
        shp_path: Path to the shapefile.

    Returns:
        pandas.DataFrame: DataFrame containing shapefile data.
    """
    sf = shapefile.Reader(shp_path)
    fields = [x[0] for x in sf.fields][1:]
    records = sf.records()
    shps = [s.points for s in sf.shapes()]
    df = pd.DataFrame(data=records, columns=fields)
    df['coords'] = shps
    sf.close()
    return df
