"""AdvancedTransportDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QPushButton
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.gui.advanced_package_dialog import AdvancedPackageDialog
from xms.mf6.gui.gwt.flow_package_data_finder import FlowPackageDataFinder
from xms.mf6.gui.options_gui import OptionsGui


class AdvancedTransportDialog(AdvancedPackageDialog):
    """A dialog used for the LKT, MWT, SFT, and UZT packages."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    @override
    def setup_ui(self) -> None:
        """Set up everything dealing with sections."""
        super().setup_ui()
        self._add_setup_from_flow_package_button()

    @override
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def change_columns(self, use_aux):
        """Changes a table's columns given the list of auxiliary variables.

        Args:
            use_aux (bool): True to include AUXILIARY variables.
        """
        table_widget = self.get_table_widget('PACKAGEDATA')
        if table_widget:
            table_widget.change_columns('PACKAGEDATA', use_aux=use_aux)

    def _add_setup_from_flow_package_button(self):
        """Creates the tabs for the list blocks."""
        page = self._find_packagedata_tab()

        # Add the button
        w = QPushButton(f'Set Up From {self._flow_package_ftype()} Package')
        page.layout().addWidget(w)
        w.clicked.connect(self.on_btn_set_up_from_flow_package)
        w.setEnabled(not self.dlg_input.locked)

    def _find_packagedata_tab(self):
        """Returns the PACKAGEDATA tab.

        Returns:
            (QWidget): See description
        """
        tab_widget = self.data_tabs_widget.uix['tab_widget']
        page = None
        for index in range(tab_widget.count()):
            if tab_widget.tabText(index) == 'PACKAGEDATA':
                page = tab_widget.widget(index)
                break
        return page

    def _flow_package_ftype(self):
        """Return the flow package corresponding to the transport package.

        Returns:
            (str): See description.
        """
        if self.dlg_input.data.ftype == 'LKT6':
            return 'LAK6'
        elif self.dlg_input.data.ftype == 'MFT6':
            return 'MFW6'
        elif self.dlg_input.data.ftype == 'SFT6':
            return 'SFR6'
        elif self.dlg_input.data.ftype == 'UZT6':
            return 'UZF6'
        # GWE
        elif self.dlg_input.data.ftype == 'LKE6':
            return 'LAK6'
        elif self.dlg_input.data.ftype == 'MWE6':
            return 'MAW6'
        elif self.dlg_input.data.ftype == 'SFE6':
            return 'SFR6'
        elif self.dlg_input.data.ftype == 'UZE6':
            return 'UZF6'

    def on_btn_set_up_from_flow_package(self):
        """Called with the 'Set Up From Flow Package' button is clicked."""
        finder = FlowPackageDataFinder(self._flow_package_ftype(), self.dlg_input, self)
        package_data = finder.find_data()
        if package_data is None:
            message_box.message_with_ok(parent=self, message=f'No {finder.flow_ftype} found.')
            return
        if not package_data:  # User canceled or didn't select anything
            return
        if 'STRT' not in package_data or not package_data['STRT']:
            message_box.message_with_ok(parent=self, message='No data found in package.')
            return
        self._update_model(package_data)

    def _update_model(self, package_data):
        """Updates the model for the table.

        Args:
            package_data (dict): Data for the table.
        """
        section = 'PACKAGEDATA'
        table_widget = self.data_tabs_widget.ui_tabs['PACKAGEDATA']['tbl']
        model = table_widget.get_model(section)
        column0 = list(package_data.keys())[0]
        model.data_frame[column0] = package_data[column0]
        model.data_frame['STRT'] = package_data['STRT']
        table_widget.set_model(model, section)
