"""ArraysToDatasetsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QCheckBox, QDialog, QDialogButtonBox, QLabel, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.gwf.array_package_data import ArrayPackageData
from xms.mf6.gui import gui_util


def run(data: BaseFileData, win_cont: QWidget) -> list[str]:
    """Run the dialog.

    Args:
        data: Package data.
        win_cont (QWidget): The window container.

    Returns:
        List of the selected arrays.
    """
    dialog = ArraysToDatasetsDialog(data, win_cont)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_arrays()
    return []


class ArraysToDatasetsDialog(XmsDlg):
    """Used to pick which arrays to create datasets from."""
    def __init__(self, data, parent=None):
        """Initializes the class.

        Args:
            data: Something derived from GriddataBase. The package data.
            parent: The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.arrays_to_datasets_dialog')

        self._data = data
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('ArraysToDatasetsDialog'))
        self.uix = {}

        self._setup()

    def _setup(self):
        """Sets up the dialog."""
        self.setWindowTitle('Arrays to Datasets')

        vertical_layout = QVBoxLayout(self)

        # Add label
        label = QLabel('Select arrays to create datasets from:')
        vertical_layout.addWidget(label)

        # Add checkboxes
        array_names = []
        if isinstance(self._data, GriddataBase):
            for block in self._data.blocks.values():
                for array in block.arrays:
                    if self._data.can_be_dataset(array.array_name):
                        array_names.append(array.array_name)
        elif isinstance(self._data, ArrayPackageData):
            array_names = self._data.tab_names()
        for name in array_names:
            self.uix[name] = QCheckBox(name)
            vertical_layout.addWidget(self.uix[name])

        # Add OK, Cancel, Help buttons
        self.btn_box = QDialogButtonBox()
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)
        vertical_layout.addWidget(self.btn_box)

    def get_selected_arrays(self) -> list[str]:
        """Returns a list of the array names that are checked.

        Returns:
            (list[str]): See description.
        """
        selected_arrays = []
        for key, item in self.uix.items():
            if item.isChecked():
                selected_arrays.append(key)
        return selected_arrays
