"""Testing utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import contextlib
from dataclasses import dataclass
import filecmp
import os
from pathlib import Path
import shutil
import sqlite3
import tempfile
from typing import Any
from unittest.mock import MagicMock
import uuid

# 2. Third party modules
from PySide2.QtCore import QPoint, Qt
from PySide2.QtTest import QTest
from PySide2.QtWidgets import QDialog, QDialogButtonBox, QWidget

# 3. Aquaveo modules
from xms.api.tree import tree_util, TreeNode
from xms.core.filesystem import filesystem as fs
from xms.data_objects.parameters import UGrid as DoGrid
from xms.guipy.dialogs.dialog_util import ensure_qapplication_exists
from xms.guipy.testing import testing_tools
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.testing import file_comparison, tools
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.components import mf6_importer
from xms.mf6.components.mf6_exporter import Mf6Exporter
from xms.mf6.data import data_util
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.file_io import io_factory, io_util, mfsim_reader
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.misc import util

SKIP_COMMENTS = ['\"LAST_IMPORT_FILE', '\"LAST_EXPORT_FILE']
"""Skip lines starting with these strings when comparing files in are_dir_trees_equal()."""


@dataclass
class OpenDlgInputs:
    """Inputs to open_close_dialog()."""
    dialog_class: Any  # The dialog class. Like SfrDialog.
    sim_dir: str | Path  # Full path to the simulation directory.
    ftype: str  # The file type used in the GWF name file (e.g. 'WEL6')
    model_filename: str = ''  # GWF/GWT file basename. Only use if there's multiple models.
    package_filename: str = ''  # Package file basename. Don't use if you want the model.
    parent_package_ftype: str | None = None  # 'RIV6' etc. Package that an obs file is mentioned in.
    test_child_dir: str | None = None  # test child directory.
    locked: bool = False  # if the dialog data is locked or not
    leave_open: bool = False  # If True, the dialog is not closed.
    tmp_path: Path | None = None  # pytest tmp_path fixture
    import_native: bool = False  # If True and files are native files, imports the simulation.


def find_mfsim_nam(test_dir: Path | str, mfsim_dir: Path | str) -> Path | None:
    """Finds the mfsim.nam file."""
    if mfsim_dir:
        mfsim_nam = Path(test_dir) / mfsim_dir / 'mfsim.nam'
    else:
        paths = list(Path(test_dir).rglob('mfsim.nam'))
        if not paths:
            return None
        mfsim_nam = paths[0]
    return mfsim_nam


def _copy_test_dir_if_asked(copy: bool, test_dir: Path | str, tmp_path: Path | None) -> Path | None:
    # Make a copy of the test directory, if desired
    new_test_dir = Path(test_dir)
    if copy:
        shutil.copytree(test_dir, tmp_path / test_dir.name, dirs_exist_ok=True)
        new_test_dir = tmp_path
    return new_test_dir


def _find_tree_file(test_dir: Path | str) -> Path | None:
    # Find tree file
    tree_dir = test_dir
    if Path(test_dir).name == 'Components':
        tree_dir = test_dir.parent.parent
    tree_paths = list(tree_dir.rglob('project_tree.json'))
    if tree_paths:
        return tree_paths[0]
    return None


def _read_tree(test_dir: Path | str) -> tuple[Path | None, TreeNode | None]:
    tree_file = _find_tree_file(test_dir)
    if tree_file:
        # return tree_file, testing_tools.read_tree_from_file(tree_file)
        return tree_file, read_and_fix_tree(tree_file)
    return None, None


def read_write_compare(in_dir: Path, base_dir: Path, tmp_path: Path) -> None:
    """Read, write, and compare to baseline.

    Handles .gpr for the one use case I had when I wrote it.

    Args:
        in_dir: If a GMS project file, the folder containing the .gpr, project_tree.json, and ugrid.xmc files.
        base_dir: Baseline directory.
        tmp_path: pytest tmp_path fixture.
    """
    # Find all sims
    paths = list(in_dir.rglob('mfsim.nam'))
    if not paths:
        return

    # Read the sims
    _tree_file, project_tree = _read_tree(in_dir)
    sim_nodes = tree_util.descendants_of_type(tree_root=project_tree, xms_types=['TI_DYN_SIM'])
    with tools.env_running_tests('TRUE'), tools.repeat_uuids(), tools.env_temp_dir(tmp_path):

        # TODO: Handle multiple simulations

        # Create mock query. It just needs to handle query.item_with_uuid() to return the DoGrid object.
        mock_query = MagicMock(project_tree=project_tree)
        ugrid_file = in_dir / 'ugrid.xmc'
        mock_query.item_with_uuid.side_effect = [DoGrid(str(ugrid_file))]

        # Make a directory where the sim will be exported
        out_dir = tmp_path / 'out'
        os.makedirs(out_dir)

        # Export the sim (which reads, then writes)
        exporter = Mf6Exporter(query=mock_query, mfsim_nam=paths[0], sim_node=sim_nodes[0], export_dir=out_dir)
        exporter.export()

        # Compare
        assert file_comparison.are_dir_trees_equal(base_dir, out_dir, skip_extensions=['.keep'])


def read_test_sims(test_dir: Path | str, mfsim_dir: str, copy: bool, tmp_path: Path | None = None) -> list[MfsimData]:
    """Read and return all simulations below test_dir.

    Args:
        test_dir: Path to test directory (typically the 'Components' directory).
        mfsim_dir: Name of directory mfsim.nam is in. Only needed when there are multiple sims.
        copy: If True, test_dir is first copied to tmp_path so any changes won't affect the original files.
        tmp_path: Pytest tmp_path where test_dir will be copied to if copy is True.

    Returns:
        List of simulations.
    """
    test_dir = _copy_test_dir_if_asked(copy, test_dir, tmp_path)

    # Find all sims
    paths = list(test_dir.rglob('mfsim.nam'))
    if not paths:
        return []

    tree_file, project_tree = _read_tree(test_dir)

    # Read the sims
    sims = []
    sim_nodes = tree_util.descendants_of_type(tree_root=project_tree, xms_types=['TI_DYN_SIM'])
    with tools.env_running_tests('TRUE'), tools.repeat_uuids(), tools.env_temp_dir(tmp_path):
        for mfsim_nam in paths:
            if mfsim_dir and mfsim_nam.parent.name != mfsim_dir:
                continue

            if not _in_component_form(mfsim_nam):  # Model native read
                mfsim = mf6_importer.import_sim(mfsim_nam, query=None)
                apply_tree(project_tree, mfsim)
            else:  # Read from Components folders
                # Check for a project_tree.json next to the project. If it exists, read the simulation from the tree
                # instead of just from the component files.
                sim_node = _sim_node_from_tree_and_mfsim_nam(sim_nodes, mfsim_nam)
                mfsim = MfsimData.from_file(mfsim_nam, 'MFSIM6', sim_node=sim_node)
            sims.append(mfsim)
    return sims


def _sim_node_from_tree_and_mfsim_nam(sim_nodes: list[TreeNode], mfsim_nam: Path | str) -> TreeNode | None:
    """Return the sim node in the project_tree matching the mfsim.nam file."""
    for sim_node in sim_nodes:
        model_filenames, mnames, ftypes = mfsim_reader.model_name_files_from_mfsim_nam_file(mfsim_nam)
        model_children = tree_util.descendants_of_type(sim_node, xms_types=['TI_COMPONENT'], unique_name='GWF6')
        if len(model_filenames) != len(model_children):
            continue
        for i in range(len(model_filenames)):
            if Path(model_filenames[i]).name != Path(model_children[i].main_file).name:
                continue
        return sim_node
    return None


def column_as_list(qt_table, column):
    """Helper function to return the table column as a list.

    Args:
        qt_table (:obj:'PySide2.QtWidgets.QTableView'): QTableView or QTableWidget
        column (int): The column

    Returns:
        (list): The column as a list of strings
    """
    my_list = []
    for row in range(qt_table.rowCount()):
        my_list.append(get_table_data(qt_table, row, column))
    return my_list


def table_row(qt_table, row):
    """Helper function to return the table column as a list.

    Args:
        qt_table (:obj:'PySide2.QtWidgets.QTableView'): QTableView or QTableWidget
        row (int): The row

    Returns:
        (list): The column as a list of strings
    """
    my_list = []
    for column in range(qt_table.model().columnCount()):
        my_list.append(get_table_data(qt_table, row, column))
    return my_list


def get_table_data(qt_table, row, column):
    """Returns the item in the table at row and column.

    Args:
        qt_table (:obj:'PySide2.QtWidgets.QTableView'): QTableView or QTableWidget
        row (int): The row in the model.
        column (int): The column in the model.

    Returns:
        The data.
    """
    index = qt_table.model().index(row, column)
    return qt_table.model().data(index)


def _setup_table_click(qt_table: QxTableView, row: int, column: int) -> tuple[QWidget, int, int]:
    """Setup for clicking in the table.

    Args:
        qt_table: A QTableView or QTableWidget
        row: The row
        column: The column

    Returns:
        Viewport, x_pos, y_pos
    """
    x_buffer = 5  # Horizontal buffer to ensure click is inside a cell
    y_buffer = 10  # Vertical buffer to ensure click is inside a cell
    x_pos = qt_table.columnViewportPosition(column) + x_buffer
    y_pos = qt_table.rowViewportPosition(row) + y_buffer
    viewport = qt_table.viewport()
    return viewport, x_pos, y_pos


def left_click_in_table(qt_table, row, column):
    """Left click in the table at the cell defined by row and column.

    From https://stackoverflow.com/questions/12604739

    Args:
        qt_table: A QTableView or QTableWidget
        row: The row
        column: The column
    """
    viewport, x_pos, y_pos = _setup_table_click(qt_table, row, column)
    QTest.mouseClick(viewport, Qt.LeftButton, pos=QPoint(x_pos, y_pos))


def right_click_in_table(qt_table: QxTableView, row: int, column: int):
    """Right click in the table at the cell defined by row and column.

    From https://stackoverflow.com/questions/12604739

    Args:
        qt_table: A QTableView or QTableWidget
        row: The row
        column: The column
    """
    viewport, x_pos, y_pos = _setup_table_click(qt_table, row, column)
    QTest.mouseClick(viewport, Qt.RightButton, pos=QPoint(x_pos, y_pos))


def get_column_names(qt_table):
    """Returns the column headings as a list of strings.

    Args:
        qt_table: A QTableView or QTableWidget

    Returns:
        List of strings
    """
    column_names = []
    if not qt_table.model():
        return 0
    column_count = qt_table.model().columnCount()
    for h in range(column_count):
        column_names.append(qt_table.model().headerData(h, Qt.Horizontal))
    return column_names


def setup_paths(files_path, folder, subfolder, file):
    """Sets up the input, output, and base paths.

     Creates output directory if needed, or clears it if it exists.

    Args:
        files_path (str): Path to 'files' folder.
        folder (str): Folder name inside 'files' folder. E.g. 'io_tests/usgs_examples_dmi_base'
        subfolder (str): Subfolder name inside 'folder'. E.g. 'ex01-twri'
        file (str): File name inside 'subfolder'. E.g. 'twri.ims'

    Returns:
        (tuple): tuple containing:
            filein(str): Full path to input file.
            fileout(str): Full path to output file.
            baseline(str): Full path to baseline file.
    """
    file = os.path.basename(file)
    filein = os.path.join(files_path, folder, subfolder, file)
    baseline = os.path.join(files_path, folder + '_base', subfolder, file)
    fileout = os.path.join(files_path, folder + '_out', subfolder, file)
    output_folder = os.path.dirname(fileout)
    if os.path.isdir(output_folder):
        fs.clear_folder(output_folder)
    else:
        os.makedirs(output_folder, exist_ok=True)
    return filein, fileout, baseline


def _find_file(folder, extension):
    """Finds a file with the given extension by searching recursively under folder.

    Args:
        folder (str): Folder.
        extension (str): Extension (e.g. '.zon')

    Returns:
        (str) Path to file.
    """
    for dirpath, _, filenames in os.walk(folder):
        for filename in filenames:
            if filename.endswith(extension):
                return os.path.join(dirpath, filename)
    return ''


def clear_folder(folder, except_extensions):
    """Finds a file with the given extension by searching recursively under folder.

    Args:
        folder (str): Folder.
        except_extensions (set[str]): Extensions of files to not delete.

    Returns:
        (str) Path to file.
    """
    for the_file in os.listdir(folder):
        filepath = os.path.join(folder, the_file)
        try:
            if os.path.isfile(filepath) and os.path.splitext(filepath)[1] not in except_extensions:
                os.unlink(filepath)
            elif os.path.isdir(filepath):
                clear_folder(filepath, except_extensions)
        except Exception as e:
            print(e)


def _in_component_form(mfsim_nam: Path | str) -> bool:
    """Returns True if the sim is saved in separate uuid named folders in a 'Components' folder."""
    mfsim_dir = Path(mfsim_nam).parent
    return util.is_valid_uuid(str(mfsim_dir.name)) and mfsim_dir.parent.name == 'Components'


def get_packages(inputs: OpenDlgInputs, sim_dir: str):
    """Finds and returns the packages based on the inputs.

    Reads mfsim.nam and the model name file.

    Note that this should only be used for tests because in production, the mfsim.nam file
     may be out of date and the project explorer tree should be used instead.

    Args:
        inputs: All that's needed to open the dialog.
        sim_dir (str): Path to directory where the simulation is located.

    Returns:
        (tuple): tuple containing:
            - mfsim(MfsimData): MfsimData object.
            - packages(list[BaseFileData]): The packages
    """
    mfsim_nam = find_mfsim_nam(sim_dir, '')
    reader = io_factory.reader_from_ftype('MFSIM6')
    mfsim = reader.read(mfsim_nam)

    # Import native files, if necessary
    if inputs.import_native and not _in_component_form(mfsim.filename):
        mfsim = mf6_importer.import_sim(mfsim.filename, query=None)

    # Create some shorter variable names for readability
    ftype = inputs.ftype
    parent_package_ftype = inputs.parent_package_ftype
    package_filename = inputs.package_filename
    model_filename = inputs.model_filename

    packages = []
    if ftype == 'MFSIM6':
        packages.append(mfsim)
    elif ftype in data_util.model_ftypes():
        if inputs.model_filename:
            packages.append(mfsim.model_from_filename(inputs.model_filename))
        else:
            packages.extend(mfsim.models)
    elif ftype in {'IMS6', 'EMS6'}:
        for solution_group in mfsim.solution_groups:
            packages.extend(solution_group.solution_list)
    elif ftype == 'TDIS6':
        packages.append(mfsim.tdis)
    elif ftype in data_util.exchange_ftypes():
        for exchange in mfsim.exchanges:
            if exchange.exgtype.upper() == ftype:
                packages.append(exchange)
    elif parent_package_ftype in data_util.exchange_ftypes():
        parent_packages = mfsim.packages_from_ftype(parent_package_ftype)
        if parent_packages and len(parent_packages) > 0:
            filename = os.path.join(os.path.dirname(parent_packages[0].filename), package_filename)
            reader = io_factory.reader_from_ftype(ftype)
            packages.append(reader.read(filename, mfsim=mfsim, model=None, parent_package_ftype=parent_package_ftype))
    else:
        for model in mfsim.models:
            if model_filename and os.path.basename(model.filename) != model_filename:
                continue

            # Read zone and pest obs data separately because they aren't included in mfsim.nam or model name file
            if ftype == 'ZONE6' or ftype == 'POBS6':
                extension = data_util.extension_from_ftype(ftype)
                file_name = _find_file(os.path.abspath(os.path.join(mfsim_nam, '../..')), extension)
                reader = io_factory.reader_from_ftype('POBS6')
                data = reader.read(file_name, mfsim=mfsim, model=model)
                model.packages.append(data)
                packages.append(data)
            else:
                for package in model.packages:
                    if parent_package_ftype and package.ftype == parent_package_ftype:
                        reader = io_factory.reader_from_ftype(ftype)
                        filename = os.path.join(os.path.dirname(package.filename), package_filename)
                        data = reader.read(filename, mfsim=mfsim, model=model)
                        packages.append(data)
                        if data and hasattr(data, 'parent_package_ftype'):
                            data.parent_package_ftype = parent_package_ftype
                    elif package_filename and os.path.basename(package.filename) != package_filename:
                        continue
                    elif package.ftype == ftype:
                        packages.append(package)
    return mfsim, packages


def run_open_close_dialog_import(test_dir, ftype, dialog_class, running_multiple_tests, tmp_path: Pathlike = None):
    """Call open_close_dialog() with model native files.

    Args:
        test_dir (str): Path to the directory containing the mfsim.nam
        ftype (str): The package ftype
        dialog_class (class): The package's dialog class
        running_multiple_tests (bool): If False, leaves the dialog open. The pytest fixture.
        tmp_path: Optional path to temp folder.
    """
    with tools.repeat_uuids():
        leave_open = not running_multiple_tests
        inputs = OpenDlgInputs(
            dialog_class, test_dir, ftype, import_native=True, leave_open=leave_open, tmp_path=tmp_path
        )
        rv = open_close_dialog(inputs)
    assert rv is True


def open_close_dialog(inputs: OpenDlgInputs):
    """Runs the test.

    Args:
        inputs: All that's needed to open the dialog.

    Returns:
        The message returned by test_util.are_dir_trees_equal().
    """
    inputs.sim_dir = str(inputs.sim_dir)
    ensure_qapplication_exists()
    try:
        dialog, temp_dir = set_up_dialog(inputs)
        rv = QDialog.Rejected
        if inputs.leave_open:
            with tools.env_running_tests('MANUAL'):
                rv = dialog.exec()
        else:
            dialog.show()
            QTest.mouseClick(dialog.ui.buttonBox.button(QDialogButtonBox.Ok), Qt.LeftButton)

        if inputs.locked or (inputs.leave_open and rv == QDialog.Rejected):
            return True
        return write_and_compare(dialog, inputs, temp_dir)
    except Exception as error:
        print(error)


def set_up_dialog(inputs: OpenDlgInputs):
    """Set up the dialog.

    Args:
        inputs: All that's needed to open the dialog.

    Returns:
        (tuple): The dialog object and the temp dir.
    """
    sim_dir = str(inputs.sim_dir)
    temp_dir = os.path.join(tempfile.gettempdir(), str(uuid.uuid4())) if not inputs.tmp_path else str(inputs.tmp_path)
    with tools.env_temp_dir(temp_dir):
        # Copy the simulation to a temporary directory
        shutil.copytree(sim_dir, temp_dir, dirs_exist_ok=True)
        mfsim, packages = get_packages(inputs, temp_dir)
        data = packages[0] if packages else None

        # Create dialog
        dlg_input = DialogInput(data=data, locked=inputs.locked)
        dialog = inputs.dialog_class(dlg_input=dlg_input, parent=None)
    return dialog, temp_dir


def write_and_compare(dialog, inputs: OpenDlgInputs, temp_dir) -> bool:
    """Return True if, after writing the data, the files match the baseline.

    Args:
        dialog: The dialog object.
        inputs: All that's needed to open the dialog.
        temp_dir: Temporary directory.

    Returns:
        See description.
    """
    data = dialog.dlg_input.data
    mfsim_nam = dialog.dlg_input.data.mfsim.filename
    filename = data.filename
    base_dir, out_dir = get_base_and_out_dirs(inputs.dialog_class, filename, inputs.sim_dir, inputs.test_child_dir)

    # Write the package file
    writer_options = WriterOptions(mfsim_dir=os.path.dirname(mfsim_nam), use_open_close=True, dmi_sim_dir=temp_dir)
    writer = io_factory.writer_from_ftype(data.ftype, writer_options)
    if not writer and inputs.ftype == 'GWF6':
        writer = io_factory.writer_from_ftype('GWF6', writer_options)
    writer.write(data)

    # Copy to output dir
    shutil.rmtree(path=out_dir, ignore_errors=True)
    from_dir = os.path.dirname(filename)
    shutil.copytree(from_dir, out_dir)

    # Compare to a baseline and return the result
    return file_comparison.are_dir_trees_equal(base_dir, out_dir, comments=SKIP_COMMENTS)


def get_base_and_out_dirs(dialog_class, filename, sim_dir, test_child_dir):
    """Returns the output and baseline directories (files/gui/PackageDialog/out/ex01.../twri.rch/).

    Args:
        dialog_class (): The dialog class. Like SfrDialog.
        filename:
        sim_dir (str): Full path to the simulation directory.
        test_child_dir:

    Returns:
        (tuple(str, str)): base_dir, out_dir
    """
    test_files = util.get_test_files_path()
    dialog_dir = dialog_class.__name__
    if test_child_dir:
        sim_dir_base = test_child_dir
    else:
        sim_dir_base = os.path.basename(sim_dir)
    out_dir = os.path.join(test_files, 'gui', 'out', dialog_dir, sim_dir_base, os.path.basename(filename))
    base_dir = os.path.join(test_files, 'gui', 'base', dialog_dir, sim_dir_base, os.path.basename(filename))
    return base_dir, out_dir


def delete_temp_files(temp_file_list):
    """Deletes a list of files.

    Args:
        temp_file_list (list(str)): List of files.
    """
    for file in temp_file_list:
        fs.removefile(file)


def get_list_of_simulations(examples_dir, pattern, exclusions):
    """Returns a list of parent_folder, sim_folder tuples by searching examples_dir.

    Args:
        examples_dir (str): The folder below tests/files, e.g. 'io_tests/usgs_examples_2'
        pattern (str): File pattern to look for, like 'mfsim.nam' or '*.adv'
        exclusions (list of str): Tests to exclude, e.g. ['test001e_UZF_spring']

    Returns:
        See description.
    """
    # List of parent_dir and child_dir simulations to test.
    test_files = util.get_test_files_path()
    examples_path = os.path.join(test_files, examples_dir)
    sim_list = []
    for path in Path(examples_path).rglob(pattern):
        rel_dir = os.path.dirname(str(path.relative_to(examples_path)))
        if not exclusions or rel_dir not in exclusions:
            sim_list.append((examples_dir, os.path.dirname(str(path.relative_to(examples_path)))))
    return sim_list


def assert_array_initialized(array_layer: ArrayLayer, numeric_type: str, constant: float, factor: float):
    """Asserts that an array is initialized correctly.

    Args:
        array_layer: The ArrayLayer object.
        numeric_type: ArrayLayer.numeric_type - 'float' or 'int'
        constant: ArrayLayer.constant - constant value for the array.
        factor: ArrayLayer.factor - multiplication factor.
    """
    assert 'CONSTANT' == array_layer.storage
    assert numeric_type == array_layer.numeric_type
    assert constant == array_layer.constant
    assert factor == array_layer.factor


def assert_filecmp(base: str | Path, out: str | Path, do_assert: bool = True) -> bool:
    """Compare two files. If different, copy out to be next to base, or if update_baselines() is True, overwrite base.

    Args:
        base: Path to the baseline file.
        out: Path to the output file.
        do_assert: If True, calls assert on result of filecmp.cmp().

    Returns:
        (bool): True if equal, else False.
    """
    rv = filecmp.cmp(base, out, shallow=False)
    if not rv:
        if update_baselines():
            shutil.copy(out, base)
        elif Path(base).parent.resolve() != Path(out).parent.resolve():
            # Copy out to be next to base with _out appended to stem
            out_name = f'{Path(out).stem}_out{Path(out).suffix}'
            shutil.copy(out, Path(base).parent / out_name)
    if do_assert:
        assert rv is True
    return rv is True


def update_baselines() -> bool:
    """Set to True if you want to update the baseline files when the output files are different."""
    return False


def dump_database(db_filepath: str | Path, filepath: Path | str) -> Path:
    """Dumps the package database to a text file.

    Args:
        db_filepath: The database file path.
        filepath: Path to the text file. If '', will be set to db_filepath with '.txt' suffix.

    Returns:
        Filepath to text file.
    """
    with sqlite3.connect(db_filepath) as cxn:
        if not filepath:
            filepath = Path(db_filepath).with_suffix('.txt')
        with open(filepath, 'w') as file:
            for line in cxn.iterdump():
                file.write('%s\n' % line)
    return Path(filepath)


def database_dumps_equal(base_filepath: Path | str, db_filepath: Path | str) -> bool:
    """Returns True if the text dump of database at db_path equals base_txt_path.

    Args:
        base_filepath: Baseline text file from a previous dump.
        db_filepath: Path to sqlite database file.

    Returns:
        See description.
    """
    db_filepath = Path(db_filepath)
    dump_database(db_filepath, db_filepath.with_suffix('.txt'))
    return file_comparison.ascii_files_equal(base_filepath, db_filepath.with_suffix('.txt'))


def _replace_first_part_of_path(old_path: Path | str, new_start: Path, index: int) -> Path:
    """Replaces the first part of a path up to and including index with new_start.

    path: C:/a/b/c/d/e/file.txt
    new_start: Z:/1/2
    index: 3
    result = Z:/1/2/d/e/file.txt

    Args:
        old_path: Original path.
        new_start: New path to replace the beginning of path.
        index: Index in the path (starting from the left) to the directory that will be cut off.
    """
    return new_start.joinpath(*Path(old_path).parts[index:])


def read_and_fix_tree(tree_file: Path | str) -> TreeNode | None:
    """Reads the tree file and returns the tree with main_file paths fixed if needed to start with test_files path.

    Args:
        tree_file: Filepath to project_tree.json file.

    Returns:
        See description.
    """
    tree = testing_tools.read_tree_from_file(tree_file)
    if not tree:
        return None
    _fix_tree_paths(tree_file, tree)
    return tree


def _fix_tree_paths(tree_file: Path | str, tree: TreeNode) -> None:
    """Fixes the paths in the tree so that they are correct with respect to location of tree_file.

    Needed so paths work on the CI as well as locally.

    Args:
        tree_file: Filepath to project_tree.json file.
        tree: The project tree.
    """
    gpr_dir = tree_file.parent

    def action(node):
        if node.main_file:
            index = Path(node.main_file).parts.index('Components')
            left_dir = gpr_dir
            if node.unique_name == 'Sim_Manager':  # The sim component will always be unlocked in temp.
                # Hope there is one and only one file/folder ending in '_data' in the gpr_dir. Note that we will
                # only get the correct sim mainfile path if the components are structured as a GMS project. Figure
                # something else out if this is not the case.
                paths = list(gpr_dir.glob('*_data'))
                if paths:
                    left_dir = list(gpr_dir.glob('*_data'))[0]
            else:
                index -= 1
            node.main_file = str(_replace_first_part_of_path(node.main_file, left_dir, index))

    tree_util.traverse_tree(tree, action)


def _find_sim_node(project_tree: TreeNode, mfsim: MfsimData) -> TreeNode:
    """Find and return the sim node in the tree the matches the mfsim data.

    Args:
        project_tree: Project Explorer tree.
        mfsim: The sim.

    Returns:
        See description.
    """
    xms_types = ['TI_DYN_SIM']
    sim_nodes = tree_util.descendants_of_type(tree_root=project_tree, xms_types=xms_types)
    sim_node = None
    if len(sim_nodes) == 1:
        sim_node = sim_nodes[0]
    else:
        for node in sim_nodes:
            # Find the matching sim by comparing the models
            model_nodes = tree_util.descendants_of_type(tree_root=node, unique_name='GWF6')
            model_nodes.extend(tree_util.descendants_of_type(tree_root=node, unique_name='GWT6'))

            # Number of models must match
            if len(model_nodes) != len(mfsim.models):
                continue

            # Names of models must match
            for i in range(len(model_nodes)):
                model_node = model_nodes[i]
                model = mfsim.models[i]
                if model_node.name != model.pname:
                    break
            else:
                sim_node = node
                break
    return sim_node


def apply_tree(project_tree: TreeNode, mfsim: MfsimData) -> None:
    """When testing, populates package.tree_node variables with actual TreeNode objects.

    Tree file must match simulation.

    Args:
        project_tree: Project Explorer tree.
        mfsim: The simulation.
    """
    if not project_tree:
        return
    try:
        sim_node = _find_sim_node(project_tree, mfsim)
        _set_package_tree_node(mfsim, sim_node)

        solution_count = 0
        exchange_count = 0
        model_count = 0
        for child in sim_node.children:
            if child.unique_name == 'TDIS6':
                _set_package_tree_node(mfsim.tdis, child)
            elif child.unique_name in {'IMS6', 'EMS6'}:
                _set_package_tree_node(mfsim.solution_groups[0].solution_list[solution_count], child)
                solution_count += 1
            elif child.unique_name in data_util.exchange_ftypes():
                _set_package_tree_node(mfsim.exchanges[exchange_count], child)
                exchange_count += 1
            elif child.unique_name in data_util.model_ftypes():
                model = mfsim.models[model_count]
                _set_package_tree_node(model, child)
                model_count += 1

                # Do model packages
                package_count = 0
                for model_child in child.children:
                    if model_child.item_typename != 'TI_UGRID_PTR' and model_child.unique_name != 'POBS6':
                        _set_package_tree_node(model.packages[package_count], model_child)
                        package_count += 1
    except Exception:
        raise ValueError('Could not apply tree file to simulation')


def _set_package_tree_node(data, child: TreeNode):
    """Helper function."""
    if data.ftype != 'MFSIM6':  # Only the hidden sim will have main_file
        # Set node main_file and uuid to match package
        child.main_file = data.filename
        uuid_str = io_util.uuid_from_path(data.filename)
        if uuid_str:
            child.uuid = uuid_str
    data.tree_node = child


@contextlib.contextmanager
def file_context(filepath: Path | str, mode: str = 'w'):
    """Context manager for a file that will automatically get deleted.

    Parent directories must exist (we don't want to go deleting directory trees).

    Args:
        filepath: The filepath.
        mode: The mode in which to open the file.

    Returns:
        (TextIO) The file object.
    """
    filepath = Path(filepath)
    fp = open(filepath, mode=mode)
    yield fp
    fp.close()
    Path(filepath).unlink()


def test_get_app_icon():
    """Test get_app_icon()."""
    ensure_qapplication_exists()
    icon1 = util.get_app_icon()
    icon2 = util.get_app_icon()
    assert icon1 == icon2
    assert icon1 is icon2  # get_app_icon() stores the icon and returns the same one
