"""Module for ExportSimThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = [
    'ExportSimThread', 'LINKED_MESH_FILE_NAME', 'LINKED_BC_FILE_NAME', 'LINKED_NEIGHBORS_FILE_NAME',
    'LINKED_HYDRO_FILE_NAME', 'LINKED_SEDIMENTS_FILE_NAME'
]

# 1. Standard Python modules
from contextlib import suppress
from datetime import datetime, timedelta
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.dmi.xms_data import XmsData
from xms.constraint import GridType as CoGridType
from xms.datasets.dataset_reader import DatasetReader
from xms.gmi.data_bases.sim_base_data import SimBaseData
from xms.grid.ugrid import UGrid
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import ExitError, FeedbackThread
from xms.guipy.time_format import datetime_to_string
from xms.ptmio.pcf.control_writer import write_control
from xms.ptmio.pcf.program_control import FlowFormat, MeshFormat, ProgramControl, SedimentFormat

# 4. Local modules
from xms.ptm.components.sources_component import PtmSourcesComponent
from xms.ptm.components.traps_component import PtmTrapsComponent
from xms.ptm.file_io.control import (
    LINKED_BC_FILE_NAME, LINKED_HYDRO_FILE_NAME, LINKED_MESH_FILE_NAME, LINKED_NEIGHBORS_FILE_NAME,
    LINKED_SEDIMENTS_FILE_NAME, to_control
)
from xms.ptm.file_io.datasets import write_datasets
from xms.ptm.file_io.grid import write_grid
from xms.ptm.file_io.sources.source_writer import write_sources
from xms.ptm.file_io.traps.trap_writer import write_traps
from xms.ptm.model.model import HydroDefinition, MeshDefinition, SedimentDefinition, simulation_model


class ExportSimThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading PTM Control File',
            'working_prompt': 'Reading control file. Please wait...',
        }
        self._data = data
        self._ok = True

        # Control and related stuff
        self._control: Optional[ProgramControl] = None
        self._stop_time: datetime = datetime(year=1, month=1, day=1)
        self._mesh_is_linked: bool = True
        self._hydro_is_linked: bool = True
        self._sediment_is_linked: bool = True
        self._linked_elevation_dataset: str = ''
        self._linked_flow_dataset: str = ''
        self._linked_d35_dataset: str = ''
        self._linked_d50_dataset: str = ''
        self._linked_d90_dataset: str = ''

    def _run(self):
        """Run the thread."""
        self._log.info('Exporting simulation...')

        self._get_model_control()

        self._check_grid()
        self._check_sources()
        self._check_hydrodynamic_datasets()
        self._check_sediments()
        self._check_neighbors_and_bc()

        if not self._ok:
            raise ExitError()

        self._write_grid()
        self._write_sources()
        self._write_traps()
        self._write_hydrodynamics()
        self._write_sediments()
        self._write_neighbors_and_bc()
        self._write_control()  # Must be last since other steps might alter file paths.

    def _get_model_control(self):
        """Initialize self._control and some related flags."""
        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        self._control = to_control(section)

        if self._control.stop_run is not None:
            self._stop_time = self._control.stop_run
        else:
            start = self._control.start_run
            seconds = float(self._control.duration)
            duration = timedelta(seconds=seconds)
            self._stop_time = start + duration

        mesh = section.group('mesh')
        self._mesh_is_linked = mesh.parameter('mesh_type').value == MeshDefinition.linked
        self._hydro_is_linked = mesh.parameter('hydro_type').value == HydroDefinition.linked
        self._sediment_is_linked = mesh.parameter('sediment_type').value == SedimentDefinition.linked
        self._linked_elevation_dataset = mesh.parameter('linked_elevation_dataset').value
        self._linked_flow_dataset = mesh.parameter('linked_flow_dataset').value
        self._linked_d35_dataset = mesh.parameter('d35_dataset').value
        self._linked_d50_dataset = mesh.parameter('d50_dataset').value
        self._linked_d90_dataset = mesh.parameter('d90_dataset').value

    def _check_grid(self):
        """Check that a linked or external grid is valid."""
        if self._mesh_is_linked:
            self._check_linked_grid()
        else:
            self._check_external_grid()

    def _check_linked_grid(self):
        """Check that a linked grid is valid."""
        grid = self._data.linked_grid
        if not grid:
            self._ok = False
            self._log.error('Simulation is set to use linked grid, but no mesh or grid was linked to the simulation.')
            return

        if grid.grid_type == CoGridType.quadtree_2d or grid.grid_type == CoGridType.rectilinear_2d:
            self._ok = False
            self._log.error(
                'The PTM interface only supports triangle cells. Quadtrees and rectilinear grids can be triangulated '
                'by using the "2D Mesh from 2D UGrid" tool on the grid\'s cell centers. After triangulation, existing '
                'datasets can be interpolated to the new mesh from the right-click menu on the original grid.'
            )
            return

        if not grid.check_all_cells_are_of_type(UGrid.cell_type_enum.TRIANGLE):
            self._ok = False
            self._log.error(
                'The PTM interface only supports triangle cells. Non-triangle cells must be converted to '
                'triangles before exporting.'
            )

    def _check_external_grid(self):
        """Check that an external grid is valid."""
        external_file = self._control.mesh_file

        if not external_file:
            self._ok = False
            self._log.error('Simulation is set to use external grid, but no external grid file was set.')
            return

        if not Path(external_file).exists():
            self._ok = False
            self._log.error('Simulation is set to use external grid, but external grid file does not exist.')
            return

    def _write_grid(self):
        """Write the grid file."""
        if not self._mesh_is_linked:
            return

        self._control.mesh_file = LINKED_MESH_FILE_NAME
        write_grid(self._data.linked_grid, LINKED_MESH_FILE_NAME)

    def _check_neighbors_and_bc(self):
        """Check that the .neighbors and .bc file parameters are valid."""
        if self._mesh_is_linked:
            return

        neighbor_file = self._control.neighbor_file
        if not neighbor_file:
            self._ok = False
            self._log.error('External .bc and .neighbor files are in use, but .neighbors file path was not set.')
        if not Path(neighbor_file).exists():
            message = (
                'External .bc and .neighbor files are in use, but .neighbors file path refers to nonexistent file. '
                'File will be generated when simulation is run.'
            )
            self._log.warning(message)

        if self._control.mesh_format == MeshFormat.cms_2d:
            need_bc = True
        elif self._control.mesh_format == MeshFormat.adcirc:
            need_bc = False
        else:
            # This is normally unreachable due to the section-to-control converter.
            raise AssertionError('Unsupported mesh format.')  # pragma: nocover

        bc_file = self._control.bc_file
        if need_bc and not bc_file:
            self._ok = False
            self._log.error('External .bc and .neighbor files are in use, but .bc file path was not set.')
        if need_bc and not Path(bc_file).exists():
            message = (
                'External .bc and .neighbor files are in use, but .bc file path refers to nonexistent file. '
                'File will be generated when simulation is run.'
            )
            self._log.warning(message)

    def _write_neighbors_and_bc(self):
        """
        Write the .neighbors and .bc files.

        There isn't actually anything to write, this really just fixes the output names as necessary.
        """
        if not self._mesh_is_linked:
            return

        self._control.bc_file = LINKED_BC_FILE_NAME
        self._control.neighbor_file = LINKED_NEIGHBORS_FILE_NAME

        # We just replaced the mesh, so delete the .neighbors file to ensure PTM regenerates it to match the mesh.
        Path(LINKED_NEIGHBORS_FILE_NAME).unlink(missing_ok=True)
        # We always run PTM in ADCIRC mode, which doesn't use the .bc file, so no need to delete it.

    def _write_control(self):
        """Write the control file."""
        self._log.info('Writing control file...')

        simulation_name = self._data.simulation_name
        write_control(self._control, f'{simulation_name}.pcf')

    def _check_sources(self):
        """Check that the source coverage can be written and log warnings if not."""
        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        if not coverage:
            self._ok = False
            self._log.error('Exporting requires a PTM Sources coverage linked to the simulation.')
            return

        have_points = data.component_id_map[TargetType.point]
        have_arcs = data.component_id_map[TargetType.arc]
        have_polygons = data.component_id_map[TargetType.polygon]

        if not have_points and not have_arcs and not have_polygons:
            self._ok = False
            self._log.error('The linked Sources coverage has no assigned features.')
            return

    def _write_sources(self):
        """Write the sources."""
        where = self._control.source_file

        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        time = datetime_to_string(self._stop_time)
        write_sources(coverage, data, time, where)

    def _write_traps(self):
        """Write the traps if necessary."""
        where = self._control.trap_file
        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        write_traps(coverage, data, where)

    def _check_hydrodynamic_datasets(self) -> None:
        """Check the hydrodynamic datasets."""
        if self._hydro_is_linked:
            self._check_internal_hydro_datasets()
        else:
            self._check_external_hydro_datasets()

    def _check_external_hydro_datasets(self):
        """Check that external hydro datasets are valid."""
        hydro_file = self._control.flow_file_xmdf
        if not hydro_file:
            self._ok = False
            self._log.error('Flow file was not selected in the model control.')
        if not Path(hydro_file).exists():
            self._ok = False
            self._log.warning(f'External flow file does not exist yet: {hydro_file}')

        elevation_path = self._control.xmdf_wse_path
        if not elevation_path:
            self._ok = False
            self._log.error('Elevation dataset path was not selected in the model control.')
        flow_path = self._control.xmdf_vel_path
        if not flow_path:
            self._ok = False
            self._log.error('Flow dataset path was not selected in the model control.')

        if not self._ok:
            return

        elevation_dataset = flow_dataset = None
        with suppress(Exception):
            elevation_dataset = DatasetReader(hydro_file, group_path=elevation_path)
            flow_dataset = DatasetReader(hydro_file, group_path=flow_path)

        if not elevation_dataset or not flow_dataset:
            self._ok = False
            self._log.error('Hydro file set in model control did not contain datasets at specified paths.')
            return

        if not elevation_dataset.ref_time:
            self._ok = False
            self._log.error('Elevation dataset specified in the model control has no reference time.')
        elif self._control.start_run < elevation_dataset.ref_time:
            self._ok = False
            self._log.error('Elevation dataset specified in the model control starts after simulation.')
        elif _last_step_in(elevation_dataset) < self._stop_time:
            self._ok = False
            self._log.error('Elevation dataset specified in the model control ends before simulation.')

        if not flow_dataset.ref_time:
            self._ok = False
            self._log.error('Flow dataset specified in the model control has no reference time.')
        elif self._control.start_run < flow_dataset.ref_time:
            self._ok = False
            self._log.error('Flow dataset specified in the model control starts after simulation.')
        elif _last_step_in(flow_dataset) < self._stop_time:
            self._ok = False
            self._log.error('Flow dataset specified in the model control ends before simulation.')

    def _check_internal_hydro_datasets(self):
        """Check that internal hydro datasets are valid."""
        if not self._data.linked_grid:
            # Linked datasets require a linked UGrid. If we're here, the UGrid checks should log an error for us.
            self._ok = False
            return

        if not self._mesh_is_linked:
            self._ok = False
            self._log.error('Cannot use linked hydro datasets with an external mesh/grid.')

        elevation_uuid = self._linked_elevation_dataset
        if not elevation_uuid:
            self._ok = False
            self._log.error('Elevation dataset was not selected in the model control.')
        elif not self._data.datasets[elevation_uuid]:
            self._ok = False
            self._log.error('Elevation dataset selected in the model control no longer exists.')
        elif not self._data.datasets[elevation_uuid].ref_time:
            self._ok = False
            self._log.error('Elevation dataset selected in the model control has no reference time set.')
        elif self._control.start_run < self._data.datasets[elevation_uuid].ref_time:
            self._ok = False
            self._log.error('Elevation dataset selected in the model control starts after simulation.')
        elif _last_step_in(self._data.datasets[elevation_uuid]) < self._stop_time:
            self._ok = False
            self._log.error('Elevation dataset selected in the model control ends before simulation.')
        elif self._data.datasets[elevation_uuid].num_values != self._data.linked_grid.ugrid.point_count:
            self._ok = False
            self._log.error('"Elevation Dataset" set in model control does not match linked mesh.')

        flow_uuid = self._linked_flow_dataset
        if not flow_uuid:
            self._ok = False
            self._log.error('Flow dataset was not selected in the model control.')
        elif not self._data.datasets[flow_uuid]:
            self._ok = False
            self._log.error('Flow dataset selected in the model control no longer exists.')
        elif not self._data.datasets[flow_uuid].ref_time:
            self._ok = False
            self._log.error('Flow dataset selected in the model control has no reference time set.')
        elif self._control.start_run < self._data.datasets[flow_uuid].ref_time:
            self._ok = False
            self._log.error('Flow dataset selected in the model control starts after simulation.')
        elif _last_step_in(self._data.datasets[flow_uuid]) < self._stop_time:
            self._ok = False
            self._log.error('Flow dataset selected in the model control ends before simulation.')
        elif self._data.datasets[flow_uuid].num_values != self._data.linked_grid.ugrid.point_count:
            self._ok = False
            self._log.error('"Flow Dataset" set in model control does not match linked mesh.')

    def _write_hydrodynamics(self):
        """Write the hydrodynamic datasets."""
        if self._hydro_is_linked:
            self._write_linked_hydro()
        elif self._control.flow_format == FlowFormat.adcirc_ascii:
            pass  # Files already written, and no need to override the model control's flow start time.
        elif self._control.flow_format == FlowFormat.adcirc_xmdf:
            self._write_xmdf_hydro()
        else:
            # This should be unreachable normally because converting a section to a control will fail.
            raise AssertionError('Unsupported hydro type.')  # pragma: nocover

    def _write_linked_hydro(self):
        """Write linked hydro files."""
        self._control.flow_format = FlowFormat.adcirc_xmdf
        self._control.flow_file_xmdf = LINKED_HYDRO_FILE_NAME
        self._control.xmdf_wse_path = 'Datasets/elevation'
        self._control.xmdf_vel_path = 'Datasets/flow'

        elevation_name = 'elevation'
        elevation_uuid = self._linked_elevation_dataset
        elevation_dataset = self._data.datasets[elevation_uuid]

        flow_name = 'flow'
        flow_uuid = self._linked_flow_dataset
        flow_dataset = self._data.datasets[flow_uuid]

        datasets = [(elevation_dataset, elevation_name), (flow_dataset, flow_name)]
        write_datasets(datasets, LINKED_HYDRO_FILE_NAME)

        self._control.start_flow = elevation_dataset.ref_time

    def _write_xmdf_hydro(self):
        """Write external XMDF hydro files."""
        hydro_file = self._control.flow_file_xmdf
        dataset_path = self._control.xmdf_wse_path
        reader = DatasetReader(h5_filename=hydro_file, group_path=dataset_path)
        self._control.start_flow = reader.ref_time

    def _check_sediments(self) -> None:
        """Check the sediment datasets."""
        sediment_definition = self._control.sediment_format

        if self._sediment_is_linked:
            self._check_internal_sediment_datasets()
        elif sediment_definition in [SedimentFormat.adcirc, SedimentFormat.xmdf_dataset]:
            self._check_external_sediment_datasets()
        else:
            # This should be unreachable because converting section to control will fail before we get here.
            raise AssertionError('Unsupported sediment type.')  # pragma: nocover

    def _check_external_sediment_datasets(self):
        """Check that external sediment datasets are valid."""
        sediment_definition = self._control.sediment_format
        sediment_file = self._control.sediment_file

        if not sediment_file:
            self._ok = False
            self._log.error('Sediment file was not selected in the model control.')
        if not Path(sediment_file).exists():
            self._ok = False
            self._log.warning(f'External sediment file does not exist yet: {sediment_file}')

        d35_path = self._control.xmdf_d35_path
        if sediment_definition == SedimentFormat.xmdf_dataset and not d35_path:
            self._ok = False
            self._log.error('"D35 Path" value not set in the model control.')
        d50_path = self._control.xmdf_d50_path
        if sediment_definition == SedimentFormat.xmdf_dataset and not d50_path:
            self._ok = False
            self._log.error('"D50 Path" value not set in the model control.')
        d90_path = self._control.xmdf_d90_path
        if sediment_definition == SedimentFormat.xmdf_dataset and not d90_path:
            self._ok = False
            self._log.error('"D90 Path" value not set in the model control.')

    def _check_internal_sediment_datasets(self):
        """Check that internal sediment datasets are valid."""
        if not self._data.linked_grid:
            # Linked datasets require a linked UGrid. If we're here, the UGrid checks should log an error for us.
            self._ok = False
            return

        if not self._mesh_is_linked:
            self._ok = False
            self._log.error('Cannot use linked sediment datasets with an external mesh/grid.')

        d35_uuid = self._linked_d35_dataset
        if not d35_uuid:
            self._ok = False
            self._log.error('"D35 Dataset" value not set in the model control.')
        elif not self._data.datasets[d35_uuid]:
            self._ok = False
            self._log.error('"D35 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d35_uuid].num_values != self._data.linked_grid.ugrid.point_count:
            self._ok = False
            self._log.error('"D35 Dataset" set in model control does not match linked mesh.')

        d50_uuid = self._linked_d50_dataset
        if not d50_uuid:
            self._ok = False
            self._log.error('"D50 Dataset" value not set in the model control.')
        elif not self._data.datasets[d50_uuid]:
            self._ok = False
            self._log.error('"D50 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d50_uuid].num_values != self._data.linked_grid.ugrid.point_count:
            self._ok = False
            self._log.error('"D50 Dataset" set in model control does not match linked mesh.')

        d90_uuid = self._linked_d90_dataset
        if not d90_uuid:
            self._ok = False
            self._log.error('"D90 Dataset" value not set in the model control.')
        elif not self._data.datasets[d90_uuid]:
            self._ok = False
            self._log.error('"D90 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d90_uuid].num_values != self._data.linked_grid.ugrid.point_count:
            self._ok = False
            self._log.error('"D90 Dataset" set in model control does not match linked mesh.')

    def _write_sediments(self):
        """Write the sediment datasets."""
        if not self._sediment_is_linked:
            return

        self._control.sediment_format = SedimentFormat.xmdf_dataset
        self._control.sediment_file = LINKED_SEDIMENTS_FILE_NAME

        d35_name = 'd35'
        d35_dataset = self._data.datasets[self._linked_d35_dataset]
        self._control.xmdf_d35_path = 'Datasets/d35'

        d50_name = 'd50'
        d50_dataset = self._data.datasets[self._linked_d50_dataset]
        self._control.xmdf_d50_path = 'Datasets/d50'

        d90_name = 'd90'
        d90_dataset = self._data.datasets[self._linked_d90_dataset]
        self._control.xmdf_d90_path = 'Datasets/d90'

        datasets = [(d35_dataset, d35_name), (d50_dataset, d50_name), (d90_dataset, d90_name)]
        write_datasets(datasets, LINKED_SEDIMENTS_FILE_NAME)


def _last_step_in(dataset: DatasetReader) -> datetime:
    """Get the last time step in a dataset."""
    ref_time = dataset.ref_time
    delta = dataset.timestep_offset(dataset.num_times - 1)
    last_step = ref_time + delta
    return last_step
