"""Module for reading and writing model control."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = [
    'to_control', 'to_section', 'LINKED_MESH_FILE_NAME', 'LINKED_BC_FILE_NAME', 'LINKED_NEIGHBORS_FILE_NAME',
    'LINKED_HYDRO_FILE_NAME', 'LINKED_SEDIMENTS_FILE_NAME', 'SOURCE_FILE', 'TRAP_FILE'
]

# 1. Standard Python modules
import os
from pathlib import Path
from typing import Optional
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section
from xms.guipy.dialogs.feedback_thread import ExpectedError
from xms.guipy.time_format import datetime_to_string, string_to_datetime
from xms.ptmio.pcf.program_control import (
    AdvectionMethod, CentroidMethod, EulerianMethod, EulerianTransportMethod, FlowFormat, MeshFormat, NumericalScheme,
    ProgramControl, SedimentFormat, VelocityMethod
)

# 4. Local modules
from xms.ptm.model.model import (
    AdvectionDefinition, HydroDefinition, MeshDefinition, SedimentDefinition, simulation_model, StopMode
)

LINKED_MESH_FILE_NAME = 'mesh.grd'
LINKED_BC_FILE_NAME = 'mesh.bc'
LINKED_NEIGHBORS_FILE_NAME = 'mesh.neighbors'
LINKED_HYDRO_FILE_NAME = 'hydro.h5'
LINKED_SEDIMENTS_FILE_NAME = 'sediments.h5'
SOURCE_FILE = 'sources.source'
TRAP_FILE = 'traps.trap'

read_mesh_format_map = {
    MeshFormat.adcirc: MeshDefinition.adcirc,
    MeshFormat.cms_2d: MeshDefinition.cms,
}
write_mesh_format_map = {
    MeshDefinition.adcirc: MeshFormat.adcirc,
    MeshDefinition.cms: MeshFormat.cms_2d,
    MeshDefinition.linked: MeshFormat.adcirc,
}

read_flow_format_map = {
    FlowFormat.adcirc_ascii: HydroDefinition.adcirc_ascii,
    FlowFormat.adcirc_xmdf: HydroDefinition.adcirc_xmdf,
}
write_flow_format_map = {
    HydroDefinition.adcirc_ascii: FlowFormat.adcirc_ascii,
    HydroDefinition.adcirc_xmdf: FlowFormat.adcirc_xmdf,
    HydroDefinition.linked: FlowFormat.adcirc_xmdf,
}

read_sediment_type_map = {
    SedimentFormat.adcirc: SedimentDefinition.ascii,
    SedimentFormat.xmdf_dataset: SedimentDefinition.xmdf
}
write_sediment_type_map = {
    SedimentDefinition.xmdf: SedimentFormat.xmdf_dataset,
    SedimentDefinition.ascii: SedimentFormat.adcirc,
    SedimentDefinition.linked: SedimentFormat.xmdf_dataset
}

read_advection_method_map = {
    AdvectionMethod.two_d: AdvectionDefinition.two_d,
    AdvectionMethod.three_d: AdvectionDefinition.three_d
}
write_advection_method_map = {
    AdvectionDefinition.one_d: AdvectionMethod.one_d,
    AdvectionDefinition.two_d: AdvectionMethod.two_d,
    AdvectionDefinition.three_d: AdvectionMethod.three_d
}

read_centroid_method_map = {CentroidMethod.rouse: 'Rouse', CentroidMethod.van_rijn: 'Van Rijn'}
write_centroid_method_map = {'Rouse': CentroidMethod.rouse, 'Van Rijn': CentroidMethod.van_rijn}

read_eulerian_method_map = {EulerianMethod.ptm: 'PTM', EulerianMethod.van_rijn: 'Van Rijn'}
write_eulerian_method_map = {'PTM': EulerianMethod.ptm, 'Van Rijn': EulerianMethod.van_rijn}

read_eulerian_transport_map = {
    EulerianTransportMethod.soulsby_van_rijn: 'Soulsby-Van Rijn',
    EulerianTransportMethod.lund: 'Lund',
    EulerianTransportMethod.camenen_larson: 'Camenen Larson'
}
write_eulerian_transport_map = {
    'Soulsby-Van Rijn': EulerianTransportMethod.soulsby_van_rijn,
    'Lund': EulerianTransportMethod.lund,
    'Camenen Larson': EulerianTransportMethod.camenen_larson
}

read_numerical_scheme_map = {NumericalScheme.two: '2', NumericalScheme.four: '4'}
write_numerical_scheme_map = {'2': NumericalScheme.two, '4': NumericalScheme.four}

read_velocity_method_map = {VelocityMethod.two_d_log: '2D Logarithmic', VelocityMethod.three_d_sigma: '3DS'}
write_velocity_method_map = {'2D Logarithmic': VelocityMethod.two_d_log, '3DS': VelocityMethod.three_d_sigma}


def to_section(control: ProgramControl) -> Section:
    """
    Convert a ProgramControl to a Section.

    Args:
        control: The ProgramControl to convert.

    Returns:
        A Section representing the ProgramControl.
    """
    try:
        return _to_section_helper(control)
    except KeyError as e:
        raise ExpectedError(f'Unrecognized card value: {e}')


def _to_section_helper(control: ProgramControl) -> Section:
    """Helper for converting to a Section that does most of the work except handling errors."""
    section = simulation_model()

    group = section.group('time')
    group.parameter('duration').value = control.duration
    group.parameter('flow_update').value = control.flow_update
    group.parameter('grid_update').value = control.grid_update
    group.parameter('last_step_trap').value = control.last_step_trap
    group.parameter('mapping_inc').value = control.mapping_inc
    group.parameter('output_inc').value = control.output_inc
    group.parameter('start_run').value = datetime_to_string(control.start_run)
    group.parameter('start_trap').value = datetime_to_string(control.start_trap)
    group.parameter('start_waves').value = datetime_to_string(control.start_waves)
    if control.stop_run is not None:
        group.parameter('stop_run').value = datetime_to_string(control.stop_run)
        group.parameter('stop_mode').value = StopMode.date_time
    else:
        group.parameter('stop_mode').value = StopMode.duration
    group.parameter('stop_trap').value = datetime_to_string(control.stop_trap)
    group.parameter('time_step').value = control.time_step
    group.parameter('wave_step').value = control.wave_step

    group = section.group('mesh')
    if control.mesh_format == MeshFormat.adcirc:
        group.parameter('mesh_file').value = _to_section_path(control.mesh_file)
    elif control.mesh_format == MeshFormat.cms_2d:
        group.parameter('grid_file').value = _to_section_path(control.mesh_file)
    else:
        raise AssertionError('Unknown mesh format')  # pragma: nocover
    group.parameter('mesh_type').value = read_mesh_format_map[control.mesh_format]
    group.parameter('bc_file').value = _to_section_path(control.bc_file)
    group.parameter('neighbor_file').value = _to_section_path(control.neighbor_file)
    group.parameter('xmdf_hydro_file').value = _to_section_path(control.flow_file_xmdf)
    group.parameter('hydro_type').value = read_flow_format_map[control.flow_format]
    group.parameter('xmdf_elevation_path').value = control.xmdf_wse_path
    group.parameter('xmdf_flow_path').value = control.xmdf_vel_path
    if control.sediment_format == SedimentFormat.xmdf_dataset:
        group.parameter('xmdf_sediment_file').value = _to_section_path(control.sediment_file)
    elif control.sediment_format == SedimentFormat.adcirc:
        group.parameter('ascii_sediment_file').value = _to_section_path(control.sediment_file)
    else:
        raise AssertionError('Unrecognized sediment format.')
    if control.start_flow is not None:
        group.parameter('start_flow').value = datetime_to_string(control.start_flow)
    group.parameter('sediment_type').value = read_sediment_type_map[control.sediment_format]
    group.parameter('d35_dataset_path').value = control.xmdf_d35_path
    group.parameter('d50_dataset_path').value = control.xmdf_d50_path
    group.parameter('d90_dataset_path').value = control.xmdf_d90_path

    group = section.group('computations')
    group.parameter('advection_method').value = read_advection_method_map[control.advection_method]
    group.parameter('bed_interaction').value = control.bed_interaction
    group.parameter('bed_porosity').value = control.bed_porosity
    group.parameter('bedforms').value = control.bedforms
    group.parameter('centroid_method').value = read_centroid_method_map[control.centroid_method]
    group.parameter('currents').value = control.currents
    if control.by_weight:
        group.parameter('distribution').value = 'By weight'
    else:
        group.parameter('distribution').value = 'By grain size'
    group.parameter('etmin').value = control.etmin
    group.parameter('eulerian_method').value = read_eulerian_method_map[control.eulerian_method]
    group.parameter('eulerian_transport_method').value = read_eulerian_transport_map[control.eulerian_transport_method]
    group.parameter('evmin').value = control.evmin
    group.parameter('hiding_exposure').value = control.hiding_exposure
    group.parameter('ket').value = control.ket
    group.parameter('kev').value = control.kev
    group.parameter('kew').value = control.kew
    group.parameter('min_depth').value = control.min_depth
    group.parameter('morphology').value = control.morphology
    group.parameter('neutrally_buoyant').value = control.neutrally_buoyant
    group.parameter('numerical_scheme').value = read_numerical_scheme_map[control.numerical_scheme]
    group.parameter('residence_calc').value = control.residence_calc
    group.parameter('rhos').value = control.rhos
    group.parameter('salinity').value = control.salinity
    group.parameter('source_to_datum').value = control.source_to_datum
    group.parameter('temperature').value = control.temperature
    group.parameter('turbulent_shear').value = control.turbulent_shear
    group.parameter('velocity_method').value = read_velocity_method_map[control.velocity_method]
    group.parameter('wave_mass_transport').value = control.wave_mass_transport

    group = section.group('output')
    group.parameter('output_prefix').value = control.output_prefix
    group.parameter('bed_level_change_mapping').value = control.bed_level_change_mapping
    group.parameter('bed_level_mapping').value = control.bed_level_mapping
    group.parameter('bedform_mapping').value = control.bedform_mapping
    group.parameter('density_output').value = control.density_output
    group.parameter('fall_velocity_output').value = control.fall_velocity_output
    group.parameter('flow_mapping').value = control.flow_mapping
    group.parameter('flow_output').value = control.flow_output
    group.parameter('grain_size_output').value = control.grain_size_output
    group.parameter('height_output').value = control.height_output
    group.parameter('mobility_mapping').value = control.mobility_mapping
    group.parameter('mobility_output').value = control.mobility_output
    group.parameter('parcel_mass_output').value = control.parcel_mass_output
    group.parameter('population_record').value = control.population_record
    group.parameter('shear_stress_mapping').value = control.shear_stress_mapping
    group.parameter('source_output').value = control.source_output
    group.parameter('state_output').value = control.state_output
    group.parameter('tau_cr_output').value = control.tau_cr_output
    group.parameter('tecplot_maps').value = control.tecplot_maps
    group.parameter('tecplot_parcels').value = control.tecplot_parcels
    group.parameter('paths').value = control.paths
    group.parameter('transport_mapping').value = control.transport_mapping
    group.parameter('wave_mapping').value = control.wave_mapping
    group.parameter('xmdf_compressed').value = control.xmdf_compressed

    return section


def to_control(section: Section) -> ProgramControl:
    """
    Convert a Section to a ProgramControl.

    Args:
        section: The model control section to convert.

    Returns:
        The converted ProgramControl.
    """
    try:
        return _to_control_helper(section)
    except KeyError as e:
        raise AssertionError(f'Unrecognized value in model control: {e}')


def _to_control_helper(section: Section) -> ProgramControl:
    """Helper for writing that does most of the work but doesn't handle exceptions."""
    control = ProgramControl()

    group = section.group('time')
    control.flow_update = group.parameter('flow_update').value
    control.grid_update = group.parameter('grid_update').value
    control.last_step_trap = group.parameter('last_step_trap').value
    control.mapping_inc = group.parameter('mapping_inc').value
    control.output_inc = group.parameter('output_inc').value
    control.start_run = string_to_datetime(group.parameter('start_run').value)
    control.start_trap = string_to_datetime(group.parameter('start_trap').value)
    control.start_waves = string_to_datetime(group.parameter('start_waves').value)
    control.stop_trap = string_to_datetime(group.parameter('stop_trap').value)
    control.time_step = group.parameter('time_step').value
    control.wave_step = group.parameter('wave_step').value
    if group.parameter('stop_mode').value == StopMode.duration:
        control.duration = group.parameter('duration').value
    else:
        control.stop_run = string_to_datetime(group.parameter('stop_run').value)

    group = section.group('mesh')
    control.mesh_file = group.parameter('mesh_file').value
    control.bc_file = group.parameter('bc_file').value
    control.neighbor_file = group.parameter('neighbor_file').value
    control.mesh_format = write_mesh_format_map[group.parameter('mesh_type').value]
    if group.parameter('hydro_type').value == HydroDefinition.adcirc_ascii:
        control.start_flow = string_to_datetime(group.parameter('start_flow').value)
    else:
        control.start_flow = None

    control.bc_file = group.parameter('bc_file').value
    control.neighbor_file = group.parameter('neighbor_file').value

    group = section.group('mesh')
    control.flow_file_xmdf = group.parameter('xmdf_hydro_file').value
    control.flow_format = write_flow_format_map[group.parameter('hydro_type').value]
    control.xmdf_vel_path = group.parameter('xmdf_flow_path').value
    control.xmdf_wse_path = group.parameter('xmdf_elevation_path').value
    sediment_format = group.parameter('sediment_type').value
    if sediment_format == SedimentDefinition.xmdf or sediment_format == SedimentDefinition.linked:
        control.sediment_file = group.parameter('xmdf_sediment_file').value
    elif sediment_format == SedimentDefinition.ascii:
        control.sediment_file = group.parameter('ascii_sediment_file').value
    else:
        raise AssertionError('Unsupported sediment format.')
    control.sediment_format = write_sediment_type_map[group.parameter('sediment_type').value]
    control.xmdf_d35_path = group.parameter('d35_dataset_path').value
    control.xmdf_d50_path = group.parameter('d50_dataset_path').value
    control.xmdf_d90_path = group.parameter('d90_dataset_path').value

    group = section.group('computations')
    control.advection_method = write_advection_method_map[group.parameter('advection_method').value]
    control.bed_interaction = group.parameter('bed_interaction').value
    control.bed_porosity = group.parameter('bed_porosity').value
    control.bedforms = group.parameter('bedforms').value
    control.centroid_method = write_centroid_method_map[group.parameter('centroid_method').value]
    control.currents = group.parameter('currents').value
    control.etmin = group.parameter('etmin').value
    control.eulerian_method = write_eulerian_method_map[group.parameter('eulerian_method').value]
    control.eulerian_transport_method = write_eulerian_transport_map[group.parameter('eulerian_transport_method').value]
    control.evmin = group.parameter('evmin').value
    control.hiding_exposure = group.parameter('hiding_exposure').value
    control.ket = group.parameter('ket').value
    control.kev = group.parameter('kev').value
    control.kew = group.parameter('kew').value
    control.min_depth = group.parameter('min_depth').value
    control.morphology = group.parameter('morphology').value
    control.neutrally_buoyant = group.parameter('neutrally_buoyant').value
    control.numerical_scheme = write_numerical_scheme_map[group.parameter('numerical_scheme').value]
    control.residence_calc = group.parameter('residence_calc').value
    control.rhos = group.parameter('rhos').value
    control.salinity = group.parameter('salinity').value
    control.source_to_datum = group.parameter('source_to_datum').value
    control.temperature = group.parameter('temperature').value
    control.turbulent_shear = group.parameter('turbulent_shear').value
    control.velocity_method = write_velocity_method_map[group.parameter('velocity_method').value]
    control.wave_mass_transport = group.parameter('wave_mass_transport').value
    control.by_weight = group.parameter('distribution').value == 'By weight'

    group = section.group('output')
    control.output_prefix = group.parameter('output_prefix').value
    control.source_file = SOURCE_FILE
    control.trap_file = TRAP_FILE
    control.bed_level_change_mapping = group.parameter('bed_level_change_mapping').value
    control.bed_level_mapping = group.parameter('bed_level_mapping').value
    control.bedform_mapping = group.parameter('bedform_mapping').value
    control.density_output = group.parameter('density_output').value
    control.fall_velocity_output = group.parameter('fall_velocity_output').value
    control.flow_mapping = group.parameter('flow_mapping').value
    control.flow_output = group.parameter('flow_output').value
    control.grain_size_output = group.parameter('grain_size_output').value
    control.height_output = group.parameter('height_output').value
    control.mobility_mapping = group.parameter('mobility_mapping').value
    control.mobility_output = group.parameter('mobility_output').value
    control.parcel_mass_output = group.parameter('parcel_mass_output').value
    control.population_record = group.parameter('population_record').value
    control.shear_stress_mapping = group.parameter('shear_stress_mapping').value
    control.source_output = group.parameter('source_output').value
    control.state_output = group.parameter('state_output').value
    control.tau_cr_output = group.parameter('tau_cr_output').value
    control.tecplot_maps = group.parameter('tecplot_maps').value
    control.tecplot_parcels = group.parameter('tecplot_parcels').value
    control.paths = group.parameter('paths').value
    control.transport_mapping = group.parameter('transport_mapping').value
    control.wave_mapping = group.parameter('wave_mapping').value
    control.xmdf_compressed = group.parameter('xmdf_compressed').value

    control.particle_set_uuid = str(uuid.uuid4())

    return control


def _to_section_path(value: Optional[str | Path]):
    if not value:
        return ''

    return os.path.abspath(value)
