"""WCD (Water Control District) data."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd
from xms.rsm.data import monitor_data_def as mdd


class _WcdDataDef:
    """A class to define the lake data for the generic model."""
    def __init__(self):
        self.generic_model = GenericModel()
        self._pp = self.generic_model.polygon_parameters
        self._polygon_data_def()
        self._arc_data_def()

    def _arc_data_def(self):
        """Defines the arc data for the generic model."""
        self._pp = self.generic_model.arc_parameters
        self._add_canal_group()
        self._arc_monitor()

    def _add_canal_group(self):
        canal_gp = self.generic_model.arc_parameters.add_group(group_name='wcd_canal', label='WCD canal')
        canal_gp.add_float(name='segwidth', label='Segment width (<segwidth>)', default=0.0)
        canal_gp.add_float(name='botelv', label='Bottom elevation (<botelv>)', default=0.0)
        canal_gp.add_float(name='leakagecoeff', label='Leakage coefficient (<leakagecoeff>)', default=0.0)
        canal_gp.add_float(name='bankheight', label='Height of canal lip (<bankheight>)', default=0.0)
        canal_gp.add_float(name='bankcoeff', label='Coefficient for flow over canal lip (<bankcoeff>)', default=0.0)

    def _arc_monitor(self):
        pp = self._pp
        outputs = [
            ('seep_flow', 'Monitor wcdmover seep flow (<flow>)'),
            ('bank_flow', 'Monitor wcdmover bank flow (<flow>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)

    def _polygon_data_def(self):
        """Defines the polygon data for the generic model."""
        self._add_wcd_group()
        self._add_bc_group()
        self._poly_monitor()

    def _add_wcd_group(self):
        """Adds the lake group to the polygon parameters."""
        gp = self._pp.add_group(group_name='wcd', label='Water Control District')
        gp.add_text(name='label', label='WCD label (<label>)')
        # inputs for Stage Volume Table
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Volume', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        gp.add_table(name='sv', label='Stage-volume table (<sv>)', table_definition=td, default=default_vals)
        # initial head
        parent = gp.add_boolean(name='b_init_head', label='WCD initial head (<initialcondfile>)', default=False)
        p = gp.add_float(name='init_head', label='', default=0.0)
        flags = {True: True, False: False}
        p.add_dependency(parent=parent, flags=flags)

    def _add_bc_group(self):
        """Adds the HPM group to the polygon parameters."""
        bc_gp = self._pp.add_group(group_name='bc', label='WCD Boundary Conditions')
        bc_gp.add_text(name='label', label='Label (<label>)')
        bc_gp.add_integer(name='bc_id', label='Boundary condition id (<id>)', default=-1)
        opts = ['wcdsource', 'wcdhead']
        bc_gp.add_option(name='bc_type', label='BC type', default=opts[0], options=opts)
        # add the boundary condition values
        bvdd.add_bc_val_to_group(group=bc_gp)

    def _poly_monitor(self):
        pp = self._pp
        outputs = [
            ('head', 'Monitor head (<head>)'),
            ('head_iter', 'Monitor head at each iteration (<head_iter>)'),
            ('seepflow', 'Monitor seepage flow (<seepflow>)'),
            ('bankflow', 'Monitor bank flow (<bankflow>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)


def generic_model():
    """Returns a generic model with the definition for polygon properties.

    Returns:
        (GenericModel): see above
    """
    wcd_dd = _WcdDataDef()
    return wcd_dd.generic_model
