"""Parameter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class Parameter:
    """Represents a parameter used in ParametersManager and the Advanced Simulation dialog."""
    def __init__(
        self,
        use=0,
        id_number=-1,
        type='',
        description='',
        default=0.0,
        string_value='',
        value=0.0,
        optimized_value=0.0,
        min=0.0,
        max=0.0
    ):
        """Initializes the class.

        Args:
            use (int): 1 if using this parameter, 0 if not
            id_number (int): Number used in the id_string.
            type (str): E.g. "Manning's N", 'Inlet Q', 'Exit H', 'Time step', 'Initial condition'
            description (str): Typically the arc number, or material name.
            default (float): default value
            string_value (str): string value
            value (float): value
            optimized_value (float): optimized value
            min (float): min
            max (float): max
        """
        self.use = use  # This is an int because bools don't serialize as well to json
        self.id_number = id_number
        self.type = type
        self.description = description
        self.default = default
        self.string_value = string_value
        self.value = value
        self.optimized_value = optimized_value
        self.min = min
        self.max = max
        # If you add/remove something above, you'll probably need to update the .to_dict() method below

        self._column_dtype = {}  # Dict of runs table column names and their dtypes

    def update(self, **kwargs):
        """Updates some parameter values.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            use (int): 1 if using this parameter, 0 if not
            value (float): value
            min (float): min
            max (float): max
        """
        self.use = kwargs.get('use', self.use)
        self.value = kwargs.get('value', self.value)
        self.min = kwargs.get('min', self.min)
        self.max = kwargs.get('max', self.max)

    def id_string(self):
        """Returns a string for use in the 'id' column.

        Returns:
            (str): See description.
        """
        return ''

    def column_base_name(self):
        """Returns a string that uniquely identifies this parameter.

        Combination of the parameter type and possibly the description if it's needed too.

        Returns:
            (str): See description.
        """
        if not self.description:
            return self.type
        else:
            return f'{self.type}\n{self.description}'

    def to_dict(self, params_dict):
        """Adds this parameter's values to the dict.

        Args:
            params_dict (dict): The dict.
        """
        params_dict.setdefault('id', []).append(self.id_string())
        params_dict.setdefault('Use', []).append(self.use)
        params_dict.setdefault('Type', []).append(self.type)
        params_dict.setdefault('Description', []).append(self.description)
        params_dict.setdefault('Default', []).append(self.default)
        params_dict.setdefault('String Value', []).append(self.string_value)
        params_dict.setdefault('Value', []).append(self.value)
        params_dict.setdefault('Optimized Value', []).append(self.optimized_value)
        params_dict.setdefault('Min', []).append(self.min)
        params_dict.setdefault('Max', []).append(self.max)

    def add_run_data(self, disk_data, run_count, param_data):
        """Adds columns to the runs table.

        If disk_data has information about this parameter, uses it.

        Args:
            param_data (ParameterData): Param data.
            run_count (int): Number of runs.
            disk_data (dict): Param data from file.
        """
        base_name = self.column_base_name()
        self._add_column(disk_data, run_count, f'{base_name} Transient', 'int', param_data)
        self._add_column(disk_data, run_count, base_name, 'float', param_data)
        self._add_column(disk_data, run_count, f'{base_name} XY Series', object, param_data)

    def _add_column(self, disk_data, run_count, column_name, dtype, param_data):
        """Adds columns to the runs table.

        If disk_data has information about this parameter, uses it.

        Args:
            param_data (ParameterData): Param data.
            run_count (int): Number of runs.
            column_name (str): Name of the column.
            dtype: Data type of the column. Either a string like 'int' or object.
            disk_data (dict): Param data from file.
        """
        if disk_data and column_name in disk_data['runs']:
            param_data.runs[column_name] = disk_data['runs'][column_name]
        else:  # Add default values
            param_data.runs[column_name] = [self.column_default(column_name)] * run_count
        self._column_dtype[column_name] = dtype

    def column_dtype(self, column_name):
        """Given a column name in the runs table, returns the dtype.

        Args:
            column_name (str): The name of the column in the runs table.

        Returns:
            The dtype of the column.
        """
        assert column_name in self._column_dtype
        return self._column_dtype[column_name]

    def column_delegate(self, column_name, xy_series_names, dialog):
        """Give the name of a column in the runs table, returns the delegate, or None.

        Args:
            column_name (str): The name of the column in the runs table.
            xy_series_names (list): List of the names of the xy series.
            dialog: Parent of the delegate.

        Returns:
            The delegate or None
        """
        return None

    def column_default(self, column_name):
        """Given a column name, return the default value.

        Args:
            column_name (str): The name of the column in the runs table.

        Returns:
            Default value.
        """
        return self.default

    def column_count(self):
        """Returns the number of columns this parameter has in the runs table.

        Returns:
            (int): See description.
        """
        return len(self._column_dtype)  # Assumes the columns have been added already

    def has_column_name(self, column_name):
        """Return true if this Parameter has the column name.

        Args:
            column_name (str): The column name.

        Returns:
            (bool): True
        """
        return column_name in self._column_dtype

    def calibratable(self):
        """Returns True if the parameter can be used for calibration.

        Returns:
            (bool): See description.
        """
        return False

    def flags(self, index, column_name, flags):
        """Returns the flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index.
            column_name (str): Name of the column.
            flags (Qt.ItemFlags): The flags in their initial state

        Returns:
            (Qt.ItemFlags): Updated flags for the item
        """
        del index  # Unused parameter
        del column_name  # Unused parameter
        return flags

    def copy_disk_data(self, disk_data):
        """Copies certain things from the data on disk to the data in RAM.

        Args:
            disk_data (dict): Dict we are getting data from.
        """
        id_str = self.id_string()
        if id_str not in disk_data['params']['id']:
            id_str += '.0'
        if id_str not in disk_data['params']['id']:
            raise ValueError('Unable to find parameter {id_str}')
        disk_row = disk_data['params']['id'].index(id_str)
        self.use = disk_data['params']['Use'][disk_row]
        if 'Description' in disk_data['params']:
            self.description = disk_data['params']['Description'][disk_row]
        if 'Default' in disk_data['params']:
            self.default = disk_data['params']['Default'][disk_row]
        if 'String Value' in disk_data['params']:
            self.string_value = disk_data['params']['String Value'][disk_row]
        if 'Value' in disk_data['params']:
            self.value = disk_data['params']['Value'][disk_row]
        if 'Optimized Value' in disk_data['params']:
            self.optimized_value = disk_data['params']['Optimized Value'][disk_row]
        if 'Min' in disk_data['params']:
            self.min = disk_data['params']['Min'][disk_row]
        if 'Max' in disk_data['params']:
            self.max = disk_data['params']['Max'][disk_row]

    def get_values(self, runs, run):
        """Given the runs table and the run, returns the values for the run.

        Args:
            runs (dict): The runs dict.
            run (int): Run index.

        Returns
            (list): The values.
        """
        values = []
        for key in self._column_dtype.keys():
            values.append(runs[key][run])
        return values
